/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DotPlotTasks.h"

#include <gobjects/DNASequenceObject.h>
#include <document_format/DocumentFormatUtils.h>

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/DocumentFormatConfigurators.h>

#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/AddDocumentTask.h>


namespace GB2 {

void SaveDotPlotTask::run() {

    QFile file(filename);
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return;
    }

    QTextStream stream(&file);
    saveDotPlot(stream);

    file.close();
}

void LoadDotPlotTask::run() {

    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        return;
    }

    QTextStream stream(&file);
    if (!loadDotPlot(stream, file.size())) {

        stateInfo.setError(tr("Wrong dotplot format"));
    }

    file.close();
}

// check if the file opens
DotPlotDialogs::Errors SaveDotPlotTask::checkFile(const QString &filename) {

    QFile file(filename);
    if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
        return DotPlotDialogs::ErrorOpen;
    }

    file.close();
    return DotPlotDialogs::NoErrors;
}

void SaveDotPlotTask::saveDotPlot(QTextStream &stream){

	Q_ASSERT (sequenceX);
	Q_ASSERT (sequenceY);

    stream << sequenceX->getGObjectName() << endl;
    stream << sequenceY->getGObjectName() << endl;

    stream << minLen << " " << identity << endl;

	Q_ASSERT(list);

    int listSize = list->size();
    int i=0;
    foreach(const DotPlotResults &r, *list) {
        if (stateInfo.cancelFlag) {
            return;
        }

        stream << r.x << " " << r.y << " " << r.len << endl;

		Q_ASSERT (listSize);
        stateInfo.progress = (100*i)/listSize;

        i++;
    }
}

// check if the file opens and sequence names are the same
DotPlotDialogs::Errors LoadDotPlotTask::checkFile(const QString &filename, const QString &seqXName, const QString &seqYName) {

    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {

        return DotPlotDialogs::ErrorOpen;
    }

    QTextStream stream(&file);

    QString readedXName;
    QString readedYName;

    readedXName = stream.readLine();
    readedYName = stream.readLine();

    DotPlotDialogs::Errors err = DotPlotDialogs::NoErrors;

    if (seqXName != readedXName || seqYName != readedYName) {
        err = DotPlotDialogs::ErrorNames;
    }

    file.close();

    return err;
}

bool LoadDotPlotTask::loadDotPlot(QTextStream &stream, int fileSize) {

    QString readedXName;
    QString readedYName;

    readedXName = stream.readLine();
    readedYName = stream.readLine();

    // not checking names, just loading dotplot

    int newMinLen, newIdentity;
    QList<DotPlotResults> newDotPlotList;

    stream >> newMinLen >> newIdentity;

    if (newMinLen<2 || newIdentity<50) {
		// wrong format

        return false;
    }

    while (!stream.atEnd() && !stateInfo.cancelFlag) {

        DotPlotResults r;
        stream >> r.x >> r.y >> r.len;
        newDotPlotList.push_back(r);

		Q_ASSERT (stream.device());
		Q_ASSERT (fileSize);

        int streamPos = stream.device()->pos();
        stateInfo.progress = (100*streamPos)/fileSize;
    }

	Q_ASSERT(list);
	Q_ASSERT(minLen);
	Q_ASSERT(identity);

    list->clear();
    *list = newDotPlotList;
    *minLen = newMinLen;
    *identity = newIdentity;

    return true;
}


DotPlotLoadDocumentsTask::DotPlotLoadDocumentsTask(QString firstF, int firstG, QString secondF, int secondG)
: Task(tr("DotPlot loading"), TaskFlags(TaskFlag_NoRun | TaskFlag_FailOnSubtaskCancel))
{
	firstFile = firstF;
	firstGap = firstG;

	secondFile = secondF;
	secondGap= secondG;
}

void DotPlotLoadDocumentsTask::prepare() {

	// load sequences
	Document *doc = loadFile(firstFile, firstGap);
	if (doc) {
		docs << doc;
	}

	if (hasErrors()) {
		return;
	}
	doc = loadFile(secondFile, secondGap);
	if (doc) {
		docs << doc;
	}
}

Document *DotPlotLoadDocumentsTask::loadFile(QString inFile, int gapSize) {

    GUrl URL(inFile);

    Project *project = AppContext::getProject();

	Q_ASSERT(project);
    Document *doc = project->findDocumentByURL(URL);

	// document already present in the project
    if (doc) {
        return doc;
    }

    QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(inFile);
    if (formats.isEmpty()) {
        stateInfo.setError(tr("Detecting format error for file %1").arg(inFile));
        return NULL;
    }

    DocumentFormat* format = formats.first();
    Q_ASSERT(format);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(URL));

	QVariantMap formatSettings;
	if (gapSize >= 0) {
		QString mergeToken = MERGE_MULTI_DOC_GAP_SIZE_SETTINGS;
		formatSettings[mergeToken] = gapSize;
	}

    doc = new Document(format, iof, URL, QList<UnloadedObjectInfo>(), formatSettings);

    addSubTask(new AddDocumentTask(doc)); // add document to the project
	addSubTask(new LoadUnloadedDocumentTask(doc)); // load document

    return doc;
}

DotPlotLoadDocumentsTask::~DotPlotLoadDocumentsTask() {

	// error while loading documents
    if (hasErrors()) {
        Project *project = AppContext::getProject();

		// skip added to the project documents
		if (project) {
	        QList<Document*> projectDocs = project->getDocuments();

		    foreach (Document *doc, projectDocs) {
			    docs.removeAll(doc);
			}
		}

		// delete loaded but not added to the project documents
//		qDeleteAll(docs);
		foreach (Document *doc, docs) {
//			docs.removeAll(doc);
			delete doc;
		}
   }
}

} // namespace
