/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemoteQueryTask.h"

#include <core_api/DocumentFormats.h>

namespace GB2 {

class BaseIOAdapters;
class BaseDocumentFormats;

static LogCategory log(ULOG_CAT_PLUGIN_REMOTE_QUERY);


RemoteQueryToAnnotationsTask::RemoteQueryToAnnotationsTask( QString _dbChoosen, 
                                DNATranslation * _aminoT, DNATranslation * _complT, const QByteArray & _query, int _qoffs, 
                                AnnotationTableObject* _ao, const QString &_url,const QString & _group, QString _params, int _retries ):
Task( tr("remote_query_task"), TaskFlags_NR_FOSCOE ), offsInGlobalSeq(_qoffs), aobj(_ao), group(_group), url(_url) {
    GCOUNTER( cvar, tvar, "RemoteQueryToAnnotationsTask" );
    RemoteQueryTaskSettings cfg;
    cfg.dbChoosen = _dbChoosen;
    cfg.aminoT = _aminoT;
    cfg.complT = _complT;
    cfg.query = _query;
    cfg.params = _params;
    cfg.retries = _retries;
    
    queryTask = new RemoteQueryTask(cfg);
    addSubTask(queryTask);
}


QList<Task*> RemoteQueryToAnnotationsTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    
    if(subTask->hasErrors() && subTask == queryTask) {
        stateInfo.setError(subTask->getError());
        return res;
    }

    if (hasErrors() || isCanceled()) {
        return res;
    }

    if (aobj.isNull()) {
        stateInfo.setError(  tr("obj_was_removed\n") );
        return res;
    }
    if (subTask == queryTask) {
        //shift annotations according to offset first
        
        RemoteQueryTask * rrTask = qobject_cast<RemoteQueryTask *>(queryTask);
        assert( rrTask );
        
        QList<SharedAnnotationData> anns = rrTask->getResultedAnnotations();

        if(!anns.isEmpty()) {		
           // Document* d = AppContext::getProject()->findDocumentByURL(url);
            //assert(d==NULL);
            if(!url.isEmpty()) {
                IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
                DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_GENBANK);
                Document *d = df->createNewDocument(iof, url);
                d->addObject(aobj);
                AppContext::getProject()->addDocument(d);
            }
            
            for(QMutableListIterator<SharedAnnotationData> it_ad(anns); it_ad.hasNext(); ) {
                AnnotationData * ad = it_ad.next().data();
                QList<LRegion> & regions = ad->location;
                for( QMutableListIterator<LRegion> it(regions); it.hasNext(); ) {
                    it.next().startPos += offsInGlobalSeq;
                }
            }
            
            res.append(new CreateAnnotationsTask(aobj, group, anns));
        }
    }
    return res;
}


RemoteQueryTask::RemoteQueryTask( const RemoteQueryTaskSettings & cfg_) :
Task( tr("remote_query_task"), TaskFlag_None ), timeout(true), cfg(cfg_)
{
}

void RemoteQueryTask::prepareQueries() {
    Query q;
    if(cfg.aminoT) {
        q.amino = true;
        QByteArray complQuery( cfg.query.size(), 0 );
        cfg.complT->translate( cfg.query.data(), cfg.query.size(), complQuery.data(), complQuery.size() );
        TextUtils::reverse( complQuery.data(), complQuery.size() );
        for( int i = 0; i < 3; ++i ) {
            QByteArray aminoQuery( cfg.query.size() / 3, 0 );
            cfg.aminoT->translate( cfg.query.data()+i, cfg.query.size()-i, aminoQuery.data(), aminoQuery.size() );
            q.seq = aminoQuery;
            q.offs = i;
            q.complement = false;
            queries.push_back(q);
            QByteArray aminoQueryCompl( cfg.query.size() / 3, 0 );
            cfg.aminoT->translate(complQuery.data()+i, complQuery.size()-i, aminoQueryCompl.data(), aminoQueryCompl.size());
            q.seq = aminoQueryCompl;
            q.offs = i;
            q.complement = true;
            queries.push_back(q);
        }
    }
    else {
        q.seq = cfg.query;
        queries.push_back(q);
    }
}

void RemoteQueryTask::prepare() {
    prepareQueries();
    log.info("Sequences prepared");
    for (QList<Query>::iterator it = queries.begin(),end = queries.end();it!=end;it++) {
        DataBaseFactory *dbf = AppContext::getDataBaseRegistry()->getFactoryById(cfg.dbChoosen);
        if(dbf == NULL) {
            stateInfo.setError(tr("Incorrect database"));
            return;
        }
        HttpRequest *tmp = dbf->getRequest(this);
        httpRequest.append(tmp);
    }
    log.info("Requests formed");
    connect(&timer,SIGNAL(timeout()),SLOT(sl_timeout()));
    timeout = true;
    timer.setSingleShot(true);
    int mult = cfg.aminoT ? 6 : 1; //if 3 requests - 3 times more wait
    timer.start(cfg.retries*1000*60*mult);
}


void RemoteQueryTask::run() {	
    for( int i = 0;i < queries.count();i++ ) {
        bool error = true;
        if( isCanceled() ) {
            return;
        }
        
        httpRequest[i]->sendRequest(cfg.params,QString(queries[i].seq.data()));
        error = httpRequest[i]->connectionError;
        if(error) {
            stateInfo.setError(httpRequest[i]->getError());
            return;
        }
    
        createAnnotations(queries[i],httpRequest[i]);
    }
}

QList<SharedAnnotationData> RemoteQueryTask::getResultedAnnotations() const {
    return resultAnnotations;
}

void RemoteQueryTask::createAnnotations(const Query &q, HttpRequest *t) {
    QList<SharedAnnotationData> annotations = t->getAnnotations();
    if(annotations.isEmpty()) return;

    for( QList<SharedAnnotationData>::iterator it = annotations.begin(), end = annotations.end(); end != it; ++it ) {
        for( QList<LRegion>::iterator jt = it->data()->location.begin(), eend = it->data()->location.end(); eend != jt; ++jt ) {
                int & s = jt->startPos;
                int & l = jt->len;

                if( q.complement ) {
                    s = q.seq.size() - s - l;
                    it->data()->complement = !it->data()->complement;
                }
                if( q.amino ) {
                    s = s * 3 + (q.complement ? 2 - q.offs : q.offs);
                    l = l * 3;
                }
        }
    }
    resultAnnotations << annotations;
}


}
