/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "HMMIOWorker.h"

#include <workflow/Datatype.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/GUrlUtils.h>
#include <util_tasks/FailTask.h>
#include <util_tasks/TaskSignalMapper.h>

#include "HMMIO.h"
#include "u_search/HMMSearchDialogController.h"
#include "u_build/HMMBuildWorker.h"
#include "u_search/HMMSearchWorker.h"

/* TRANSLATOR GB2::LocalWorkflow::HMMLib */

namespace GB2 {
namespace LocalWorkflow {

const QString HMMReader::ACTOR("uhmmer.read");
const QString HMMWriter::ACTOR("uhmmer.write");

const QString HMMLib::HMM_PROFILE_TYPE_ID("hmm.profile");

DataTypePtr HMMLib::HMM_PROFILE_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(HMM_PROFILE_TYPE_ID, tr("HMM Profile"), "")));
        startup = false;
    }
    return dtr->getById(HMM_PROFILE_TYPE_ID);
}

const Descriptor HMMLib::HMM_CATEGORY() {return Descriptor("hmmer", tr("HMMER tools"), "");}

static LogCategory log(ULOG_CAT_WD);

HMMIOProto::HMMIOProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: BusActorPrototype(_desc, _ports, _attrs) {
}

bool HMMIOProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = GUrlUtils::getUncompressedExtension(url);
            if (ext == HMMIO::HMM_EXT) {
                if (params) {
                    params->insert(urlAttrId, url);
                }
                return true;
            }
        }
    }
    return false;
}

ReadHMMProto::ReadHMMProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: HMMIOProto( _desc, _ports, _attrs ) {
    
    attrs << new Attribute(CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true);
    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR_ID] = new URLDelegate( HMMIO::getHMMFileFilter(), HMMIO::HMM_ID, true );
    setEditor(new DelegateEditor(delegateMap));
    setIconPath( ":/hmm2/images/hmmer_16.png" );
}

bool ReadHMMProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return HMMIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR_ID );
}

WriteHMMProto::WriteHMMProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: HMMIOProto( _desc, _ports, _attrs ) {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);
    
    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR_ID] = new URLDelegate(HMMIO::getHMMFileFilter(), HMMIO::HMM_ID, false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(false);
    
    setEditor(new DelegateEditor(delegateMap));
    setIconPath( ":/hmm2/images/hmmer_16.png" );
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR_ID, ports.first()->getId(), CoreLibConstants::URL_SLOT_ID));
    setPortValidator(CoreLibConstants::IN_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT_ID));
}

bool WriteHMMProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return HMMIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR_ID );
}

void HMMIOWorkerFactory::init() {
 
     WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
     assert(r);
     DataTypePtr dt = HMMLib::HMM_PROFILE_TYPE();
     {
         Descriptor id(CoreLibConstants::IN_PORT_ID, HMMLib::tr("HMM profile"), HMMLib::tr("Input HMM profile"));
         Descriptor ud(CoreLibConstants::URL_SLOT_ID, HMMLib::tr("Location"), HMMLib::tr("Location hint for the target file."));

         QMap<Descriptor, DataTypePtr> m;
         m[ud] = CoreDataTypes::STRING_TYPE();
         m[*dt] = dt;
         DataTypePtr t(new MapDataType(Descriptor("write.hmm.content"), m));

         QList<PortDescriptor*> p; QList<Attribute*> a;
         p << new PortDescriptor(id, t, true /*input*/);
         
         Descriptor desc(HMMWriter::ACTOR, HMMLib::tr("Write HMM profile"), HMMLib::tr("Saves all input HMM profiles to specified location."));
         BusActorPrototype* proto = new WriteHMMProto(desc, p, a);
         proto->setPrompter(new HMMWritePrompter());
         r->registerProto(HMMLib::HMM_CATEGORY(), proto);
     }
     {
         Descriptor od(CoreLibConstants::OUT_PORT_ID, HMMLib::tr("HMM profile"), HMMLib::tr("Loaded HMM profile"));

         QList<PortDescriptor*> p; QList<Attribute*> a;
         p << new PortDescriptor(od, dt, false /*output*/, true);
         Descriptor desc(HMMReader::ACTOR, HMMLib::tr("Read HMM profile"), HMMLib::tr("Reads HMM profiles from file(s). The files can be local or Internet URLs."));
         BusActorPrototype* proto = new ReadHMMProto(desc, p, a);
         proto->setPrompter(new HMMReadPrompter());
         r->registerProto(HMMLib::HMM_CATEGORY(), proto);
     }
 
     DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
     localDomain->registerEntry(new HMMIOWorkerFactory(HMMReader::ACTOR));
     localDomain->registerEntry(new HMMIOWorkerFactory(HMMWriter::ACTOR));
}

QString HMMReadPrompter::composeRichDoc() {
    return tr("Read HMM profile(s) from %1").arg(getURL(CoreLibConstants::URL_IN_ATTR_ID));
}

QString HMMWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(CoreLibConstants::IN_PORT_ID);
    if (!producer) {
        return getURL(CoreLibConstants::URL_OUT_ATTR_ID);
    }
    QString url = getScreenedURL(input, CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID); 
    QString doc = tr("Save HMM profile(s) from <u>%1</u> to <u>%2</u>.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void HMMIOWorkerFactory::cleanup() {
     DomainFactory* ld = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
     DomainFactory* f = ld->unregisterEntry(HMMReader::ACTOR); 
     delete f;
     f = ld->unregisterEntry(HMMWriter::ACTOR);
     delete f;
 
     WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
     ActorPrototype* p = r->unregisterProto(HMMReader::ACTOR);
     assert(p); delete p;
     p = r->unregisterProto(HMMWriter::ACTOR);
     assert(p); delete p;
}

Worker* HMMIOWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
     if (HMMReader::ACTOR == a->getProto()->getId()) {
         w = new HMMReader(a);
     } 
     else if (HMMWriter::ACTOR == a->getProto()->getId()) {
         w = new HMMWriter(a);
     }

    return w;    
}

void HMMReader::init() {
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    urls = DesignerUtils::expandToUrls(actor->getParameter(CoreLibConstants::URL_IN_ATTR_ID)->getAttributeValue<QString>());
}

Task* HMMReader::tick() {
    Task* t = new HMMReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void HMMReader::sl_taskFinished() {
    HMMReadTask* t = qobject_cast<HMMReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<plan7_s*>(t->getHMM());
            output->put(Message(HMMLib::HMM_PROFILE_TYPE(), v));
        }
        if (urls.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded HMM profile from %1").arg(t->getURL()));
    }
}

void HMMWriter::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
}

Task* HMMWriter::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    url = actor->getParameter(CoreLibConstants::URL_OUT_ATTR_ID)->getAttributeValue<QString>();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->getAttributeValue<uint>();
    QVariantMap data = inputMessage.getData().toMap();
    
    plan7_s* hmm = data.value(HMMLib::HMM_PROFILE_TYPE_ID).value<plan7_s*>();
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    }
    if (anUrl.isEmpty() || hmm == NULL) {
        QString err = (hmm == NULL) ? tr("Empty HMM passed for writing to %1").arg(anUrl) : tr("Unspecified URL for writing HMM");
        if (failFast) {
            return new FailTask(err);
        } else {
            log.error(err);
            return NULL;
        }
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = GUrlUtils::prepareFileName(anUrl, count, QStringList(HMMIO::HMM_EXT));
    } else {
        anUrl = GUrlUtils::ensureFileExt( anUrl, QStringList(HMMIO::HMM_EXT)).getURLString();
    }
    log.info(tr("Writing HMM profile to %1").arg(anUrl));
    return new HMMWriteTask(anUrl, hmm, fileMode);
}


void HMMLib::init() {
    HMMIOWorkerFactory::init();
    HMMBuildWorkerFactory::init();
    HMMSearchWorkerFactory::init();
}

void HMMLib::cleanup() {
    //FIXME need locking
    //HMMIOWorkerFactory::cleanup();
    //HMMBuildWorkerFactory::cleanup();
    //HMMSearchWorkerFactory::cleanup();

    //DataTypeRegistry* dr = WorkflowEnv::getDataTypeRegistry();
    //dr->unregisterEntry(HMM_PROFILE_TYPE->getId());
}

} //namespace LocalWorkflow
} //namespace GB2
