/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_ORFFINDALGORITHM_H_
#define _U2_ORFFINDALGORITHM_H_

#include <U2Core/LRegion.h>
#include <U2Core/AnnotationData.h>

#include <QtCore/QList>

namespace U2 {

class DNATranslation;

class U2ALGORITHM_EXPORT ORFFindResult {
public:
    ORFFindResult () : region(0, 0), frame(0){}
    ORFFindResult (const LRegion& _r, int frame) 
        : region(_r), frame(frame){}
    
    void clear() {region.startPos = 0; region.len = 0; frame = 0;}
    
    bool isEmpty() const {return region.startPos == 0 && region.len == 0;}

    bool operator ==(const ORFFindResult& o) const {
        return region == o.region && frame == o.frame;
    }

    SharedAnnotationData toAnnotation(const QString& name) const {
        SharedAnnotationData data;
        data = new AnnotationData;
        data->name = name;
        data->location.append(region);
        data->complement = (frame < 0);
        data->aminoFrame = TriState_Yes;
        //data->qualifiers.append(Qualifier("frame", QString::number(frame)));
        data->qualifiers.append(Qualifier("dna_len", QString::number(region.len)));
        if (region.len >= 6) { // 3 bp - end codon
            data->qualifiers.append(Qualifier("protein_len", QString::number(region.len/3)));
        }
        return data;
    }

    LRegion region;
    int frame;

    static QList<SharedAnnotationData> toTable(const QList<ORFFindResult>& res, const QString& name)
    {
        QList<SharedAnnotationData> list;
        foreach (const ORFFindResult& f, res) {
            list.append(f.toAnnotation(name));
        }
        return list;
    }
};

class U2ALGORITHM_EXPORT ORFFindResultsListener {
public:
    virtual ~ORFFindResultsListener(){}
    virtual void onResult(const ORFFindResult& r) = 0;
};

enum ORFAlgorithmStrand {
    ORFAlgorithmStrand_Both,
    ORFAlgorithmStrand_Direct,
    ORFAlgorithmStrand_Complement
};


class U2ALGORITHM_EXPORT ORFAlgorithmSettings {
public:
    ORFAlgorithmSettings(ORFAlgorithmStrand strand = ORFAlgorithmStrand_Both,
        DNATranslation* complementTT = NULL,
        DNATranslation* proteinTT = NULL,
        const LRegion& searchRegion = LRegion(),
        int minLen = 0,
        bool mustFit = false,
        bool mustInit = true,
        bool allowAltStart = false
        ) : strand(strand), complementTT(complementTT), proteinTT(proteinTT),
        searchRegion(searchRegion), minLen(minLen), mustFit(mustFit), 
        mustInit(mustInit), allowAltStart(allowAltStart) {}

    ORFAlgorithmStrand strand;
    DNATranslation*     complementTT;
    DNATranslation*     proteinTT;
    LRegion             searchRegion;
    int                 minLen;
    bool                mustFit;
    bool                mustInit;
    bool                allowAltStart;
};


class U2ALGORITHM_EXPORT ORFFindAlgorithm {
public:

    static void find(
        ORFFindResultsListener* rl,
        const ORFAlgorithmSettings& config,
        const char* sequence, 
        int seqLen, 
        int& stopFlag, 
        int& percentsCompleted);
};

}//namespace

#endif
