/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DnaAssemblyMultiTask.h"

#include <U2Core/LoadDocumentTask.h>
#include <U2Core/AddDocumentTask.h>
#include <U2Gui/OpenViewTask.h>
#include <U2Algorithm/DnaAssemblyAlgRegistry.h>
#include <U2Core/AppContext.h>
#include <U2Core/DocumentUtils.h>
#include <U2Core/MAlignmentObject.h>

namespace U2 {

DnaAssemblyMultiTask::DnaAssemblyMultiTask( const DnaAssemblyToRefTaskSettings& s, bool view, bool _justBuildIndex )
: Task("DnaAssemblyMultiTask", TaskFlags_NR_FOSCOE), settings(s),
assemblyToRefTask(NULL), addDocumentTask(NULL), loadDocumentTask(NULL),
doc(NULL), shortReadUrls(s.shortReadUrls), openView(view), justBuildIndex(_justBuildIndex)
{

}

void DnaAssemblyMultiTask::prepare() {
    // perform assembly
    QString algName = settings.algName;
    DnaAssemblyAlgorithmEnv* env= AppContext::getDnaAssemblyAlgRegistry()->getAlgorithm(algName);
    assert(env);
    if (env == NULL) {
        setError(QString("Algorithm %1 is not found").arg(algName));
        return;
    }
    assemblyToRefTask = env->getTaskFactory()->createTaskInstance(settings, justBuildIndex);
    addSubTask(assemblyToRefTask);
}

Task::ReportResult DnaAssemblyMultiTask::report() {
    return ReportResult_Finished;
}

QList<Task*> DnaAssemblyMultiTask::onSubTaskFinished( Task* subTask ) {
    QList<Task*> subTasks;
    if (subTask->hasErrors() || isCanceled()) {
        return subTasks;
    }

    if ( subTask == assemblyToRefTask && settings.loadResultDocument ) {
        assert(!settings.resultFileName.isEmpty());
        GUrl resultUrl(settings.resultFileName); 
        QList<DocumentFormat*> detectedFormats = DocumentUtils::detectFormat(resultUrl);    
        if (!detectedFormats.isEmpty()) {
            IOAdapterFactory* factory = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
            DocumentFormat* format = detectedFormats.first();
            loadDocumentTask = new LoadDocumentTask(format->getFormatId(), resultUrl, factory);
            subTasks.append(loadDocumentTask);
        }  

    } else if (subTask == loadDocumentTask ) {
        doc = loadDocumentTask->getDocument();
        if (openView) {
            DocumentFormat* format = doc->getDocumentFormat(); 
            IOAdapterFactory * iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
            Document* clonedDoc = new Document(format, iof, doc->getURLString());
            clonedDoc->loadFrom(doc); // doc was loaded in a separate thread -> clone all GObjects
            assert(!clonedDoc->isTreeItemModified());
            assert(clonedDoc->isLoaded());

            // Leak alert: if this signal isn't handled somewhere, memory allocated for this document will be lost.
            emit documentAvailable(clonedDoc);
        }
    }

    return subTasks;
}

const MAlignmentObject* DnaAssemblyMultiTask::getAssemblyResult() {
    if ( doc == NULL ) {
        return NULL;
    }

    QList<GObject*> objs = doc->getObjects();
    if ( objs.size() == 0 ) {
        return NULL;
    }

    return qobject_cast<MAlignmentObject*> (objs.first());
}

} // namespace