/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_ANNOTATION_DATA_H_
#define _U2_ANNOTATION_DATA_H_

#include <U2Core/LRegion.h>

#include <QtCore/QSharedData>
#include <QtCore/QVector>
#include <QtCore/QStringList>

namespace U2 {


class U2CORE_EXPORT Qualifier {
    friend U2CORE_EXPORT QDataStream& operator>>(QDataStream& dataStream, Qualifier& q);
    friend U2CORE_EXPORT QDataStream& operator<<(QDataStream& dataStream, const Qualifier& q);
public:
    Qualifier(){}
    Qualifier(const QString& _name, const QString& _val) : name(_name), val(_val) {}

    bool operator== ( const Qualifier & q ) const { return q.name == name && q.val == val; }
    bool operator!= ( const Qualifier & q ) const { return q.name != name ||  q.val != val; }

    const QString& getQualifierName() const {return name;}

    const QString& getQualifierValue() const {return val;}
    
    bool isEmpty() const { return name.isEmpty(); }
    
private:
    QString name;
    QString val;
};

class U2CORE_EXPORT AnnotationData : public QSharedData {
public:
    bool                    complement;
    bool                    order; // for multi-region annotations means 'order' if true or 'join' if false
    TriState                aminoFrame;
    QString                 name;
    QList<LRegion>          location;
    QVector<Qualifier>      qualifiers;
    
    AnnotationData() : complement(false), order(false), aminoFrame(TriState_Unknown) {}

    inline AnnotationData& operator= ( const AnnotationData & a );

    inline void removeAllQualifiers(const QString& name, QStringList& values);

    inline void findQualifiers(const QString& name, QVector<Qualifier>& res) const;

    inline QString findFirstQualifierValue(const QString& name) const;

    bool isJoin() const {return !order;}
    
    bool isOrder() const {return order;}
};

typedef QSharedDataPointer<AnnotationData> SharedAnnotationData;

U2CORE_EXPORT QDataStream& operator>>(QDataStream& dataStream, TriState& state);
U2CORE_EXPORT QDataStream& operator<<(QDataStream& dataStream, const TriState& state);
U2CORE_EXPORT QDataStream& operator>>(QDataStream& dataStream, Qualifier& q);
U2CORE_EXPORT QDataStream& operator<<(QDataStream& dataStream, const Qualifier& q);
U2CORE_EXPORT QDataStream& operator>>(QDataStream& dataStream, AnnotationData& data);
U2CORE_EXPORT QDataStream& operator<<(QDataStream& dataStream, const AnnotationData& data);

AnnotationData& AnnotationData::operator= ( const AnnotationData & a ) { 
    name = a.name; 
    complement = a.complement;
    aminoFrame = a.aminoFrame;
    location = a.location; 
    qualifiers = a.qualifiers;
    return *this;
}

void AnnotationData::removeAllQualifiers(const QString& name, QStringList& values) {
    for (int i=qualifiers.size(); --i>=0;) {
        const Qualifier& q = qualifiers.at(i);
        if (q.getQualifierName() == name) {
            values.append(q.getQualifierValue());
            qualifiers.remove(i);
        }
    }
}

void AnnotationData::findQualifiers(const QString& name, QVector<Qualifier>& res) const {
    foreach (const Qualifier& q, qualifiers) {
        if (q.getQualifierName() == name) {
            res.append(q);
        }
    }
}

QString AnnotationData::findFirstQualifierValue(const QString& name) const {
    foreach (const Qualifier& q, qualifiers) {
        if (q.getQualifierName() == name) {
            return q.getQualifierValue();
        }
    }
    return QString::null;
}


}//namespace

Q_DECLARE_METATYPE(U2::AnnotationData);
Q_DECLARE_METATYPE(U2::SharedAnnotationData);
Q_DECLARE_METATYPE(QList<U2::SharedAnnotationData>);

#endif
