/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "Notification.h"


namespace U2 {

Notification::Notification(const QString &message, NotificationType _type, QAction *_action):QLabel(NULL),text(message), action(_action), type(_type) {
    setMinimumWidth(TT_WIDTH);
    setMaximumWidth(TT_WIDTH);
    setMaximumHeight(TT_HEIGHT);

    setFrameStyle(QFrame::StyledPanel);
    close = new QLabel(this);
    QBoxLayout *h = new QHBoxLayout(this);
    setLayout(h);

    QFontMetrics metrics(font());
    setText(metrics.elidedText(message, Qt::ElideRight, width()-50));
    setToolTip(message);
    
    generateCSS(false);
    generateCSSforCloseButton(false);

    //setWindowFlags (Qt::FramelessWindowHint);
    setWindowFlags (Qt::ToolTip);
    close->installEventFilter(this);
    h->addStretch();
    h->addWidget(close);
    close->hide();
    setMouseTracking(true);

    close->setAttribute(Qt::WA_Hover);
    close->setFixedSize(16,16);
}

void Notification::generateCSS(bool isHovered) {
    QString bgColor;
    QString img;
    QString fontColor;
    QString border;
    switch(type) {
        case Info_Not: bgColor = "background-color: #BDE5F8;";
            fontColor = "color: #00529B;";
            img = "background-image: url(':core/images/info_notification.png');" ;
            break;
        case Error_Not: bgColor = "background-color: #FFBABA;";
            fontColor = "color: #D8000C;";
            img = "background-image: url(':core/images/error_notification.png');" ;
            break;
        case Report_Not: bgColor = "background-color: #BDE5F8;";
            fontColor = "color: #00529B;";
            img = "background-image: url(':core/images/info_notification.png');" ;
            break;
        default: assert(0);
    }

    if(isHovered) {
        border = "border: 2px solid;";
    } else {
        border = "border: 1px solid;";
    }
    QString css;

    css.append(border);
    css.append("padding: 2px 2px 2px 20px;");
    css.append("background-repeat: no-repeat;");
    css.append("background-position: left center;");
    css.append(fontColor);
    css.append(bgColor);
    css.append(img);

    setStyleSheet(css);
}

void Notification::generateCSSforCloseButton(bool isHovered) {
    QString css;
    QString background;
    if(isHovered) {
        css = "border: 1px solid;";
        background = "background-color: #C0C0C0;";
    } else {
        css = "border: none;";
        background = "background-color: transparent;";
    }
    
    css.append("border-radius: 3px;");
    css.append("background-position: center center;");
    css.append("paddling: 2px 2px 2px 2px;");
    css.append(background);
    css.append("background-image: url(':core/images/close.png');");
    close->setStyleSheet(css);
}

bool Notification::event(QEvent *e) {
    if(e->type() == QEvent::ToolTip) {
        QHelpEvent *hEvent = static_cast<QHelpEvent *>(e);
        QToolTip::showText(hEvent->globalPos(), QString(text));
        return true;
    }
    if(e->type() == QEvent::HoverEnter) {
        generateCSS(true);
    }
    if(e->type() == QEvent::HoverLeave) {
        generateCSS(false);
    }
    return QWidget::event(e);
}

void Notification::mousePressEvent(QMouseEvent *ev) {
    if(ev->button() == Qt::LeftButton) {
        if(timer.isActive()) {
            dissapear();
        }
        if(action) {
            action->trigger();
        } else if(!timer.isActive()){
            QDialog dlg;
            QVBoxLayout vLayout;
            QHBoxLayout hLayout;
            QPushButton ok;
            QCheckBox isDelete;

            ok.setText("Ok");
            isDelete.setText(tr("Remove notification after closing"));
            isDelete.setChecked(true);
            connect(&ok, SIGNAL(clicked()), &dlg, SLOT(accept()));
            hLayout.addWidget(&isDelete);
            hLayout.addWidget(&ok);

            dlg.setLayout(&vLayout);
            QTextEdit txtEdit;
            txtEdit.setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
            dlg.setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
            txtEdit.setReadOnly(true);
            txtEdit.setText(text);
            vLayout.addWidget(&txtEdit);
            vLayout.addLayout(&hLayout);

            dlg.setWindowTitle(tr("Detailed message"));

            AppContext::getMainWindow()->getNotificationStack()->setFixed(true);
            if(dlg.exec() == QDialog::Accepted) {
                if(isDelete.isChecked()) {
                    emit si_delete();
                }
            }
            AppContext::getMainWindow()->getNotificationStack()->setFixed(false);
        }
    }
}



bool Notification::eventFilter(QObject *watched, QEvent *event) {
    if(event->type() == QEvent::MouseButtonPress) {
        QMouseEvent *e= static_cast<QMouseEvent*>(event);
        if(e->button() == Qt::LeftButton) {
            emit si_delete();
            return true;
        }
    } else if(event->type() == QEvent::HoverEnter) {
        generateCSSforCloseButton(true);
    } else if(event->type() == QEvent::HoverLeave) {
        generateCSSforCloseButton(false);
    }
    return false;
}

void Notification::showNotification(int x, int y) {
    counter = 0;
    timer.setInterval(20);
    connect(&timer, SIGNAL(timeout()), SLOT(sl_timeout()));
    timer.start();
    show();
    move(x,y);
    resize(TT_WIDTH, 0);
}

void Notification::dissapear() {
    setMinimumHeight(TT_HEIGHT);
    close->show();
    setAttribute(Qt::WA_Hover);
    hide();
    timer.stop();
    emit si_dissapear();
}


void Notification::sl_timeout() {
    QRect rect = geometry();
    if(rect.height() >= TT_HEIGHT) {
        ++counter;
        if(counter > 100) {
            dissapear();
        }
    } else {        
        move(rect.topLeft().x(), rect.topLeft().y() - 10);
        resize(TT_WIDTH, rect.height() + 10);
    }
}

QString Notification::getText() const {
    return text;
}

NotificationType Notification::getType() const {
    return type;
}

NotificationStack::NotificationStack(QObject *o /* = NULL */): QObject(o), notificationPosition(0), notificationNumber(0) {
    w = new NotificationWidget(AppContext::getMainWindow()->getQMainWindow());
}


NotificationStack::~NotificationStack() {
    foreach(Notification* t, notifications) {
        delete t;
    }
    delete w;
}

bool NotificationStack::hasError() const {
    foreach(Notification *n, notifications) {
        if(n->getType() == Error_Not) {
            return true;
        }
    }
    return false;
}

void NotificationStack::addNotification(Notification *t) {
    if(notifications.count() >= MAX_NOTIFICATION) {
        Notification *toRemove = notifications.takeAt(0);
        w->removeNotification(toRemove);
        toRemove->deleteLater();
    }

    notifications.append(t);
    emit si_changed();
    
    connect(t, SIGNAL(si_delete()), this, SLOT(sl_delete()), Qt::DirectConnection);
    QPoint pos = AppContext::getMainWindow()->getQMainWindow()->geometry().bottomRight();
    t->showNotification(pos.x() - TT_WIDTH, pos.y() - 50 - notificationPosition);
    notificationNumber++;
    notificationPosition += TT_HEIGHT;
    connect(t, SIGNAL(si_dissapear()), SLOT(sl_notificationDissapear()));
}

void NotificationStack::sl_notificationDissapear() {
    notificationNumber--;
    if(notificationNumber == 0) {
        notificationPosition = 0;
    }
    
    Notification *t = qobject_cast<Notification*>(sender());
    t->show();
    t->setParent(w);
    w->addNotification(t);
}

void NotificationStack::sl_delete() {
    Notification *t = qobject_cast<Notification*>(sender());
    int i = notifications.indexOf(t);
    assert(i != -1);
    w->removeNotification(t);
    notifications.takeAt(i);
    emit si_changed();
    //t->deleteLater();
}

int NotificationStack::count() const {
    return notifications.count();
}

Notification *NotificationStack::getNotification(int row) const {
    return notifications[row];
}

QList <Notification *> NotificationStack::getItems() const {
    return notifications;
}

void NotificationStack::showStack() {
    QPoint pos = AppContext::getMainWindow()->getQMainWindow()->geometry().bottomRight();
    w->move(pos.x() - w->width(), pos.y() - w->height());
    w->show();
    w->setWindowState(Qt::WindowActive);
}

void NotificationStack::setFixed(bool val) {
    w->setFixed(val);
}


}
