/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AddNewDocumentDialogImpl.h"
#include "DocumentFormatComboboxController.h"

#include <U2Gui/GUIUtils.h>
#include <U2Misc/DialogUtils.h>
#include <U2Core/L10n.h>
#include <U2Core/Settings.h>
#include <U2Core/ProjectModel.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/AppContext.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>


namespace U2 {

//BUG:419: add label to dialog with state description!

#define SETTINGS_LASTFORMAT		"add_new_document/last_format"
#define SETTINGS_LASTDIR		"add_new_document/last_dir"

AddNewDocumentDialogImpl::AddNewDocumentDialogImpl(QWidget* p, AddNewDocumentDialogModel& m, const DocumentFormatConstraints& c) 
: QDialog(p), model(m)
{
	setupUi(this);

	if (model.format.isEmpty()) {
		model.format = AppContext::getSettings()->getValue(SETTINGS_LASTFORMAT, QString("")).toString();
	}
	
	documentURLEdit->setText(model.url);
	formatController = new DocumentFormatComboboxController(this, documentTypeCombo, c, model.format);
	model.successful = false;

	connect(documentURLButton, SIGNAL(clicked()), SLOT(sl_documentURLButtonClicked()));
	connect(createButton, SIGNAL(clicked()), SLOT(sl_createButtonClicked()));
	connect(documentURLEdit, SIGNAL(editingFinished()), SLOT(sl_documentURLEdited()));
	connect(documentTypeCombo, SIGNAL(currentIndexChanged(int)), SLOT(sl_typeComboCurrentChanged(int)));
	connect(gzipCheckBox, SIGNAL(toggled(bool)), SLOT(sl_gzipChecked(bool)));

	updateState();
}

void AddNewDocumentDialogController::run(QWidget* p, AddNewDocumentDialogModel& m, const DocumentFormatConstraints& c) {
    Project* proj = AppContext::getProject();
    if (proj->isStateLocked()) {
        QMessageBox::critical(NULL, L10N::errorTitle(), AddNewDocumentDialogImpl::tr("Project is locked"));
        m.successful = false;
        return;
    }

	AddNewDocumentDialogImpl d(p, m, c);
	d.exec();
	m = d.model;
    assert(proj->findDocumentByURL(m.url) == NULL);
}

void AddNewDocumentDialogImpl::updateState() {
	bool ready = formatController->hasSelectedFormat();
	
	if (ready) {
		const QString& url = currentURL();
		ready = !url.isEmpty() && QFileInfo(url).absoluteDir().exists();
		if (ready) {
			Project* p = AppContext::getProject();
			ready = p->findDocumentByURL(url) == NULL;
		}
	}
	
	createButton->setDisabled(!ready);
}


QString AddNewDocumentDialogImpl::currentURL() {
	QString url = documentURLEdit->text();
	if (url.isEmpty()) {
		return url;
	}
	bool extOk = false;
	QString extraExt = ".gz";
	if (url.endsWith(extraExt)) {
		url.chop(extraExt.size());
	}

	DocumentFormatId fid = formatController->getActiveFormatId();
	DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
	if (df) {
		foreach(QString ext, df->getSupportedDocumentFileExtensions()) {
			if (url.endsWith(ext)) {
				extOk = true;
				break;
			}
		}
		if (!extOk) {
			url+="."+df->getSupportedDocumentFileExtensions().first();
		}
	}
	if (gzipCheckBox->isChecked()) {
		url += extraExt;
	}
	documentURLEdit->setText(url);
	
	QFileInfo fi(url);
	return fi.absoluteFilePath();
}

void AddNewDocumentDialogImpl::sl_documentURLButtonClicked() {
	QString url = currentURL();
    if (url.isEmpty()) {
		url = AppContext::getSettings()->getValue(SETTINGS_LASTDIR, QString("")).toString();
    }
    QString filter = DialogUtils::prepareDocumentsFileFilter(formatController->getActiveFormatId(), false);
	QString name = QFileDialog::getSaveFileName(this, tr("Save file"), url, filter);
	if (!name.isEmpty()) {
		documentURLEdit->setText(name);	
		AppContext::getSettings()->setValue(SETTINGS_LASTDIR, QFileInfo(name).absoluteDir().absolutePath());
		updateState();
	}
}

void AddNewDocumentDialogImpl::sl_createButtonClicked() {
	model.format = formatController->getActiveFormatId();
	assert(!model.format.isEmpty());
	model.url = currentURL();
	model.io = gzipCheckBox->isChecked() ? BaseIOAdapters::GZIPPED_LOCAL_FILE : BaseIOAdapters::LOCAL_FILE;
	model.successful = true;
	AppContext::getSettings()->setValue(SETTINGS_LASTFORMAT, model.format);
	accept();
}



void AddNewDocumentDialogImpl::sl_documentURLEdited() {
	updateState();
}

void AddNewDocumentDialogImpl::sl_typeComboCurrentChanged(int i) {
    Q_UNUSED(i);
	QString url = documentURLEdit->text();
	if (!url.isEmpty()) {
		/* chop obsolete extensions */
		if (gzipCheckBox->isChecked() && url.endsWith(".gz")) {
			url.chop(3);
		}
		int dot = url.lastIndexOf('.');
		if (dot > 0) {
			url.chop(url.size() - dot);
		}
		documentURLEdit->setText(url);
	}

	updateState();
}

void AddNewDocumentDialogImpl::sl_gzipChecked(bool on) {
    Q_UNUSED(on);
	updateState();
}

}//namespace
