/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "QDConstraint.h"

#include <U2Lang/CoreDataTypes.h>


namespace U2 {

const QDConstraintType QDConstraintTypes::DISTANCE("QD_Distance_Constraint");

QDDistanceConstraint::QDDistanceConstraint(const QList<QDSchemeUnit*>& _units, QDDistanceType type, int min, int max)
: QDConstraint(_units, QDConstraintTypes::DISTANCE), distType(type) {
    assert(_units.size()==2);
    cfg = new QDParameters;

    Descriptor mind(QDConstraintController::MIN_LEN_ATTR, QObject::tr("Min distance"), QObject::tr("Minimum distance"));
    Descriptor maxd(QDConstraintController::MAX_LEN_ATTR, QObject::tr("Max distance"), QObject::tr("Maximum distance"));

    Attribute* minAttr = new Attribute(mind, CoreDataTypes::NUM_TYPE(), true, QVariant(min));
    Attribute* maxAttr = new Attribute(maxd, CoreDataTypes::NUM_TYPE(), true, QVariant(max));

    cfg->addParameter(minAttr->getId(), minAttr);
    cfg->addParameter(maxAttr->getId(), maxAttr);
}

QDDistanceConstraint::~QDDistanceConstraint() {
    delete cfg;
}

int QDDistanceConstraint::getMin() const {
    return cfg->getParameter(QDConstraintController::MIN_LEN_ATTR)->getAttributeValue<int>();
}

int QDDistanceConstraint::getMax() const {
    return cfg->getParameter(QDConstraintController::MAX_LEN_ATTR)->getAttributeValue<int>();
}

void QDDistanceConstraint::setMin(int min) {
    cfg->setParameter(QDConstraintController::MIN_LEN_ATTR, qVariantFromValue(min));
}

void QDDistanceConstraint::setMax(int max) {
    cfg->setParameter(QDConstraintController::MAX_LEN_ATTR, qVariantFromValue(max));
}

QString QDDistanceConstraint::getText(QDSchemeUnit*, QDSchemeUnit*) const {
    int minVal = getMin();
    int maxVal = getMax();
    if(minVal==maxVal) {
        return QString("%1 bp").arg(QString::number(minVal));
    }
    return QString("%1..%2 bp").arg(QString::number(minVal)).arg(QString::number(maxVal));
}

void QDDistanceConstraint::invert() {
    units.swap(0,1);
    setMin(-getMax());
    setMax(-getMin());
    distType = QDConstraintController::getInvertedType(distType);
}

//QDConstraintController
//////////////////////////////////////////////////////////////////////////
const QString QDConstraintController::DISTANCE_CONSTRAINT_EL("distance");
const QString QDConstraintController::TYPE_ATTR("type");
const QString QDConstraintController::MIN_LEN_ATTR("min");
const QString QDConstraintController::MAX_LEN_ATTR("max");
const QString QDConstraintController::SRC_ATTR("src");
const QString QDConstraintController::DST_ATTR("dst");

QDDistanceType QDConstraintController::getInvertedType(QDDistanceType type) {
    if (type==E2E) {
        return S2S;
    }
    if (type==S2S) {
        return E2E;
    }
    return type;
}

bool QDConstraintController::match(QDConstraint* c, const QDResultUnit& r1, const QDResultUnit& r2) {
    QDDistanceConstraint* dc = static_cast<QDDistanceConstraint*>(c);
    assert(dc);

    QDActor* a1 = r1->owner->getActor();
    QDActor* a2 = r2->owner->getActor();
    if (!a1->hasAnyDirection() && !a2->hasAnyDirection()) {
        if (a1->getDirection()==a2->getDirection() && r1->complement!=r2->complement) {
            return false;
        }
        if (a1->getDirection()!=a2->getDirection() && r1->complement==r2->complement) {
            return false;
        }
    }

    const LRegion& reg1 = r1->region;
    const LRegion& reg2 = r2->region;
    QDDistanceType dist = dc->distanceType();
    int min = dc->getMin();
    int max = dc->getMax();

    if(dc->getSource()==r1->owner) {
        if ( ( !a1->hasAnyDirection() && r1->complement) || ( !a2->hasAnyDirection() && r2->complement) ) {
            return match(reg2, reg1, getInvertedType(dist), min, max);
        } else {
            return match(reg1, reg2, dist, min, max);
        }
    } else {
        assert(dc->getSource()==r2->owner);
        if ( ( !a1->hasAnyDirection() && r1->complement) || ( !a2->hasAnyDirection() && r2->complement) ) {
            return match(reg1, reg2, getInvertedType(dist), min, max);
        } else {
            return match(reg2, reg1, dist, min, max);
        }
    }
}

bool QDConstraintController::match(const LRegion& srcReg,
                                   const LRegion& dstReg,
                                   QDDistanceType type,
                                   int min,
                                   int max)
{
    switch (type)
    {
    case E2S:
        {
            int srcEnds = srcReg.endPos();
            int dstStarts = dstReg.startPos;
            int distance = dstStarts - srcEnds;
            if(distance<=max && distance>=min) {
                return true;
            }
        }
        return false;
    case S2E:
        {
            int srcStarts = srcReg.startPos;
            int dstEnds = dstReg.endPos();
            int distance = dstEnds - srcStarts;
            if(distance<=max && distance>=min) {
                return true;
            }
        }
        return false;
    case S2S:
        {                    
            int srcStarts = srcReg.startPos;
            int dstStarts = dstReg.startPos;
            int distance = dstStarts - srcStarts;
            if(distance<=max && distance>=min) {
                return true;
            }
        }
        return false;
    case E2E:
        {                    
            int srcEnds = srcReg.endPos();
            int dstEnds = dstReg.endPos();
            int distance = dstEnds - srcEnds;
            if(distance<=max && distance>=min) {
                return true;
            }
        }
        return false;
    }
    return false;
}

LRegion QDConstraintController::matchLocation(QDDistanceConstraint* dc, const QDResultUnit& r) {
    int start = 0;
    int end = 0;

    QDSchemeUnit* src = NULL;
    QDSchemeUnit* dst = NULL;
    QDDistanceType type = dc->distanceType();
    int minDist = dc->getMin();
    int maxDist = dc->getMax();

    if ( !r->owner->getActor()->hasAnyDirection() && r->complement ) {
        src = dc->getDestination();
        dst = dc->getSource();
        type = getInvertedType(type);
    } else {
        src = dc->getSource();
        dst = dc->getDestination();
    }

    if (src==r->owner) {
        int len = dst->getActor()->getMaxResultLen();
        switch (type)
        {
        case E2S:
            start = r->region.endPos() + minDist;
            end = r->region.endPos() + maxDist + len;
            break;
        case E2E:
            end = r->region.endPos() + maxDist;
            start = r->region.endPos() + minDist - len;
            break;
        case S2S:
            start = r->region.startPos + minDist;
            end = r->region.startPos + maxDist + len;
            break;
        case S2E:
            end = r->region.startPos + maxDist;
            start = r->region.startPos + minDist - len;
            break;
        }
    } else {
        assert(dst==r->owner);
        int len = src->getActor()->getMaxResultLen();
        switch (type)
        {
        case E2S:
            start = r->region.startPos - maxDist - len;
            end = r->region.startPos - minDist;
            break;
        case E2E:
            end = r->region.endPos() - minDist;
            start = r->region.endPos() - maxDist - len;
            break;
        case S2S:
            start = r->region.startPos - maxDist;
            end = r->region.startPos - minDist + len;
            break;
        case S2E:
            start = r->region.endPos() - maxDist;
            end = r->region.endPos() - minDist + len;
            break;
        }
    }

    return LRegion(start, end-start);
}

}//namespace
