/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QFile>
#include <QtCore/QDir>
#include <QtCore/QDirIterator>

#include <U2Core/AppContext.h>
#include <U2Core/AppSettings.h>
#include <U2Core/global.h>
#include <U2Core/Log.h>
#include <U2Core/UserApplicationsSettings.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/WorkflowManager.h>
#include <U2Lang/WorkflowRunTask.h>
#include <U2Lang/WorkflowUtils.h>

#include "RunSchemaForTask.h"

namespace U2 {

using namespace Workflow;

/* class CreateSchemaForTask: public Task */

const QString CreateSchemaForTask::SCHEMAS_DIR = QString(PATH_PREFIX_DATA) + "/remote_tasks";

CreateSchemaForTask::CreateSchemaForTask(const QString schemaName_) :
        Task(tr("CreateSchemaTask"), TaskFlags_NR_FOSCOE),
        schema(0),
        schemaName(schemaName_),
        loadTask(0)
{
}

void CreateSchemaForTask::prepare() {
    schema = new Schema();
    schema->setDeepCopyFlag(true);

    QString schemaFile = findSchemaFile(schemaName);
    if (schemaFile.isEmpty()) {
        QString msg = tr("Can't find schema file %1").arg(schemaFile);
        coreLog.error(msg);
        setError(msg);
        return;
    }

    loadTask = new LoadWorkflowTask(schema, (Metadata*)0, schemaFile);
    addSubTask(loadTask);
}

QList<Task*> CreateSchemaForTask::onSubTaskFinished(Task *subTask)
{
    assert(subTask);
    QList<Task*> subTasks;

    propagateSubtaskError();
    if(hasErrors() || isCanceled())
    {
        return subTasks;
    }

    if (loadTask == subTask)
    {
        Schema *schema = loadTask->getSchema();
        assert(schema);

        if (schema->getDomain().isEmpty())
        {
            schema->setDomain(WorkflowEnv::getDomainRegistry()->getAllIds().value(0));
        }
    }

    return subTasks;
}

QString CreateSchemaForTask::findSchemaFile(const QString &schemaName) {
    foreach(const QString & ext, WorkflowUtils::WD_FILE_EXTENSIONS) {
        QString path = SCHEMAS_DIR + "/" + schemaName + "." + ext;
        if(QFile::exists(path)) {
            return path;
        }
    }
    return QString();
}

/* class RunSchemaForTask: public U2::Task */
RunSchemaForTask::RunSchemaForTask(const QString &schemaName_, const QVariantMap &schemaParameters_, const QString &inFile_, const QString &outFile_) :
        Task(tr("RunSchemaForTask"), TaskFlags_NR_FOSCOE),
        schema(0),
        schemaName(schemaName_),
        inFile(inFile_), outFile(outFile_),
        schemaParameters(schemaParameters_),
        createTask(0), runTask(0)
{
    schemaParameters["in_file"] = inFile;
    schemaParameters["out_file"] = outFile;
}

RunSchemaForTask::~RunSchemaForTask()
{
}

void RunSchemaForTask::prepare()
{
    createTask = prepareCreateSchemaTask();
    addSubTask(createTask);
}

QList<Task*> RunSchemaForTask::onSubTaskFinished(Task *subTask)
{
    assert(subTask);
    QList<Task*> subTasks;

    propagateSubtaskError();
    if(hasErrors() || isCanceled())
    {
        return subTasks;
    }

    if (createTask == subTask)
    {
        schema = createTask->getSchema();
        setSchemaParameters(schema, schemaParameters);

        runTask = prepareWorkflowRunTask();
        subTasks << runTask;
    }
    else if (runTask == subTask)
    {
    }

    return subTasks;
}

CreateSchemaForTask* RunSchemaForTask::prepareCreateSchemaTask()
{
    return new CreateSchemaForTask(schemaName);
}

Task* RunSchemaForTask::prepareWorkflowRunTask()
{
    return new WorkflowRunTask(*schema, schema->getIterations());
}

void RunSchemaForTask::setSchemaParameters(Schema *schema, const QVariantMap &params)
{
    QVariantMap::const_iterator i = params.begin();
    while (i != params.end())
    {
        setSchemaParameter(schema, i.key(), i.value());
        ++i;
    }
}

static Actor* findActorByParamAlias(Schema *schema, const QString &alias, QString &attrName)
{
    assert(schema);
    QList<Actor*> actors;
    foreach (Actor *actor, schema->getProcesses())
    {
        assert(actor);
        if (actor->getParamAliases().values().contains( alias ))
        {
            actors << actor;
        }
    }

    if (actors.isEmpty()) {
        return 0;
    } else if(actors.size() > 1) {
        coreLog.error(RunSchemaForTask::tr( "%1 actors in schema have '%2' alias" ).arg(actors.size()).arg(alias));
    }

    Actor *ret = actors.first();
    attrName = ret->getParamAliases().key(alias);
    return ret;
}

void RunSchemaForTask::setSchemaParameter(Schema *schema, const QString &param, const QVariant &value)
{
    assert(schema);

    QString paramName;
    Actor *actor = findActorByParamAlias(schema, param, paramName);

    if(actor == 0) {
        assert(paramName.isEmpty());
        coreLog.info(RunSchemaForTask::tr("alias '%1' not set in schema").arg(param));
        return;
    }

    Attribute *attr = actor->getParameter(paramName);
    if(attr == 0) {
        coreLog.error(tr("actor parameter '%1' not found").arg(paramName));
        return;
    }

    ActorId id = actor->getId();

    QList<Iteration> &iterations = schema->getIterations();

    QList<Iteration>::iterator it = iterations.begin();
    while(it != iterations.end()) // TODO: make different values for different iterations
    {
        it->cfg[id].insert(paramName, value);
        ++it;
    }
}

/* class SchemaForTaskUtils */
QString SchemaForTaskUtils::createTempSubDdir(qint64 taskId)
{
    QString tmpDirName = AppContext::getAppSettings()->getUserAppsSettings()->getTemporaryDirPath();

    if (tmpDirName.isEmpty()) {
        tmpDirName = QDir::tempPath();
    }

    tmpDirName += "/";

    QString subDirName = QString::number(taskId, 16);
    subDirName += QString::number(QDateTime::currentDateTime().toTime_t(), 16);

    if (!QDir(tmpDirName).mkdir(subDirName)) {
        return QString();
    }

    return QDir(tmpDirName + subDirName).path();
}

void SchemaForTaskUtils::removeTempSubDir(const QString &dirName) {
    QDir dir(dirName);
    if (dir.exists()) {
        QDirIterator it(dir);
        while (it.hasNext()) {
            QFile(it.next()).remove();
        }
        dir.rmdir(dir.path());
    }
}

}    // namespace U2
