/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtXml/qdom.h>

#include <U2Core/Log.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/Schema.h>
#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/QVariantUtils.h>
#include "SchemaSerializer.h"

//Q_DECLARE_METATYPE(U2::Workflow::CfgMap)

namespace U2 {
namespace Workflow {

const QMap<QString, QString> SchemaSerializer::ATTRIBUTE_ID_MAP = SchemaSerializer::initAttributesMap();
const QMap<QString, QString> SchemaSerializer::ELEM_TYPES_MAP = SchemaSerializer::initElemTypesMap();
const QMap<QString, QMap<QString, QString> > SchemaSerializer::ELEM_PORT_ID_MAP = SchemaSerializer::initElemPortIdMap();
const QMap<QString, QMap<QString, QString> > SchemaSerializer::PORT_SLOT_ID_MAP = SchemaSerializer::initPortSlotIdMap();

const QString SchemaSerializer::WORKFLOW_DOC = "GB2WORKFLOW";

static const QString WORKFLOW_EL = "workflow";
static const QString DOMAIN_EL = "workflow";
static const QString PROCESS_EL = "process";
static const QString ITERATION_EL = "iteration";
static const QString PORT_EL = "port";
static const QString PARAMS_EL = "params";
static const QString DATAFLOW_EL = "dataflow";
static const QString ID_ATTR = "id";
static const QString NAME_ATTR = "name";
static const QString TYPE_ATTR = "type";
static const QString SRC_PORT_ATTR = "sprt";
static const QString SRC_PROC_ATTR = "sprc";
static const QString DST_PORT_ATTR = "dprt";
static const QString DST_PROC_ATTR = "dprc";
static const QString ALIASES_EL = "paramAliases";
static const QString SCRIPT_TEXT = "scriptText";

static void saveConfiguration(const Configuration& cfg, QDomElement& owner) {
    QVariantMap qm;
    foreach (Attribute* a, cfg.getParameters()) {
        qm[a->getId()] = a->toVariant();
    }
    QDomElement el = owner.ownerDocument().createElement(PARAMS_EL);
    owner.appendChild(el);
    el.appendChild(owner.ownerDocument().createTextNode(QVariantUtils::map2String(qm)));
}

static void saveParamAliases(const QMap<QString, QString> & aliases, QDomElement & owner ) {
    QDomElement el = owner.ownerDocument().createElement( ALIASES_EL );

    QMap<QString, QString>::const_iterator it = aliases.constBegin();
    while( it != aliases.constEnd() ) {
        el.setAttribute( it.key(), it.value() );
        ++it;
    }
    owner.appendChild( el );
}

QMap<QString, QString> SchemaSerializer::initAttributesMap() {
    QMap<QString, QString> map;
    map["URL_IN"] = "url-in";
    map["URL_OUT"] = "url-out";
    return map;
}

//QMap<QString, QString> SchemaSerializer::initSlotMap() {
//    QMap<QString, QString> map;
//    map["URL_SLOT"] = CoreLibConstants::URL_SLOT_ID;
//    return map;
//}

QMap<QString, QMap<QString, QString> > SchemaSerializer::initPortSlotIdMap() {
    QMap<QString, QMap<QString, QString> > map;
    map[QString("read-msa")+QString("out-msa")].insert("url_slot", "url");
    map[QString("read-msa")+QString("out-msa")].insert("malignment", "msa");
    map[QString("fetch-sequence")+QString("out-sequence")].insert("seq", "sequence");
    map[QString("fetch-sequence")+QString("out-sequence")].insert("ann_table", "annotations");
    map[QString("read-sequence")+QString("out-sequence")].insert("url_slot", "url");
    map[QString("read-sequence")+QString("out-sequence")].insert("seq", "sequence");
    map[QString("read-sequence")+QString("out-sequence")].insert("ann_table", "annotations");
    map[QString("read-text")+QString("out-text")].insert("url_slot", "url");
    map[QString("read-text")+QString("out-text")].insert("string", "text");
    map[QString("write-msa")+QString("in-msa")].insert("url_slot", "url");
    map[QString("write-msa")+QString("in-msa")].insert("malignment", "msa");
    map[QString("write-clustalw")+QString("in-msa")].insert("url_slot", "url");
    map[QString("write-clustalw")+QString("in-msa")].insert("malignment", "msa");
    map[QString("write-fasta")+QString("in-sequence")].insert("url_slot", "url");
    map[QString("write-fasta")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("write-fasta")+QString("in-sequence")].insert("fasta.header", "fasta-header");
    map[QString("write-fastq")+QString("in-sequence")].insert("url_slot", "url");
    map[QString("write-fastq")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("write-genbank")+QString("in-sequence")].insert("url_slot", "url");
    map[QString("write-genbank")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("write-genbank")+QString("in-sequence")].insert("ann_table", "annotations");
    map[QString("write-text")+QString("in-text")].insert("url_slot", "url");
    map[QString("write-text")+QString("in-text")].insert("string", "text");
    map[QString("write-sequence")+QString("in-sequence")].insert("url_slot", "url");
    map[QString("write-sequence")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("write-sequence")+QString("in-sequence")].insert("ann_table", "annotations");
    map[QString("write-stockholm")+QString("in-msa")].insert("url_slot", "url");
    map[QString("write-stockholm")+QString("in-msa")].insert("malignment", "msa");
    map[QString("collocated-annotation-search")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("collocated-annotation-search")+QString("in-sequence")].insert("ann_table", "annotations");
    map[QString("collocated-annotation-search")+QString("out-annotations")].insert("ann_table", "annotations");
    map[QString("extract-annotated-sequence")+QString("in-sequence")].insert("ann_table", "annotations");
    map[QString("extract-annotated-sequence")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("extract-annotated-sequence")+QString("out-sequence")].insert("ann_table", "annotations");
    map[QString("extract-annotated-sequence")+QString("out-sequence")].insert("seq", "sequence");
    map[QString("repeats-search")+QString("in-sequence")].insert("seq", "sequence");
    map[QString("repeats-search")+QString("out-annotations")].insert("ann_table", "annotations");
    return map;
}

QString SchemaSerializer::getElemPortSlot(const QString& elemId, const QString portId, const QString& slot) {
    QString key = elemId + portId;
    if(!PORT_SLOT_ID_MAP.contains(key)) {
        return slot;
    }
    QMap<QString, QString> map = PORT_SLOT_ID_MAP.value(key);
    if(map.contains(slot)) {
        return map.value(slot);
    }
    return slot;
}

QMap<QString, QString> SchemaSerializer::initElemTypesMap() {
    QMap<QString, QString> map;
    map["read.malignment"] = "read-msa";
    map["read.remote"] = "fetch-sequence";
    map["read.sequence"] = "read-sequence";
    map["read.text"] = "read-text";
    map["write.malignment"] = "write-msa";
    map["write.clustalw"] = "write-clustalw";
    map["write.fasta"] = "write-fasta";
    map["write.fastq"] = "write-fastq";
    map["write.gbk"] = "write-genbank";
    map["write.text"] = "write-text";
    map["write.sequence"] = "write-sequence";
    map["write.stockholm"] = "write-stockholm";
    map["annotator.collocation"] = "collocated-annotation-search";
    map["sequence.extract"] = "extract-annotated-sequence";
    map["repeat.finder"] = "repeats-search";
    map["find.pattern"] = "search";
    map["import.phred.quality"] = "import-phred-qualities";
    map["blastall"] = "blast";
    map["blastPlus"] = "blast-plus";
    map["orf.marker"] = "orf-search";
    map["remote.query"] = "blast-ncbi";
    map["find.smithwaterman"] = "ssearch";
    map["bowtie.assembly"] = "bowtie";
    map["bowtie.indexer"] = "bowtie-build-index";
    map["bowtie.index.reader"] = "bowtie-read-index";
    map["uhmmer.build"] = "hmm2-build";
    map["uhmmer.search"] = "hmm2-search";
    map["uhmmer.read"] = "hmm2-read-profile";
    map["uhmmer.write"] = "hmm2-write-profile";
    map["clustalw.align"] = "clustalw";
    map["kalign.align"] = "kalign";
    map["mafft.align"] = "mafft";
    map["muscle.align"] = "muscle";
    map["tcoffee.align"] = "tcoffee";
    map["pfmatrix.build"] = "fmatrix-build";
    map["sitecon.build"] = "sitecon-build";
    map["pwmatrix.build"] = "wmatrix-build";
    map["pfmatrix.convert"] = "fmatrix-to-wmatrix";
    map["pfmatrix.read"] = "fmatrix-read";
    map["sitecon.read"] = "sitecon-read";
    map["pwmatrix.read"] = "wmatrix-read";
    map["sitecon.search"] = "sitecon-search";
    map["pwmatrix.search"] = "wmatrix-search";
    map["pfmatrix.write"] = "fmatrix-write";
    map["sitecon.write"] = "sitecon-write";
    map["pwmatrix.write"] = "wmatrix-write";
    return map;
}

QString SchemaSerializer::getElemType(const QString & t) {
    if(ELEM_TYPES_MAP.contains(t)) {
        return ELEM_TYPES_MAP.value(t);
    }
    return t;
}

QMap<QString, QMap<QString, QString> > SchemaSerializer::initElemPortIdMap() {
    QMap<QString, QMap<QString, QString> > map;
    map["read-msa"].insert("data", "out-msa");
    map["read-sequence"].insert("data", "out-sequence");
    map["fetch-sequence"].insert("out", "out-sequence");
    map["read-text"].insert("string", "out-text");
    map["write-msa"].insert("data", "in-msa");
    map["write-clustalw"].insert("data", "in-msa");
    map["write-fasta"].insert("data", "in-sequence");
    map["write-fastq"].insert("data", "in-sequence");
    map["write-genbank"].insert("data", "in-sequence");
    map["write-text"].insert("data", "in-text");
    map["write-sequence"].insert("data", "in-sequence");
    map["write-stockholm"].insert("data", "in-msa");
    map["collocated-annotation-search"].insert("in", "in-sequence");
    map["collocated-annotation-search"].insert("out", "out-annotations");
    map["extract-annotated-sequence"].insert("in", "in-sequence");
    map["extract-annotated-sequence"].insert("out", "out-sequence");
    map["repeats-search"].insert("in", "in-sequence");
    map["repeats-search"].insert("out", "out-annotations");
    map["search"].insert("in", "in-sequence");
    map["search"].insert("out", "out-annotations");
    map["import-phred-qualities"].insert("in", "in-sequence");
    map["import-phred-qualities"].insert("out", "out-sequence");
    map["blast"].insert("in", "in-sequence");
    map["blast"].insert("out", "out-annotations");
    map["blast-plus"].insert("in", "in-sequence");
    map["blast-plus"].insert("out", "out-annotations");
    map["orf-search"].insert("in", "in-sequence");
    map["orf-search"].insert("out", "out-annotations");
    map["blast-ncbi"].insert("in", "in-sequence");
    map["blast-ncbi"].insert("out", "out-annotations");
    map["ssearch"].insert("in", "in-sequence");
    map["ssearch"].insert("out", "out-annotations");
    map["bowtie"].insert("out", "out-msa");
    map["bowtie"].insert("reads", "in-sequence");
    map["bowtie"].insert("ebwt", "in-ebwt");
    map["bowtie-build-index"].insert("out", "out-ebwt");
    map["bowtie-read-index"].insert("out", "out-ebwt");
    map["hmm2-build"].insert("in", "in-msa");
    map["hmm2-build"].insert("out", "out-hmm2");
    map["hmm2-search"].insert("hmm", "in-hmm2");
    map["hmm2-search"].insert("seq", "in-sequence");
    map["hmm2-search"].insert("out", "out-annotations");
    map["hmm2-read-profile"].insert("out", "out-hmm2");
    map["hmm2-write-profile"].insert("in", "in-hmm2");
    map["clustalw"].insert("in", "in-msa");
    map["clustalw"].insert("out", "out-msa");
    map["kalign"].insert("in", "in-msa");
    map["kalign"].insert("out", "out-msa");
    map["mafft"].insert("in", "in-msa");
    map["mafft"].insert("out", "out-msa");
    map["muscle"].insert("in", "in-msa");
    map["muscle"].insert("out", "out-msa");
    map["tcoffee"].insert("in", "in-msa");
    map["tcoffee"].insert("out", "out-msa");
    map["fmatrix-build"].insert("in", "in-msa");
    map["fmatrix-build"].insert("out", "out-fmatrix"); 
    map["sitecon-build"].insert("in", "in-msa");
    map["sitecon-build"].insert("out", "out-sitecon");
    map["wmatrix-build"].insert("in", "in-msa");
    map["wmatrix-build"].insert("out", "out-wmatrix"); 
    map["fmatrix-to-wmatrix"].insert("in", "in-fmatrix");
    map["fmatrix-to-wmatrix"].insert("out", "out-wmatrix");
    map["fmatrix-read"].insert("out", "out-fmatrix");
    map["sitecon-read"].insert("out", "out-sitecon");
    map["wmatrix-read"].insert("out", "out-wmatrix");    
    map["sitecon-search"].insert("out", "out-annotations");
    map["sitecon-search"].insert("seq", "in-sequence");
    map["sitecon-search"].insert("model", "in-sitecon");
    map["wmatrix-search"].insert("out", "out-annotations");
    map["wmatrix-search"].insert("seq", "in-sequence");
    map["wmatrix-search"].insert("model", "in-wmatrix");
    map["fmatrix-write"].insert("in", "in-fmatrix");
    map["sitecon-write"].insert("in", "in-sitecon");
    map["wmatrix-write"].insert("in", "in-wmatrix");
    return map;
}

QString SchemaSerializer::getElemPort(const QString & elemId, const QString & portId) {
    if(!ELEM_PORT_ID_MAP.contains(elemId)){
        return portId;
    }
    QMap<QString, QString> map = ELEM_PORT_ID_MAP.value(elemId);
    if(map.contains(portId)) {
        return map.value(portId);
    }
    return portId;
}

QDomElement SchemaSerializer::saveActor(const Actor* proc, QDomElement& proj) {
    QDomElement docElement = proj.ownerDocument().createElement(PROCESS_EL);
    
    docElement.setAttribute(ID_ATTR, proc->getId());
    docElement.setAttribute(TYPE_ATTR, proc->getProto()->getId());
    docElement.setAttribute(NAME_ATTR, proc->getLabel());
    docElement.setAttribute(SCRIPT_TEXT, proc->getScript() == 0? "" : proc->getScript()->getScriptText());
    
    saveConfiguration(*proc, docElement);
    saveParamAliases( proc->getParamAliases(), docElement );
    
    proj.appendChild(docElement);
    return docElement;
}

static Actor* readActor(const QDomElement& procElement) {
    const QString name = procElement.attribute(TYPE_ATTR);
    ActorPrototype* proto = WorkflowEnv::getProtoRegistry()->getProto(name);
    if (!proto) {
        return NULL;
    }

    Actor* proc = proto->createInstance();
    if (proc) {
        SchemaSerializer::readConfiguration(proc, procElement);
        proc->setLabel(procElement.attribute(NAME_ATTR));
        SchemaSerializer::readParamAliases( proc->getParamAliases(), procElement );
    }
    return proc;
}

QDomElement SchemaSerializer::saveLink(const Link* link, QDomElement& proj) {
    QDomElement docElement = proj.ownerDocument().createElement(DATAFLOW_EL);
    docElement.setAttribute(SRC_PORT_ATTR, link->source()->getId());
    docElement.setAttribute(SRC_PROC_ATTR, link->source()->owner()->getId());
    docElement.setAttribute(DST_PORT_ATTR, link->destination()->getId());
    docElement.setAttribute(DST_PROC_ATTR, link->destination()->owner()->getId());
    proj.appendChild(docElement);
    return docElement;
}

QDomElement SchemaSerializer::savePort(const Port* port, QDomElement& owner) {
    QDomElement el = owner.ownerDocument().createElement(PORT_EL);
    el.setAttribute(ID_ATTR, port->getId());
    saveConfiguration(*port, el);
    owner.appendChild(el);
    return el;
}


void SchemaSerializer::schema2xml(const Schema& schema, QDomDocument& xml) {
    QDomElement projectElement = xml.createElement(WORKFLOW_EL);
    xml.appendChild(projectElement);
    foreach(Actor* a, schema.getProcesses()) {
        QDomElement el = saveActor(a, projectElement);
        foreach(Port* p, a->getPorts()) {
            savePort(p, el);
        }
    }
    foreach(Link* l, schema.getFlows()) {
        saveLink(l, projectElement);
    }
    QDomElement el = xml.createElement(DOMAIN_EL);
    el.setAttribute(NAME_ATTR, schema.getDomain());
    projectElement.appendChild(el);
}

void SchemaSerializer::saveIterations(const QList<Iteration>& lst, QDomElement& proj) {
    foreach(const Iteration& it, lst) {
        QDomElement el = proj.ownerDocument().createElement(ITERATION_EL);
        el.setAttribute(ID_ATTR, it.id);
        el.setAttribute(NAME_ATTR, it.name);
        QVariant v = qVariantFromValue<CfgMap>(it.cfg);
        el.appendChild(proj.ownerDocument().createTextNode(QVariantUtils::var2String(v)));
        proj.appendChild(el);
    }
}
void SchemaSerializer::readIterations(QList<Iteration>& lst, const QDomElement& proj, const QMap<ActorId, ActorId>& remapping) {
    QDomNodeList paramNodes = proj.elementsByTagName(ITERATION_EL);
    for(int i=0; i<paramNodes.size(); i++) {
        QDomElement el = paramNodes.item(i).toElement();
        if (el.isNull()) continue;
        Iteration it(el.attribute(NAME_ATTR));
        if (el.hasAttribute(ID_ATTR)) {
            it.id = el.attribute(ID_ATTR).toInt();
        }
        QVariant var = QVariantUtils::String2Var(el.text());
        if (qVariantCanConvert<CfgMap>(var)) {
            it.cfg = var.value<CfgMap>();
            foreach(const QString & oldAttrId, ATTRIBUTE_ID_MAP.keys()) {
                foreach(const ActorId& actorId, it.cfg.keys()) {
                    if(it.cfg[actorId].contains(oldAttrId)) {
                        it.cfg[actorId][ATTRIBUTE_ID_MAP.value(oldAttrId)] = it.cfg[actorId][oldAttrId];
                        it.cfg[actorId].remove(oldAttrId);
                    }
                }
            }
        }
        if (qVariantCanConvert<IterationCfg>(var)) {
            IterationCfg tmp = var.value<IterationCfg>();
            QMapIterator<IterationCfgKey, QVariant> tit(tmp);
            while (tit.hasNext())
            {
                tit.next();
                it.cfg[tit.key().first].insert(tit.key().second, tit.value());
            }
        }
        it.remap(remapping);
        lst.append(it);
    }
}

void SchemaSerializer::readConfiguration(Configuration* cfg, const QDomElement& owner) {
    QDomNodeList paramNodes = owner.elementsByTagName(PARAMS_EL);
    for(int i=0; i<paramNodes.size(); i++) {
        const QVariantMap& qm = QVariantUtils::string2Map(paramNodes.item(i).toElement().text(), true);
        QMapIterator<QString, QVariant> it(qm);
        while (it.hasNext()) {
            it.next();
            QVariant val = it.value();
            QString keyStr = ATTRIBUTE_ID_MAP.contains(it.key()) ? ATTRIBUTE_ID_MAP.value(it.key()) : it.key();
            if(keyStr == IntegralBusPort::BUS_MAP_ATTR_ID) {
                /*Port * p = dynamic_cast<Port*>(cfg);
                assert(p != NULL);
                QString elemId = p->owner()->getProto()->getId();
                QString portId = p->getId();
                QStrStrMap busMap = it.value().value<QStrStrMap>();
                foreach(const QString & s, busMap.keys()) {
                    QString slot = getElemPortSlot(elemId, portId, s);
                    if(slot != s) {
                        busMap[slot] = busMap.value(s);
                        busMap.remove(s);
                    }
                }
                val = qVariantFromValue<QStrStrMap>(busMap);*/
            }
            if(cfg->hasParameter(keyStr)) {
                cfg->getParameter(keyStr)->fromVariant(val);
            }
        }
    }
}

void SchemaSerializer::readParamAliases( QMap<QString, QString> & aliases, const QDomElement& owner ) {
    QDomNodeList alisesNodes = owner.elementsByTagName( ALIASES_EL );
    int sz = alisesNodes.size();
    for( int i = 0; i < sz; ++i ) {
        QDomNamedNodeMap map = alisesNodes.at( i ).toElement().attributes();
        int mapSz = map.length();
        for( int j = 0; j < mapSz; ++j ) {
            QDomNode node = map.item( j );
            QString nodeName = node.nodeName();
            QString nodeValue = node.nodeValue();
            aliases.insert( nodeName, nodeValue );
        }
    }
}

static const QString META_EL = "info";

static void saveMeta(const Workflow::Metadata* meta, QDomElement& proj){
    QDomElement el = proj.ownerDocument().createElement(META_EL);
    proj.appendChild(el);
    el.setAttribute(NAME_ATTR, meta->name);
    el.appendChild(proj.ownerDocument().createCDATASection(meta->comment));
}

QString SchemaSerializer::readMeta(Workflow::Metadata* meta, const QDomElement& proj) {
    QDomElement el = proj.elementsByTagName(META_EL).item(0).toElement();
    meta->name = el.attribute(NAME_ATTR);
    meta->comment = el.text();
    return el.isNull() ? tr("no metadata") : QString();
}

static Port * findPort(const QList<Actor*> & procs, const ActorId & actorId, const QString & portId) {
    foreach(Actor * a, procs) {
        if(a->getId() == actorId) {
            foreach(Port * p, a->getPorts()) {
                if(p->getId() == portId) {
                    return p;
                }
            }
            return NULL;
        }
    }
    return NULL;
}

void SchemaSerializer::updatePortBindings(const QList<Actor*> & procs) {
    foreach(Actor * actor, procs) {
        foreach(Port * p, actor->getInputPorts()) {
            IntegralBusPort * port = qobject_cast<IntegralBusPort*>(p);
            QStrStrMap busMap = port->getParameter(IntegralBusPort::BUS_MAP_ATTR_ID)->getAttributeValue<QStrStrMap>();
            foreach(const QString & key, busMap.uniqueKeys()) {
                QString val = busMap.value(key);
                QStringList vals = val.split(":", QString::SkipEmptyParts);
                if(vals.size() == 2) {
                    ActorId actorId = str2aid(vals.at(0));
                    QString slot = vals.at(1);
                    Port * inP = findPort(procs, actorId, slot); if(!inP) {continue;}
                    DataTypePtr inPType = inP->Port::getType();
                    QMap<Descriptor, DataTypePtr> inPTypeMap = inPType->getDatatypesMap();
                    if(inP != NULL && inPType->isMap() && inPTypeMap.keys().size() == 1) {
                        Descriptor d = inPTypeMap.keys().at(0);
                        QString newVal = actorId + ":" + d.getId();
                        coreLog.details(QString("remapping old xml schema for key %1: old value: %2, new value: %3").
                            arg(key).arg(val).arg(newVal));
                        port->setBusMapValue(key, newVal);
                    }
                }
            }
        }
    }
}

QString SchemaSerializer::xml2schema(const QDomElement& projectElement, Schema* schema, QMap<ActorId, ActorId>& idmap, bool stopIfError) {
    QMap<ActorId, Actor*> procMap;

    QDomElement domainEl = projectElement.elementsByTagName(DOMAIN_EL).item(0).toElement();
    if (!domainEl.isNull()) {
        schema->setDomain(domainEl.attribute(NAME_ATTR));
    }

    ActorPrototypeRegistry* registry = WorkflowEnv::getProtoRegistry();

    QDomNodeList procNodes = projectElement.elementsByTagName(PROCESS_EL);
    for(int i=0; i<procNodes.size(); i++) {
        QDomElement procElement = procNodes.item(i).toElement();
        if (projectElement.isNull()) {
            continue;
        }

        const ActorId id = str2aid(procElement.attribute(ID_ATTR));
        if (stopIfError && procMap.contains(id)) {
            return tr("Invalid content: duplicate process %1").arg(id);
        }

        const QString name = getElemType(procElement.attribute(TYPE_ATTR));
        ActorPrototype* proto = registry->getProto(name);
        if (!proto) {
            if (stopIfError) {
                return tr("Invalid content: unknown process type %1").arg(name);
            } else {
                continue;
            }
        }

        AttributeScript *script;
        const QString scriptText = procElement.attribute(SCRIPT_TEXT);
        if(scriptText.isEmpty()) {
            script = NULL;
        }
        else {
            script = new AttributeScript();
            script->setScriptText(scriptText);
        }
        Actor* proc = proto->createInstance(script);
        readConfiguration(proc, procElement);
        readParamAliases( proc->getParamAliases(), procElement );
        proc->setLabel(procElement.attribute(NAME_ATTR));
        procMap[id] = proc;
        schema->addProcess(proc);

        //read port params
        QDomNodeList nl = procElement.elementsByTagName(PORT_EL);
        for(int j=0; j<nl.size(); j++) {
            QDomElement el = nl.item(j).toElement();
            if (el.isNull()) continue;
            QString eid = getElemPort(name, el.attribute(ID_ATTR));
            Port* p = proc->getPort(eid);
            if (!p) {
                if (stopIfError) {
                    return tr("Invalid content: unknown port %1 requested for %2").arg(eid).arg(name);
                } else {
                    continue;
                }
            }
            readConfiguration(p, el);
        }
    }

    QMapIterator<ActorId, Actor*> it(procMap);
    while(it.hasNext()) {
        it.next();
        idmap[it.key()] = it.value()->getId();
    }
    foreach(Actor* a, procMap) {
        a->remap(idmap);
    }

    QDomNodeList flowNodes = projectElement.elementsByTagName(DATAFLOW_EL);
    for(int i=0; i<flowNodes.size(); i++) {
        QDomElement flowElement = flowNodes.item(i).toElement();
        if (flowElement.isNull()) {
            continue;
        }
        const ActorId inId = str2aid(flowElement.attribute(DST_PROC_ATTR));
        const ActorId outId = str2aid(flowElement.attribute(SRC_PROC_ATTR));

        if (!procMap.contains(inId)) {
            if (stopIfError) {
                return tr("Invalid content: no such process %1 to bind").arg(inId);
            } else {
                continue;
            }
        }
        if (!procMap.contains(outId)) {
            if (stopIfError) {
                return tr("Invalid content: no such process %1 to bind").arg(outId);
            } else {
                continue;
            }
        }
        QString inP = flowElement.attribute(DST_PORT_ATTR);
        QString outP = flowElement.attribute(SRC_PORT_ATTR);

        Port* input = procMap[inId]->getPort(inP);
        Port* output = procMap[outId]->getPort(outP);
        if ((!input || !output || !input->canBind(output))) {
            if (stopIfError) {
                return tr("Invalid content: cannot bind [%1 : %2] to [%3 : %4]").
                    arg(inId).arg(inP).arg(outId).arg(outP);
            }
        } else {
            Link* l = new Link(input, output);
            schema->addFlow(l);
        }
    }
    updatePortBindings(procMap.values());
    
    return QString();
}

}//namespace Workflow
}//namespace U2
