/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BuildIndexDialog.h"
#include "DnaAssemblyGUIExtension.h"

#include <U2Algorithm/DnaAssemblyAlgRegistry.h>
#include <U2Core/GUrlUtils.h>
#include <U2Misc/DialogUtils.h>
#include <U2Core/DocumentUtils.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>


namespace U2 {

QString BuildIndexDialog::genomePath;

BuildIndexDialog::BuildIndexDialog(const DnaAssemblyAlgRegistry* registry, QWidget* p)
: QDialog(p), assemblyRegistry(registry), customGUI(NULL)
{
	setupUi(this);
	QStringList names = registry->getRegisteredAlgorithmsWithIndexFileSupport();
	methodNamesBox->addItems(names);
	// TODO: change the way default method is set
	if (names.size() > 0) {
		methodNamesBox->setCurrentIndex(names.size() - 1);
	}
	sl_onAlgorithmChanged(methodNamesBox->currentText());
	connect(setIndexFileNameButton, SIGNAL(clicked()), SLOT(sl_onSetIndexFileNameButtonClicked()));
	connect(addRefButton, SIGNAL(clicked()), SLOT(sl_onAddRefButtonClicked()) );
	connect(methodNamesBox, SIGNAL(currentIndexChanged(const QString &)), SLOT(sl_onAlgorithmChanged(const QString &)));

	if (!genomePath.isEmpty()) {
		refSeqEdit->setText(genomePath);
		buildIndexUrl(genomePath);
	}
}

void BuildIndexDialog::sl_onAddRefButtonClicked() {
	LastOpenDirHelper lod;
	QString filter;

	lod.url = QFileDialog::getOpenFileName(this, tr("Open reference sequence"), lod.dir, filter);
	if (lod.url.isEmpty()) {
		return;
	}

	refSeqEdit->setText(lod.url);
	buildIndexUrl(lod.url);
}

void BuildIndexDialog::sl_onSetIndexFileNameButtonClicked() {
	LastOpenDirHelper lod;
	lod.url = QFileDialog::getSaveFileName(this, tr("Set index file name"), lod.dir);
	if (!lod.url.isEmpty()) {
		GUrl index = lod.url;
		if (index.lastFileSuffix().isEmpty() && customGUI != NULL) {
			QString extension = customGUI->getIndexFileExtension();
			if (extension.isEmpty()) {
				index = QString( "%1" ).arg( index.getURLString() );
			} else {
				index = QString( "%1.%2" ).arg( index.getURLString() ).arg(extension);
			}
		}
		indexFileNameEdit->setText(index.getURLString());
	}
}

void BuildIndexDialog::sl_onAlgorithmChanged(const QString &) {
	updateState();
}

void BuildIndexDialog::updateState() {
	addGuiExtension();
}

void BuildIndexDialog::addGuiExtension() {
	int insertPos = verticalLayout->count() - 2;

	// cleanup previous extension
	if (customGUI != NULL) {
		layout()->removeWidget(customGUI);         
		setMinimumHeight(minimumHeight() - customGUI->minimumHeight());
		delete customGUI;
		customGUI = NULL;
	}

	// insert new extension widget
	DnaAssemblyAlgorithmEnv* env = assemblyRegistry->getAlgorithm(methodNamesBox->currentText());
	if (NULL == env) {
		adjustSize();
		return;
	}
    DnaAssemblyGUIExtensionsFactory* gui = env->getGUIExtFactory();
    if (gui!=NULL && gui->hasBuildIndexWidget()) {
		customGUI = gui->createBuildIndexWidget(this);
		int extensionMinWidth = customGUI->sizeHint().width();
		int extensionMinHeight = customGUI->sizeHint().height();
		customGUI->setMinimumWidth(extensionMinWidth);
		customGUI->setMinimumHeight(extensionMinHeight);
		verticalLayout->insertWidget(insertPos, customGUI);
		// adjust sizes
		setMinimumHeight(customGUI->minimumHeight() + minimumHeight());
		if (minimumWidth() < customGUI->minimumWidth()) {
			setMinimumWidth(customGUI->minimumWidth());
		}
		if (!refSeqEdit->text().isEmpty()) {
			buildIndexUrl(refSeqEdit->text());
		}
		customGUI->show();
	} else {
		adjustSize();
	}
}

void BuildIndexDialog::buildIndexUrl(const GUrl& refUrl ) {
	QString extension("");
	if (NULL != customGUI) {
		extension = customGUI->getIndexFileExtension();
	}
	GUrl url;
	if (extension.isEmpty()) {
		url = GUrlUtils::rollFileName(refUrl.dirPath() + "/" + refUrl.baseFileName(), DocumentUtils::getNewDocFileNameExcludesHint());
	} else {
		url = GUrlUtils::rollFileName(refUrl.dirPath() + "/" + refUrl.baseFileName()+ "." + extension, DocumentUtils::getNewDocFileNameExcludesHint());
	}
	
	indexFileNameEdit->setText(url.getURLString());
}

void BuildIndexDialog::accept()
{
	if (refSeqEdit->text().isEmpty()) {
		QMessageBox::information(this, tr("Build index"), tr("Reference sequence url is not set!") );
	} else if (indexFileNameEdit->text().isEmpty() ) {
		QMessageBox::information(this, tr("Build index"), tr("Index file name is not set!") );
	} else {   
		genomePath.clear();
		genomePath = refSeqEdit->text();

		QDialog::accept();
	}
}

const GUrl BuildIndexDialog::getRefSeqUrl() {
	return refSeqEdit->text();
}

const QString BuildIndexDialog::getAlgorithmName() {
	return methodNamesBox->currentText();
}

const QString BuildIndexDialog::getIndexFileName() {
	return indexFileNameEdit->text();
}

QMap<QString, QVariant> BuildIndexDialog::getCustomSettings() {
	if (customGUI != NULL) {
		return customGUI->getBuildIndexCustomSettings();
	} else {
		return QMap<QString, QVariant>();
	}
}

}//namespace
