/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SmithWatermanSchemaTask.h"

#include <U2Core/AppContext.h>
#include <U2Core/BaseDocumentFormats.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/AnnotationTableObject.h>
#include <U2Core/LRegion.h>

#include <U2Core/Log.h>

namespace U2
{

/* class SmithWatermanSchemaTask: public Task */

const QString SmithWatermanSchemaTask::SCHEMA_NAME = "sw";

SmithWatermanSchemaTask::SmithWatermanSchemaTask(const SmithWatermanSettings &swSettings_,
                                                    AnnotationTableObject *annotationObj_, const QString &aname_, const QString &agroup_) :
        Task(tr("SmithWatermanSchemaTask"), TaskFlags_NR_FOSCOE),
        IO_FORMATID(BaseDocumentFormats::PLAIN_GENBANK),
        schemaParameters(),
        tempSubDir(), inFile(), outFile(),
        swSettings(swSettings_),
        annotationObj(annotationObj_),
        aname(aname_), agroup(agroup_),
        runTask(0), saveInputFileTask(0), openResultFileTask(0)
{
    schemaParameters = convertSWSettings(swSettings);
    schemaParameters["sw_ann_name"] = QString(aname);
}

SmithWatermanSchemaTask::~SmithWatermanSchemaTask()
{
    SchemaForTaskUtils::removeTempSubDir(tempSubDir);
}

RunSchemaForTask* SmithWatermanSchemaTask::prepareRunSchemaTask()
{
    return new RunSchemaForTask(SCHEMA_NAME, schemaParameters, inFile, outFile);
}

LoadDocumentTask* SmithWatermanSchemaTask::prepareOpenResultFileTask()
{
    IOAdapterFactory *iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    assert(iof);

    return new LoadDocumentTask(IO_FORMATID, outFile, iof);
}

Document* SmithWatermanSchemaTask::createInputDocument()
{
    LRegion reg = swSettings.globalRegion;
    DNAAlphabet *al = AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::RAW());
    assert(al);

    DNASequence *seq = new DNASequence(swSettings.sqnc.mid(reg.startPos, reg.len), al);

    DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(IO_FORMATID);
    assert(df);

    IOAdapterFactory *iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    assert(iof);

    QList<GObject*> objs;
    objs << new DNASequenceObject("data", *seq);

    assert(!inFile.isEmpty());
    Document *doc = new Document(df, iof, GUrl(inFile), objs);

    return doc;
}

void SmithWatermanSchemaTask::prepare()
{
    tempSubDir = SchemaForTaskUtils::createTempSubDdir(getTaskId());
    if (tempSubDir.isEmpty())
    {
        setError(tr("Unable to create temporary files"));
        return;
    }

    inFile = tempSubDir + "/in.gb";
    outFile = tempSubDir + "/out.gb";

    Document *doc = createInputDocument();
    assert(doc);

    saveInputFileTask = new SaveDocumentTask(doc, SaveDoc_Overwrite);
    addSubTask(saveInputFileTask);
}

QList<Task*> SmithWatermanSchemaTask::onSubTaskFinished(Task *subTask)
{
    assert(subTask);
    QList<Task*> subTasks;

    propagateSubtaskError();
    if (hasErrors() || isCanceled())
    {
        return subTasks;
    }

    if (saveInputFileTask == subTask)
    {
        runTask = prepareRunSchemaTask();
        subTasks << runTask;
    }
    else if (runTask == subTask)
    {
        openResultFileTask = prepareOpenResultFileTask();
        subTasks << openResultFileTask;
    }
    else if (openResultFileTask == subTask)
    {
        Document *doc = openResultFileTask->getDocument();
        QList <GObject*> atos = doc->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
        GObject *go = atos.first();
        AnnotationTableObject *ato = qobject_cast<AnnotationTableObject*>(go);

        QList<Annotation*> annotations;
        foreach (Annotation *a, ato->getAnnotations())
        {
            annotations.append(new Annotation(a->data()));
        }

        foreach (Annotation *a, annotations)
        {
            QList<LRegion> rs = a->getLocation();
            rs.first().startPos += swSettings.globalRegion.startPos;
            a->replaceLocationRegions(rs);
        }
        annotationObj->addAnnotations(annotations, agroup);

        openResultFileTask->takeDocument()->unload();
    }

    return subTasks;
}

QVariantMap SmithWatermanSchemaTask::convertSWSettings(const SmithWatermanSettings &settings)
{
    QVariantMap schemaSettings;

    // There are some source describing how we converting
    /* see util_smith_waterman/SmithWatermanSettings.h
    SmithWatermanSettings.member // corresponding_alias
    QByteArray ptrn;    // sw_pattern
    QByteArray sqnc;    // --

    LRegion globalRegion;   // --
    StrandOption strand;    // sw_strand

    float percentOfScore;   // sw_min_score
    GapModel gapModel;      // sw_gap_open/sw_gap_ext
    SMatrix pSm;            // sw_score_matrix

    DNATranslation* complTT;
    DNATranslation* aminoTT;
    */

    /* see plugins/workflow_designer/src/library/SWWorker.cpp
    Descriptor nd(NAME_ATTR, FindWorker::tr("Annotate as"), SWWorker::tr("Name of the result annotations marking found regions."));
    Descriptor pd(PATTERN_ATTR, FindWorker::tr("Pattern"), SWWorker::tr("A subsequence pattern to look for."));
    Descriptor scd(SCORE_ATTR, SWWorker::tr("Min score"), SWWorker::tr("The search stringency."));
    Descriptor ald(ALGO_ATTR, SWWorker::tr("Algorithm"), SWWorker::tr("Algorithm version."));
    Descriptor amd(AMINO_ATTR, FindWorker::tr("Search in translation"), SWWorker::tr("Translate a supplied nucleotide sequence to protein then search in the translated sequence."));
    Descriptor sd(STRAND_ATTR, FindWorker::tr("Search in"), SWWorker::tr("Which strands should be searched: direct, complement or both."));
    Descriptor mxd(MATRIX_ATTR, SWWorker::tr("Scoring matrix"), SWWorker::tr("The scoring matrix."));
    Descriptor frd(FILTER_ATTR, SWWorker::tr("Filter results"), SWWorker::tr("Result filtering strategy."));
    Descriptor god(GAPOPEN_ATTR, SWWorker::tr("Gap open score"), SWWorker::tr("Gap open score."));
    Descriptor ged(GAPEXT_ATTR, SWWorker::tr("Gap ext score"), SWWorker::tr("Gap extension score."));

    a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "misc_feature");
    a << new Attribute(pd, CoreDataTypes::STRING_TYPE(), true);
    a << new Attribute(mxd, CoreDataTypes::STRING_TYPE(), true, QString("Auto"));
    a << new Attribute(ald, CoreDataTypes::STRING_TYPE(), true);
    a << new Attribute(frd, CoreDataTypes::STRING_TYPE(), false, defFilter);
    a << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 90);
    a << new Attribute(sd, CoreDataTypes::NUM_TYPE(), false, StrandOption_Both);
    a << new Attribute(amd, CoreDataTypes::BOOL_TYPE(), false, false);
    a << new Attribute(god, CoreDataTypes::NUM_TYPE(), false, -10.);
    a << new Attribute(ged, CoreDataTypes::NUM_TYPE(), false, -1.);
    */

    schemaSettings["sw_ann_name"] = QString("");
    schemaSettings["sw_pattern"] = QString(settings.ptrn);
    schemaSettings["sw_strand"] = int(settings.strand);
    schemaSettings["sw_min_score"] = float(settings.percentOfScore);
    schemaSettings["sw_gap_open"] = float(settings.gapModel.scoreGapOpen);
    schemaSettings["sw_gap_ext"] = float(settings.gapModel.scoreGapExtd);
    schemaSettings["sw_score_matrix"] = QString(settings.pSm.getName());

    return schemaSettings;
}


}    // namespace U2
