/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_FIND_ENZYMES_TASK_H_
#define _U2_FIND_ENZYMES_TASK_H_

#include "EnzymeModel.h"
#include "FindEnzymesAlgorithm.h"

#include <limits>

#include <U2Core/Task.h>
#include <U2Core/LRegion.h>
#include <U2Core/DNASequence.h>
#include <U2Core/SequenceWalkerTask.h>

#include <U2Core/AnnotationTableObject.h>

#include <QtCore/QObject>
#include <QtCore/QList>
#include <QtCore/QMutex>

namespace U2 {

class FindEnzymesAlgResult {
public:
    FindEnzymesAlgResult() {pos = -1;}
    FindEnzymesAlgResult(const SEnzymeData& _enzyme, int _pos) : enzyme(_enzyme), pos(_pos){}
    SEnzymeData enzyme;
    int         pos;
};

class FindEnzymesTask;

struct FindEnzymesTaskConfig {
    FindEnzymesTaskConfig(): maxResults(0x7FFFFFFF), minHitCount(1),
        maxHitCount(INT_MAX), circular(false) {}; 
    QList<LRegion> region;
    QString groupName;
    int maxResults;
    int minHitCount;
    int maxHitCount;
    bool circular;
};

class FindEnzymesToAnnotationsTask : public Task {
    Q_OBJECT
public:
    FindEnzymesToAnnotationsTask(AnnotationTableObject* aobj, const DNASequence& seq, 
        const QList<SEnzymeData>& enzymes, const FindEnzymesTaskConfig& cfg);

    ReportResult report();
private:
    int maxHitCount;
    int minHitCount;
    bool circular;
    QList<SEnzymeData>                  enzymes;
    LRegion                             seqRange;
    QList<LRegion>                      regions;
    QPointer<AnnotationTableObject>     aObj;
    QString                             groupName;
    FindEnzymesTask*                    fTask;
};

class FindEnzymesTask : public Task, public FindEnzymesAlgListener {
    Q_OBJECT
public:
    FindEnzymesTask(const DNASequence& seq, const LRegion& region, const QList<SEnzymeData>& enzymes, int maxResults = 0x7FFFFFFF, bool _circular = false);

    QList<FindEnzymesAlgResult>  getResults() const {return results;}

    virtual void onResult(int pos, const SEnzymeData& enzyme);

    ReportResult report();
    
    QList<SharedAnnotationData> getResultsAsAnnotations(const QString& enzymeId = QString()) const;

    void cleanup();

private:
    void registerResult(const FindEnzymesAlgResult& r);

    int                                 maxResults;
    bool                                circular;
    int                                 seqlen;
    QList<FindEnzymesAlgResult>         results;
    QMutex                              resultsLock;
    
    QString                             group;
    QPointer<AnnotationTableObject*>    aObj;
};


class FindSingleEnzymeTask: public Task, public FindEnzymesAlgListener, public SequenceWalkerCallback {
    Q_OBJECT
public:
    FindSingleEnzymeTask(const DNASequence& seq, const LRegion& region, const SEnzymeData& enzyme, 
                        FindEnzymesAlgListener* l = NULL, bool circular = false, int maxResults = 0x7FFFFFFF);
    
    QList<FindEnzymesAlgResult>  getResults() const {return results;}
    virtual void onResult(int pos, const SEnzymeData& enzyme);
    virtual void onRegion(SequenceWalkerSubtask* t, TaskStateInfo& ti);
    void cleanup();
private:
    DNASequence                 seq;
    DNASequence                 resseq;
    LRegion                     region;
    SEnzymeData                 enzyme;
    int                         maxResults;
    FindEnzymesAlgListener*     resultListener;
    QList<FindEnzymesAlgResult> results;
    QMutex                      resultsLock;
    bool                        circular;
};

} //namespace

#endif
