/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GenomeAlignerIndexTask.h"

#include <U2Core/DNASequenceObject.h>
#include <U2Core/DNASequence.h>
#include <QtEndian>
#include <QtGlobal>
#include "U2Formats/StreamSequenceReader.h"
#include "GenomeAlignerIndex.h"

namespace U2 {

GenomeAlignerIndexTask::GenomeAlignerIndexTask(const DNASequenceObject* obj, const GenomeAlignerIndexSettings &settings)
: Task("Building genome aligner's index", TaskFlag_None),
sArray(NULL), bitMask(NULL), buffer(NULL), objLens(NULL), seq(NULL), unknownChar('N')
{
    baseFileName = settings.indexFileName;
    newRefFile = new QFile(baseFileName + QString(".") + GenomeAlignerIndex::REF_INDEX_EXTENSION);
    w = settings.w;

    if (w <= 0) {
        setError("Incorrect index parameters: windowSize <= 0");
        return;
    }

    bitTable = bt.getBitMaskCharBits(DNAAlphabet_NUCL);
    bitCharLen = bt.getBitMaskCharBitsNum(DNAAlphabet_NUCL);

    index = new GenomeAlignerIndex();
    index->baseFileName = baseFileName;
    index->partsInMemCache = settings.partsInMemCache;
    this->settings = settings;
}

GenomeAlignerIndexTask::~GenomeAlignerIndexTask() {
    foreach(QFile *file, tempFiles) {
        file->close();
        QFile::remove(file->fileName());
        delete file;
        file = NULL;
    }
    if (NULL != newRefFile) {
        newRefFile->close();
        delete newRefFile;
    }

    delete[] sArray;
    delete[] bitMask;
    delete[] buffer;
}

void GenomeAlignerIndexTask::run() {
    if (settings.deserializeFromFile) {
        index->deserialize(stateInfo);
    } else {
        reformatSequence();
        if (isCanceled() || hasErrors()) {
            return;
        }
        seqLength = objLens[objCount-1];
        indexLength = seqLength - (w - 1)*objCount;
        int parts = indexLength/PART_SIZE + 1;

        wCharsInMask = qMin(62 / bitCharLen, w); //62 to avoid +- overflow
        wAfterBits =  qMax(0, w - wCharsInMask);
        if (wCharsInMask * bitCharLen == 64) {
            bitFilter = Q_UINT64_C(0xFFFFFFFFFFFFFFFF);
        } else {
            bitFilter = ((quint64)1<<(bitCharLen * wCharsInMask))-1;
        }
        index->seqLength = seqLength;
        index->w = w;
        index->indexLength = indexLength;
        index->wCharsInMask = wCharsInMask;
        index->wAfterBits = wAfterBits;

        for (int i=0; i<parts; i++) {
            QString tmpFileName = baseFileName + QString(".%1.tmp").arg(i);
            QFile *tempFile = new QFile(tmpFileName);
            tempFiles.append(tempFile);
        }

        int partLen = 0;
        if (1 == parts) {
            partLen = indexLength;
        } else {
            partLen = PART_SIZE;
        }
        sArray = new quint32[partLen];
        bitMask = new quint64[partLen];
        buffer = new char[BUFF_SIZE];

        quint32 idx = 0;
        quint32 arrLen = 0;
        int curObj = 0;
        int i = 0;
        newRefFile->open(QIODevice::ReadOnly);
        foreach(QFile *file, tempFiles) {
            if (isCanceled()) {
                return;
            }
            buildPart(&idx, &curObj, &arrLen);
            file->open(QIODevice::WriteOnly);
            writePart(file, arrLen);
            file->close();
            file->open(QIODevice::ReadOnly);

            stateInfo.progress = (0.75*i)/tempFiles.size();
            i++;
        }

        mergeSort();

        index->serialize(GUrl(settings.refFileName).fileName());
    }
    if (!index->openIndexFile()) {
        stateInfo.setError("Can't open sArray index's file");
        return;
    }
    index->createMemCache();
}

void GenomeAlignerIndexTask::buildPart(quint32 *idx, int *curObj, quint32 *arrLen) {
    QByteArray refPart;
    quint32 first = initPart(idx, curObj, arrLen, refPart);
    quint32 *arunner = sArray;
    quint64 *mrunner = bitMask;
    quint64 bitValue = 0;
    quint32 expectedNext = 0;
    quint32 wCharsInMask1 = wCharsInMask - 1;

    for (quint64 *end = mrunner + *arrLen; mrunner < end; arunner++, mrunner++) {
        const char* s = seq + *arunner;
        if (*arunner == expectedNext && expectedNext != 0) { //pop first bit, push wCharsInMask1 char to the mask
            bitValue = ((bitValue << bitCharLen) | bitTable[uchar(*(s + wCharsInMask1))]) & bitFilter;
#ifdef _DEBUG
            // double check that optimization doesn't break anything
            quint64 bitValue2 = index->getBitValue(s);
            assert(bitValue == bitValue2);
#endif
        } else {
            //recompute the mask if we have some symbols skipped
            bitValue = index->getBitValue(s);
        }
        expectedNext = (s + 1) - seq;
        *mrunner = bitValue;
    }

    if (isCanceled()) {
        seq = NULL;
        return;
    }

    sort(bitMask, 0, *arrLen);
    for (quint32 i=0; i<*arrLen; i++) {
        sArray[i] += first; //+first for correct sArray value
    }
    seq = NULL;
}

quint32 GenomeAlignerIndexTask::initPart(quint32 *idx, int *curObj, quint32 *arrLen, QByteArray &refPart) {
    *arrLen = 0;

    if (*idx > (objLens[*curObj]-w) && *idx < objLens[*curObj]) {
        *idx = objLens[*curObj];
        (*curObj)++;
    }
    //calculate first and last char of sequence for this part
    quint32 f = *idx;
    quint32 l = f;
    quint32 c = 0;

    for (int i=*curObj; i<objCount; i++) {
        c += (objLens[i] - w) - l + 1;
        if (c > PART_SIZE) {
            quint32 d = c - PART_SIZE;
            l = objLens[i] - w - d;
            break;
        }
        if (objCount - 1 == i) { //last iteration
            l = objLens[i] - w;
        } else {
            l = objLens[i];
        }
    }
    //now we khow how many symbols are needed to read:
    quint32 charsToRead = l - f + w;
    newRefFile->seek(f);
    refPart = newRefFile->read(charsToRead); //read a part of the sequence for sorting arrays
    seq = refPart.data();
    quint32 seqPartLen = refPart.length();

    quint32 *arunner = sArray;
    *arrLen = 0;
    bool goodSuff = false;
    quint32 seqIdx = 0;
    quint32 tmpIdx;
    quint32 first = *idx;
    for (quint32 i=0; i<PART_SIZE; i++) { //initializing sArray and arrLen
        if (*idx > (objLens[*curObj]-w) && *idx < objLens[*curObj]) {
            seqIdx += objLens[*curObj] - *idx;
            *idx = objLens[*curObj];
            (*curObj)++;
            goodSuff = false;
        }
        if (*idx >= seqLength || seqIdx >= seqPartLen) {
            break;
        }

        if (goodSuff) {
            if (seq[seqIdx + w-1]==unknownChar) {
                goodSuff = false;
                seqIdx += w;
                *idx += w;
                i--;
                continue;
            }
        }
        if (!goodSuff) {
            int goodChars = 0;
            tmpIdx = *idx;
            quint32 j=seqIdx;
            for (; j<seqPartLen && goodChars<w; j++) {
                if (tmpIdx == objLens[*curObj]) {
                    j += objLens[*curObj] - tmpIdx;
                    tmpIdx = objLens[*curObj];
                    (*curObj)++;
                    goodChars = 0;
                    j--;
                    continue;
                }
                if (seq[j]!=unknownChar) {
                    goodChars++;
                } else {
                    goodChars = 0;
                }
                tmpIdx++;
            }

            *idx = tmpIdx - goodChars;
            seqIdx = j - w;
            if (goodChars!=w) {
                break;
            }
            goodSuff = true;
        }

        *arunner = *idx - first; //-first for positioning at a part of the sequence
        (*arrLen)++;
        (*idx)++;
        arunner++;
        seqIdx++;
    }

    return first;
}

//Stable sort of sequences
void GenomeAlignerIndexTask::sort(quint64 *x, int off, int len) {
    // Insertion sort on smallest arrays
    if (len < 7) {
        for (int i=off; i<len+off; i++){
            for (int j=i; j > off && compare(x+j-1,x+j)>0; j--) {
                swap(x+j, x+j-1);
            }
        }
        return;
    }

    // Choose a partition element, v
    quint32 m = off + len / 2;       // Small arrays, middle element
    if (len > 7) {
        quint32 l = off;
        quint32 n = off + len - 1;
        if (len > 40) {        // Big arrays, pseudo median of 9
            quint32 s = len / 8;
            l = med3(x, l,     l+s, l+2*s);
            m = med3(x, m-s,   m,   m+s);
            n = med3(x, n-2*s, n-s, n);
        }
        m = med3(x, l, m, n); // Mid-size, med of 3
    }
    quint64 *v = x + m;

    // Establish Invariant: v* (<v)* (>v)* v*
    int a = off, b = a, c = off + len - 1, d = c;
    while(true) {
        qint64 cr;
        while (b <= c && (cr = compare(v, x+b)) >=0 ) {
            if (cr == 0) {
                (x+b==v) && (v=x+a);//save middle pos value
                swap(x+a++,x+b);
            }
            b++;
        }
        while (c >= b && (cr = compare(x+c, v)) >=0 ) {
            if (cr == 0) {
                (x+c==v) && (v=x+d);//save middle pos value
                swap(x+c, x+d--);
            }
            c--;
        }
        if (b > c) {
            break;
        }
        swap(x+b++, x+c--);
    }

    // Swap partition elements back to middle
    int s, n = off + len;
    s = qMin(a-off, b-a  ); vecswap(x+off, x+b-s, s);
    s = qMin(d-c,   n-d-1); vecswap(x+b,   x+n-s, s);

    // Recursively sort non-partition-elements
    if ((s = b-a) > 1) {
        sort(x, off, s);
    }
    if ((s = d-c) > 1) {
        sort(x, n-s, s);
    }
}

qint64 GenomeAlignerIndexTask::compare(const quint64 *x1, const quint64 *x2) const {
    qint64 rc = *x1-*x2;
    if ( rc != 0 || wAfterBits == 0) {
        return rc;
    }

    quint32 a1 = *(sArray+(x1-bitMask));
    quint32 a2 = *(sArray+(x2-bitMask));
    const char* b1 = seq + a1 + wCharsInMask;
    const char* b2 = seq + a2 + wCharsInMask;
    for (const char* end = b1+wAfterBits; b1 < end; b1++, b2++) {
        rc=*b1-*b2;
        if ( rc != 0 ) {
            return rc;
        }
    }
    return 0;
}

void GenomeAlignerIndexTask::swap(quint64 *x1, quint64 *x2) const {
    assert(x1 - bitMask >= 0 && x1 - bitMask < PART_SIZE);
    assert(x2 - bitMask >= 0 && x2 - bitMask < PART_SIZE);

    quint32 *a1 = sArray+(x1-bitMask);
    quint32 *a2 = sArray+(x2-bitMask);
    qSwap(*x1, *x2);
    qSwap(*a1, *a2);
}

quint32 GenomeAlignerIndexTask::med3(quint64 *x, quint32 a, quint32 b, quint32 c) {
    qint64 bc = compare(x+b, x+c);
    qint64 ac = compare(x+a, x+c);
    return compare(x+a, x+b) < 0 ?
        (bc < 0 ? b : ac < 0 ? c : a) :
        (bc > 0 ? b : ac > 0 ? c : a);
}

void GenomeAlignerIndexTask::vecswap(quint64 *x1, quint64 *x2, quint32 n) {
    for (quint32 i=0; i<n; i++) {
        swap(x1+i, x2+i);
    }
}

void GenomeAlignerIndexTask::writePart(QFile *file, quint32 arrLen) {
    int bufIdx = 0;
    int size32 = sizeof(quint32);
    int size64 = sizeof(quint64);

    for (quint32 i=0; i<arrLen; i++) {
        if (BUFF_SIZE == bufIdx) {
            file->write(buffer, BUFF_SIZE);
            bufIdx = 0;
        }
        qToBigEndian(sArray[i], (uchar*)(&buffer[bufIdx]));
        bufIdx += size32;
        qToBigEndian(bitMask[i], (uchar*)(&buffer[bufIdx]));
        bufIdx += size64;
    }
    file->write(buffer, bufIdx);
}

void GenomeAlignerIndexTask::mergeSort() {
    QFile indexFile(baseFileName+QString(".")+GenomeAlignerIndex::SARRAY_EXTENSION);
    indexFile.open(QIODevice::WriteOnly);
    QList<QFile*> files;
    int size = tempFiles.size();
    char **buffers = new char*[size];
    int *idxs = new int[size];
    int *sizes = new int[size];

    files.append(tempFiles);
    for (int i=0; i<size; i++) {
        buffers[i] = new char[BUFF_SIZE];
        idxs[i] = 0;
        sizes[i] = files.at(i)->read(buffers[i], BUFF_SIZE);
    }

    int idx = 0;
    quint32 a;
    quint64 b;
    int size32 = sizeof(quint32);
    int size64 = sizeof(quint64);
    quint32 aMin = 0;
    quint64 bMin = 0;
    int idxMin = 0;
    bool firstIter = true;
    uchar *tmp = NULL;
    int bufIdx = 0;

    do {
        int i = 0;
        firstIter = true;
        foreach (QFile *file, tempFiles) {
            if (0 == sizes[i]) {
                i++;
                continue;
            }
            if (idxs[i] == sizes[i]) {
                idxs[i] = 0;
                sizes[i] = files.at(i)->read(buffers[i], BUFF_SIZE);
                if (0 == sizes[i]) {
                    i++;
                    continue;
                }
            }

            idx = idxs[i];
            tmp = (uchar*)buffers[i];
            a = qFromBigEndian<quint32>(&tmp[idx]);
            idx += size32;
            b = qFromBigEndian<quint64>(&tmp[idx]);

            if (firstIter || slowCompare(b, bMin, a, aMin) < 0) {
                aMin = a;
                bMin = b;
                idxMin = i;
                firstIter = false;
            }
            i++;
        }
        if (!firstIter) {
            idxs[idxMin] += size32 + size64;

            if (BUFF_SIZE == bufIdx) {
                indexFile.write(buffer, BUFF_SIZE);
                bufIdx = 0;
            }
            qToBigEndian(aMin, (uchar*)(&buffer[bufIdx]));
            bufIdx += size32;
            qToBigEndian(bMin, (uchar*)(&buffer[bufIdx]));
            bufIdx += size64;
        }
    } while (!firstIter);

    indexFile.write(buffer, bufIdx);
    indexFile.close();

    for (int i=0; i<size; i++) {
        delete[] buffers[i];
    }
    delete[] buffers;
    delete[] idxs;
    delete[] sizes;
}

void GenomeAlignerIndexTask::reformatSequence() {
    StreamSequenceReader seqReader;
    QList<GUrl> urls;
    urls.append(GUrl(settings.refFileName));
    bool init = seqReader.init(urls);
    if (!init) {
        setError(tr("Can not init short reads loader. %1").arg(seqReader.getErrorMessage()));
        return;
    }

    objCount = 0;
    QList<quint32> seqLens;

    newRefFile->open(QIODevice::WriteOnly);
    while (seqReader.hasNext()) {
        objCount++;
        const DNASequenceObject *obj = seqReader.getNextSequenceObject();
        if (DNAAlphabet_NUCL != obj->getAlphabet()->getType()) {
            setError("Unsupported file format: object type is not NUCL");
            return;
        }
        const DNASequence &seq = obj->getDNASequence();
        seqLens.append(seq.length());
        newRefFile->write(seq.constData());
    }
    newRefFile->close();

    if (0 == objCount) {
        setError(QString("Ref sequence is not found in %1").arg(settings.refFileName));
        return;
    }
    index->objLens = new quint32[objCount];
    index->objCount = objCount;
    objLens = index->objLens;
    int i = 0;
    quint32 prev = 0;
    foreach (quint32 len, seqLens) {
        objLens[i] = prev + len;
        prev = objLens[i];
        i++;
    }
}

qint64 GenomeAlignerIndexTask::slowCompare(quint64 x1, quint64 x2, quint32 a1, quint32 a2) {
    qint64 rc = x1-x2;
    if ( rc != 0 || wAfterBits == 0) {
        return rc;
    }

    newRefFile->seek(a1 + wCharsInMask);
    QByteArray by1 = newRefFile->read(wAfterBits);
    newRefFile->seek(a2 + wCharsInMask);
    QByteArray by2 = newRefFile->read(wAfterBits);
    char *b1 = by1.data();
    char *b2 = by2.data();
    for (const char* end = b1+wAfterBits; b1 < end; b1++, b2++) {
        rc=*b1-*b2;
        if ( rc != 0 ) {
            return rc;
        }
    }
    return 0;
}

} //namespace
