/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemoteBLASTTask.h"
#include <U2Core/CreateAnnotationTask.h>
#include <U2Core/Counter.h>
#include <U2Core/TextUtils.h>
#include <U2Core/DocumentModel.h>
#include <U2Core/ProjectModel.h>

#include <U2Core/BaseDocumentFormats.h>

namespace U2 {

class BaseIOAdapters;
class BaseDocumentFormats;


RemoteBLASTToAnnotationsTask::RemoteBLASTToAnnotationsTask( const RemoteBLASTTaskSettings & _cfg, int _qoffs, 
                                AnnotationTableObject* _ao, const QString &_url,const QString & _group):
Task( tr("RemoteBLASTTask"), TaskFlags_NR_FOSCOE ), offsInGlobalSeq(_qoffs), aobj(_ao), group(_group), url(_url) {
    GCOUNTER( cvar, tvar, "RemoteBLASTToAnnotationsTask" );
    
    queryTask = new RemoteBLASTTask(_cfg);
    addSubTask(queryTask);
}


QList<Task*> RemoteBLASTToAnnotationsTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    
    if(subTask->hasErrors() && subTask == queryTask) {
        stateInfo.setError(subTask->getError());
        return res;
    }

    if (hasErrors() || isCanceled()) {
        return res;
    }

    if (aobj.isNull()) {
        stateInfo.setError(  tr("The object was removed\n") );
        return res;
    }
    if (subTask == queryTask) {
        //shift annotations according to offset first
        
        RemoteBLASTTask * rrTask = qobject_cast<RemoteBLASTTask *>(queryTask);
        assert( rrTask );
        
        QList<SharedAnnotationData> anns = rrTask->getResultedAnnotations();

        if(!anns.isEmpty()) {		
           // Document* d = AppContext::getProject()->findDocumentByURL(url);
            //assert(d==NULL);
            if(!url.isEmpty()) {
                IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
                DocumentFormat* df = AppContext::getDocumentFormatRegistry()->getFormatById(BaseDocumentFormats::PLAIN_GENBANK);
                Document *d = df->createNewDocument(iof, url);
                d->addObject(aobj);
                AppContext::getProject()->addDocument(d);
            }
            
            for(QMutableListIterator<SharedAnnotationData> it_ad(anns); it_ad.hasNext(); ) {
                AnnotationData * ad = it_ad.next().data();
                QList<LRegion> & regions = ad->location;
                for( QMutableListIterator<LRegion> it(regions); it.hasNext(); ) {
                    it.next().startPos += offsInGlobalSeq;
                }
            }

            res.append(new CreateAnnotationsTask(aobj, group, anns));
        }
    }
    return res;
}


RemoteBLASTTask::RemoteBLASTTask( const RemoteBLASTTaskSettings & cfg_) :
Task( tr("RemoteBLASTTask"), TaskFlag_None ), timeout(true), cfg(cfg_)
{
}

void RemoteBLASTTask::prepareQueries() {
    Query q;
    if(cfg.aminoT) {
        q.amino = true;
        QByteArray complQuery( cfg.query.size(), 0 );
        cfg.complT->translate( cfg.query.data(), cfg.query.size(), complQuery.data(), complQuery.size() );
        TextUtils::reverse( complQuery.data(), complQuery.size() );
        for( int i = 0; i < 3; ++i ) {
            QByteArray aminoQuery( cfg.query.size() / 3, 0 );
            cfg.aminoT->translate( cfg.query.data()+i, cfg.query.size()-i, aminoQuery.data(), aminoQuery.size() );
            q.seq = aminoQuery;
            q.offs = i;
            q.complement = false;
            queries.push_back(q);
            QByteArray aminoQueryCompl( cfg.query.size() / 3, 0 );
            cfg.aminoT->translate(complQuery.data()+i, complQuery.size()-i, aminoQueryCompl.data(), aminoQueryCompl.size());
            q.seq = aminoQueryCompl;
            q.offs = i;
            q.complement = true;
            queries.push_back(q);
        }
    }
    else {
        q.seq = cfg.query;
        queries.push_back(q);
    }
}

void RemoteBLASTTask::prepare() {
    prepareQueries();
    algoLog.trace("Sequences prepared");
    for (QList<Query>::iterator it = queries.begin(),end = queries.end();it!=end;it++) {
        DataBaseFactory *dbf = AppContext::getDataBaseRegistry()->getFactoryById(cfg.dbChoosen);
        if(dbf == NULL) {
            stateInfo.setError(tr("Incorrect database"));
            return;
        }
        HttpRequest *tmp = dbf->getRequest(this);
        httpRequest.append(tmp);
    }
    algoLog.trace("Requests formed");
    connect(&timer,SIGNAL(timeout()),SLOT(sl_timeout()));
    timeout = true;
    timer.setSingleShot(true);
    int mult = cfg.aminoT ? 6 : 1; //if 6 requests - 6 times more wait
    timer.start(cfg.retries*1000*60*mult);
}


void RemoteBLASTTask::run() {	
    for( int i = 0;i < queries.count();i++ ) {
        bool error = true;
        if( isCanceled() ) {
            return;
        }
        
        httpRequest[i]->sendRequest(cfg.params,QString(queries[i].seq.data()));
        error = httpRequest[i]->connectionError;
        if(error) {
            stateInfo.setError(httpRequest[i]->getError());
            return;
        }
    
        createAnnotations(queries[i],httpRequest[i]);
    }
}

QList<SharedAnnotationData> RemoteBLASTTask::getResultedAnnotations() const {
    return resultAnnotations;
}

void RemoteBLASTTask::createAnnotations(const Query &q, HttpRequest *t) {
    QList<SharedAnnotationData> annotations = t->getAnnotations();
    if(annotations.isEmpty()) return;

    if(cfg.filterResult) {
        annotations = filterAnnotations(annotations);
    }

    for( QList<SharedAnnotationData>::iterator it = annotations.begin(), end = annotations.end(); end != it; ++it ) {
        for( QList<LRegion>::iterator jt = it->data()->location.begin(), eend = it->data()->location.end(); eend != jt; ++jt ) {
                int & s = jt->startPos;
                int & l = jt->len;

                if( q.complement ) {
                    s = q.seq.size() - s - l;
                    it->data()->complement = !it->data()->complement;
                }
                if( q.amino ) {
                    s = s * 3 + (q.complement ? 2 - q.offs : q.offs);
                    l = l * 3;
                }
        }
    }

    resultAnnotations << annotations;
}

QList<SharedAnnotationData> RemoteBLASTTask::filterAnnotations(QList<SharedAnnotationData> annotations) {
    QString selectiveQual = cfg.useEval ? "e-value" : "score";
    QList<SharedAnnotationData> resultList;

    if(cfg.filterResult & FilterResultByAccession) {
        QStringList accessions;
        foreach(const SharedAnnotationData &ann, annotations) {
            QString acc = ann->findFirstQualifierValue("accession");
            if(accessions.contains(acc)) {
                QString eval = ann->findFirstQualifierValue(selectiveQual);
                foreach(const SharedAnnotationData &a, resultList) {
                    if(a->findFirstQualifierValue("accession") == acc) {
                        if(cfg.useEval ? a->findFirstQualifierValue(selectiveQual).toDouble() < eval.toDouble() :
                        a->findFirstQualifierValue(selectiveQual).toDouble() > eval.toDouble()) {
                            resultList.removeOne(a);
                            resultList << ann;
                        }
                        break;
                    }
                } 
            } else {
                resultList << ann;
                accessions << acc;
            }
        }
        annotations = resultList;
    }
    
    if(cfg.filterResult & FilterResultByDef) {
        resultList.clear();
        QStringList defs;
        foreach(const SharedAnnotationData &ann, annotations) {
            QString def = ann->findFirstQualifierValue("def");
            if(defs.contains(def)) {
                QString eval = ann->findFirstQualifierValue(selectiveQual);
                foreach(const SharedAnnotationData &a, resultList) {
                    if(a->findFirstQualifierValue("def") == def) {
                        if(cfg.useEval ? a->findFirstQualifierValue(selectiveQual).toDouble() < eval.toDouble() :
                        a->findFirstQualifierValue(selectiveQual).toDouble() > eval.toDouble()) {
                            resultList.removeOne(a);
                            resultList << ann;
                        }
                        break;
                    }
                } 
            } else {
                resultList << ann;
                defs << def;
            }
        }
        annotations = resultList;
    }
    
    if(cfg.filterResult & FilterResultById) {
        resultList.clear();
        QStringList ids;
        foreach(const SharedAnnotationData &ann, annotations) {
            QString id = ann->findFirstQualifierValue("id");
            if(ids.contains(id)) {
                QString eval = ann->findFirstQualifierValue(selectiveQual);
                foreach(const SharedAnnotationData &a, resultList) {
                    if(a->findFirstQualifierValue("id") == id) {
                        if(cfg.useEval ? a->findFirstQualifierValue(selectiveQual).toDouble() < eval.toDouble() :
                        a->findFirstQualifierValue(selectiveQual).toDouble() > eval.toDouble()) {
                            resultList.removeOne(a);
                            resultList << ann;
                        }
                        break;
                    }
                } 
            } else {
                resultList << ann;
                ids << id;
            }
        }
    }

    
    return resultList;
}


}
