/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WMQuery.h"
#include "WeightMatrixIO.h"

#include <U2Designer/DelegateEditors.h>

#include <U2Lang/CoreDataTypes.h>

#include <U2Core/L10n.h>
#include <U2Core/Log.h>
#include <U2Core/AppContext.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Core/MultiTask.h>
#include <U2Core/TaskSignalMapper.h>
#include <U2Core/FailTask.h>

#include <U2Misc/DialogUtils.h>

#include <QtGui/QApplication>


namespace U2 {

/************************************************************************/
/* WMQDTask                                                             */
/************************************************************************/

WMQDTask::WMQDTask(const QString& url, const WeightMatrixSearchCfg& cfg,
                   DNASequenceObject* sqnc, const QString& resName, const QList<LRegion>& location)
: Task(tr("Weight matrix query"), TaskFlag_NoRun), settings(cfg), seqObj(sqnc),
resultName(resName), location(location) {
    readTask = new PWMatrixReadTask(url);
    addSubTask(readTask);
}

QList<Task*> WMQDTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> subtasks;
    if (subTask==readTask) {
        PWMatrix model = readTask->getResult();
        foreach(const LRegion& r, location) {
            subtasks << new WeightMatrixSingleSearchTask(model,
                seqObj->getSequence().constData() + r.startPos,
                r.len,
                settings,
                r.startPos);
        }
    }
    else {
        WeightMatrixSingleSearchTask* t = qobject_cast<WeightMatrixSingleSearchTask*>(subTask);
        assert(t);
        res << t->takeResults();
    }
    return subtasks;
}

QList<WeightMatrixSearchResult> WMQDTask::takeResults() {
    QList<WeightMatrixSearchResult> results = res;
    res.clear();
    return results;
}

/************************************************************************/
/* QDWMActor                                                            */
/************************************************************************/

static const QString STRAND_ATTR = "strand";
static const QString SCORE_ATTR = "min-score";
static const QString PROFILE_URL_ATTR = "matrix";

QDWMActor::QDWMActor(QDActorPrototype const* proto) : QDActor(proto) {
    units["wm"] = new QDSchemeUnit(this);
}

int QDWMActor::getMinResultLen() const {
    return 1;
}

int QDWMActor::getMaxResultLen() const {
    if (scheme->getDNA()) {
        return scheme->getDNA()->getSequenceLen();
    }
    return 30;//FIX ME: supply reasonable value
}

QString QDWMActor::getText() const {
    QMap<QString, Attribute*> params = cfg->getParameters();

    int strand = params.value(STRAND_ATTR)->getAttributePureValue().toInt();
    QString strandName;
    switch (strand) {
    case 0: strandName = tr("both strands"); break;
    case 1: strandName = tr("direct strand"); break;
    case -1: strandName = tr("complement strand"); break;
    }

    strandName = QString("<a href=%1>%2</a>").arg(STRAND_ATTR).arg(strandName);

    QString profileUrl = params.value(PROFILE_URL_ATTR)->getAttributeValue<QString>();
    if (profileUrl.isEmpty()) {
        profileUrl = "unset";
    }
    profileUrl = QString("<a href=%1>%2</a>").arg(PROFILE_URL_ATTR).arg(profileUrl);
    int score = params.value(SCORE_ATTR)->getAttributeValue<int>();
    QString scoreStr = QString("<a href=%1>%2%</a>").arg(SCORE_ATTR).arg(score);
    
    QString doc = tr("Searches TFBS with all profiles from <u>%1</u> "
        "<br> Recognizes sites with <u>similarity %2</u>, processes <u>%3</u>.")
        .arg(profileUrl)
        .arg(scoreStr)
        .arg(strandName);

    return doc;
}

Task* QDWMActor::getAlgorithmTask(const QList<LRegion>& location) {
    Task* t = NULL;
    DNASequenceObject* dna = scheme->getDNA();
    assert(dna);
    QMap<QString, Attribute*> params = cfg->getParameters();

    WeightMatrixSearchCfg config;
    int strand = params.value(STRAND_ATTR)->getAttributeValue<int>();
    config.minPSUM = params.value(SCORE_ATTR)->getAttributeValue<int>();
    const QString& modelUrl = params.value(PROFILE_URL_ATTR)->getAttributeValue<QString>();

    if (dna->getAlphabet()->getType() == DNAAlphabet_NUCL) {        
        config.complOnly = (strand < 0);
        if (strand <= 0) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(dna->getAlphabet(), DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complTT = compTTs.first();
            }
        }
        t = new WMQDTask(modelUrl, config, dna, "", location);
        connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_onAlgorithmTaskFinished(Task*)));
    } else {
        QString err = tr("%1: sequence should be nucleic.").arg(getParameters()->getLabel());
        return new FailTask(err);
    }
    return t;
}

void QDWMActor::sl_onAlgorithmTaskFinished(Task* t) {
    WMQDTask* wmqdt = qobject_cast<WMQDTask*>(t);
    assert(wmqdt);
    QList<WeightMatrixSearchResult> res = wmqdt->takeResults();
    foreach(const WeightMatrixSearchResult& r, res) {
        QDResultUnit ru(new QDResultUnitData);
        ru->complement = r.complement;
        QMapIterator<QString,QString> it(r.qual);
        while (it.hasNext()) {
            it.next();
            ru->quals.append(Qualifier(it.key(), it.value()));
        }
        ru->region = r.region;
        ru->owner = units.value("wm");
        QDResultGroup* g = new QDResultGroup;
        g->add(ru);
        results.append(g);
    }
}

static const QString STRAND_BOTH = "both";
static const QString STRAND_COMPL = "complement";
static const QString STRAND_DIRECT = "direct";

QList< QPair<QString,QString> > QDWMActor::saveConfiguration() const {
    QList< QPair<QString,QString> > res = QDActor::saveConfiguration();
    Attribute* a = cfg->getParameter(STRAND_ATTR);
    for (int i=0; i<res.size(); i++) {
        StringAttribute& attr = res[i];
        if (attr.first==a->getId()) {
            int strand = a->getAttributeValue<int>();
            if (strand==-1) {
                attr.second = STRAND_COMPL;
            }
            else if (strand==0) {
                attr.second = STRAND_BOTH;
            }
            else {
                assert(strand==1);
                attr.second = STRAND_DIRECT;
            }
            break;
        }
    }
    return res;
}

void QDWMActor::loadConfiguration( const QList< QPair<QString,QString> >& strMap ) {
    QDActor::loadConfiguration(strMap);
    foreach(const StringAttribute& attr, strMap) {
        if (attr.first==STRAND_ATTR) {
            int strand;
            const QString& strandVal = attr.second;
            if (strandVal==STRAND_BOTH) {
                strand = 0;
            }
            else if (strandVal==STRAND_COMPL) {
                strand = -1;
            }
            else if (strandVal==STRAND_DIRECT) {
                strand = 1;
            }
            cfg->setParameter(STRAND_ATTR, qVariantFromValue(strand));
        }
    }
}
/************************************************************************/
/* Factory                                                              */
/************************************************************************/

QDWMActorPrototype::QDWMActorPrototype() {
    descriptor.setId("wsearch");
    descriptor.setDisplayName(QObject::tr("Weight matrix"));
    descriptor.setDocumentation(QObject::tr("Searches the sequence for transcription factor binding sites significantly similar to the specified weight matrix."));

    Descriptor snd(STRAND_ATTR, L10N::searchIn(), QDWMActor::tr("Which strands should be searched: direct, complement or both."));
    Descriptor scd(SCORE_ATTR, QObject::tr("Min score"), QApplication::translate("PWMSearchDialog", "min_err_tip", 0, QApplication::UnicodeUTF8));
    Descriptor mx(PROFILE_URL_ATTR, QObject::tr("Matrix"), QObject::tr("Path to profile"));

    attributes << new Attribute(snd, CoreDataTypes::NUM_TYPE(), false, 0);
    attributes << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 85);
    attributes << new Attribute(mx, CoreDataTypes::STRING_TYPE(), true);

    QMap<QString, PropertyDelegate*> delegates;
    {
        QVariantMap m; m["minimum"] = 1; m["maximum"] = 100; m["suffix"] = "%";
        delegates[SCORE_ATTR] = new SpinBoxDelegate(m);
    }

    QVariantMap strandMap; 
    strandMap[QObject::tr("both strands")] = 0;
    strandMap[QObject::tr("direct strand")] = 1;
    strandMap[QObject::tr("complement strand")] = -1;
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);

    delegates[PROFILE_URL_ATTR] = new URLDelegate(WeightMatrixIO::getPWMFileFilter(), WeightMatrixIO::WEIGHT_MATRIX_ID, true);

    editor = new DelegateEditor(delegates);
}

}//namespace
