/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WeightMatrixWorkers.h"
#include "WeightMatrixIOWorkers.h"
#include "WeightMatrixSearchTask.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/DNASequence.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/AppContext.h>
#include <U2Core/L10n.h>
#include <U2Core/Log.h>
#include <U2Core/MultiTask.h>
#include <U2Core/FailTask.h>
#include <U2Core/TaskSignalMapper.h>

#include <QtGui/QApplication>
/* TRANSLATOR U2::LocalWorkflow::PWMatrixSearchWorker */

namespace U2 {
namespace LocalWorkflow {

static const QString MODEL_PORT("in-wmatrix");

static const QString NAME_ATTR("result-name");
static const QString STRAND_ATTR("strand");
static const QString SCORE_ATTR("min-score");

const QString PWMatrixSearchWorker::ACTOR_ID("wmatrix-search");

void PWMatrixSearchWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor md(MODEL_PORT, PWMatrixSearchWorker::tr("Weight matrix"), PWMatrixSearchWorker::tr("Profile data to search with."));
        Descriptor sd(CoreLibConstants::IN_SEQ_PORT_ID, PWMatrixSearchWorker::tr("Sequence"), PWMatrixSearchWorker::tr("Input nucleotide sequence to search in."));
        Descriptor od(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID, PWMatrixSearchWorker::tr("Weight matrix annotations"), 
            PWMatrixSearchWorker::tr("Annotations marking found TFBS sites."));
        
        QMap<Descriptor, DataTypePtr> modelM;
        modelM[PWMatrixWorkerFactory::WMATRIX_SLOT] = PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE();
        p << new PortDescriptor(md, DataTypePtr(new MapDataType("wmatrix.search.model", modelM)), true /*input*/, false, IntegralBusPort::BLIND_INPUT);
        QMap<Descriptor, DataTypePtr> seqM;
        seqM[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        p << new PortDescriptor(sd, DataTypePtr(new MapDataType("wmatrix.search.sequence", seqM)), true /*input*/);
        QMap<Descriptor, DataTypePtr> outM;
        outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
        p << new PortDescriptor(od, DataTypePtr(new MapDataType("wmatrix.search.out", outM)), false /*input*/, true /*multi*/);
    }
    {
        Descriptor nd(NAME_ATTR, PWMatrixSearchWorker::tr("Result annotation"), 
            PWMatrixSearchWorker::tr("Annotation name for marking found regions"));
        Descriptor snd(STRAND_ATTR, L10N::searchIn(),
            PWMatrixSearchWorker::tr("Which strands should be searched: direct, complement or both."));
        Descriptor scd(SCORE_ATTR, PWMatrixSearchWorker::tr("Min score"), 
            QApplication::translate("PWMSearchDialog", "min_err_tip", 0, QApplication::UnicodeUTF8));
        
        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "misc_feature");
        a << new Attribute(snd, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 85);
    }

    Descriptor desc(ACTOR_ID, tr("Search for TFBS with weight matrix"), 
        tr("Searches each input sequence for transcription factor binding sites significantly similar to specified weight matrices."
        " In case several profiles were supplied, searches with all profiles one by one and outputs merged set of annotations for each sequence.")
        );
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;

    {
        QVariantMap m; m["minimum"] = 1; m["maximum"] = 100; m["suffix"] = "%";
        delegates[SCORE_ATTR] = new SpinBoxDelegate(m);
    }

    QVariantMap strandMap; 
    strandMap[tr("both strands")] = 0;
    strandMap[tr("direct strand")] = 1;
    strandMap[tr("complement strand")] = -1;
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);
    
    proto->setPrompter(new PWMatrixSearchPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":weight_matrix/images/weight_matrix.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_TRANSCRIPTION(), proto);
}

QString PWMatrixSearchPrompter::composeRichDoc() {
    Actor* modelProducer = qobject_cast<IntegralBusPort*>(target->getPort(MODEL_PORT))->getProducer(MODEL_PORT);
    Actor* seqProducer = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID))->getProducer(CoreLibConstants::IN_SEQ_PORT_ID);

    QString seqName = seqProducer ? tr("For each sequence from <u>%1</u>,").arg(seqProducer->getLabel()) : "";
    QString modelName = modelProducer ? tr("with all profiles provided by <u>%1</u>,").arg(modelProducer->getLabel()) : "";

    QString resultName = getRequiredParam(NAME_ATTR);

    QString strandName;
    switch (getParameter(STRAND_ATTR).toInt()) {
    case 0: strandName = PWMatrixSearchWorker::tr("both strands"); break;
    case 1: strandName = PWMatrixSearchWorker::tr("direct strand"); break;
    case -1: strandName = PWMatrixSearchWorker::tr("complement strand"); break;
    }

    QString doc = tr("%1 search transcription factor binding sites (TFBS) %2."
        "<br>Recognize sites with <u>similarity %3%</u>, process <u>%4</u>."
        "<br>Output the list of found regions annotated as <u>%5</u>.")
        .arg(seqName)
        .arg(modelName)
        .arg(getParameter(SCORE_ATTR).toInt())
        .arg(strandName)
        .arg(resultName);

    return doc;
}

void PWMatrixSearchWorker::init() {
    modelPort = ports.value(MODEL_PORT);
    dataPort = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
    dataPort->addComplement(output);
    output->addComplement(dataPort);

    strand = actor->getParameter(STRAND_ATTR)->getAttributeValue<int>();
    cfg.minPSUM = actor->getParameter(SCORE_ATTR)->getAttributeValue<int>();
    resultName = actor->getParameter(NAME_ATTR)->getAttributeValue<QString>();
}

bool PWMatrixSearchWorker::isReady() {
    return ((!models.isEmpty() && modelPort->isEnded()) && dataPort->hasMessage()) || modelPort->hasMessage();
}

Task* PWMatrixSearchWorker::tick() {
    while (modelPort->hasMessage()) {
        models << modelPort->get().getData().toMap().value(PWMatrixWorkerFactory::WMATRIX_SLOT.getId()).value<PWMatrix>();
    }
    if (models.isEmpty() || !modelPort->isEnded() || !dataPort->hasMessage()) {
        return NULL;
    }
    DNASequence seq = dataPort->get().getData().toMap().value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();
    
    if (!seq.isNull() && seq.alphabet->getType() == DNAAlphabet_NUCL) {
        WeightMatrixSearchCfg config(cfg);
        config.complOnly = (strand < 0);
        if (strand <= 0) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complTT = compTTs.first();
            }
        }
        QList<Task*> subtasks;
        foreach(PWMatrix model, models) {
            subtasks << new WeightMatrixSingleSearchTask(model, seq.constData(), seq.length(), config, 0);
        }
        Task* t = new MultiTask(tr("Search TFBS in %1").arg(seq.getName()), subtasks);
        connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
        return t;
    }
    QString err = tr("Bad sequence supplied to Weight Matrix Search: %1").arg(seq.getName());
    //if (failFast) {
        return new FailTask(err);
    /*} else {
        algoLog.error(err);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), QVariant()));
        if (dataPort->isEnded()) {
            output->setEnded();
        }
        return NULL;
    }*/
}

void PWMatrixSearchWorker::sl_taskFinished(Task* t) {
    QList<SharedAnnotationData> res;
    foreach(Task* sub, t->getSubtasks()) {
        WeightMatrixSingleSearchTask* sst = qobject_cast<WeightMatrixSingleSearchTask*>(sub);
        res += WeightMatrixSearchResult::toTable(sst->takeResults(), resultName);
    }
    QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
    output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
    if (dataPort->isEnded()) {
        output->setEnded();
    }
    algoLog.info(tr("Found %1 TFBS").arg(res.size())); //TODO set task description for report
}

bool PWMatrixSearchWorker::isDone() {
    return dataPort->isEnded();
}

} //namespace LocalWorkflow
} //namespace U2
