/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QDebug>

#include <U2Core/Log.h>
#include <U2Lang/WorkflowUtils.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/WorkflowSettings.h>
#include <U2Lang/IntegralBusType.h>
#include <U2Lang/Datatype.h>

#include "WorkflowEditorDelegates.h"
#include "WorkflowEditor.h"
#include "ActorCfgModel.h"

namespace U2 {

/*****************************
 * ActorCfgModel
 *****************************/
static const int KEY_COLUMN = 0;
static const int VALUE_COLUMN = 1;
static const int SCRIPT_COLUMN = 2;

ActorCfgModel::ActorCfgModel(QObject *parent, QList<Iteration>& lst)
: QAbstractTableModel(parent), subject(NULL), iterations(lst), iterationIdx(-1), scriptMode(false) {
    scriptDelegate = new AttributeScriptDelegate();
}

ActorCfgModel::~ActorCfgModel() {
    delete scriptDelegate;
}

void ActorCfgModel::setActor(Actor* cfg) {
    listValues.clear();
    attrs.clear();
    //inputPortsData.reset();
    subject = cfg;
    if (cfg) {
        //attrs = cfg->getParameters().values();
        attrs = cfg->getAttributes();
        //inputPortsData.setData(cfg->getInputPorts());
        setupAttributesScripts();
    }
    reset();
}

void dumpDescriptors(const QList<Descriptor> & descriptors) {
    foreach( const Descriptor & d, descriptors ) {
        qDebug() << d.getId() << d.getDisplayName();
    }
}

void ActorCfgModel::setupAttributesScripts() {
    foreach( Attribute * attribute, attrs ) {
        assert(attribute != NULL);
        attribute->getAttributeScript().clearScriptVars();
        
        DataTypePtr attributeType = attribute->getAttributeType();
        // FIXME: add support for all types in scripting
        if(attributeType != CoreDataTypes::STRING_TYPE() && attributeType != CoreDataTypes::NUM_TYPE()) {
            continue;
        }
        

        foreach(const PortDescriptor *descr, subject->getProto()->getPortDesciptors()) {
            if(descr->isInput()) {
                DataTypePtr dataTypePtr = descr->getType();
                if(dataTypePtr->isMap()) {
                    QMap<Descriptor, DataTypePtr> map = dataTypePtr->getDatatypesMap();
                    foreach(const Descriptor &desc,  map.keys()) {
                        //QString id = ptr->getId();
                        //QString displayName = ptr->getDisplayName();
                        //QString doc = ptr->getDocumentation();
                        //attribute->getAttributeScript().setScriptVar(Descriptor(id,displayName,doc), QVariant());
                        QString id = desc.getId().replace(QRegExp("[^a-zA-Z0-9_]"), "_");
                        if(id.at(0).isDigit()) {
                            id.prepend("_");
                        }
                        Descriptor d(id, desc.getDisplayName(), desc.getDocumentation());
                        attribute->getAttributeScript().setScriptVar(d, QVariant());
                    }
                }
                else  if(dataTypePtr->isList()) {
                    foreach(const Descriptor & typeDescr, dataTypePtr->getAllDescriptors()) {
                        QString id = typeDescr.getId().replace(QRegExp("[^a-zA-Z0-9_]"), "_");
                        if(id.at(0).isDigit()) {
                            id.prepend("_");
                        }                        
                        Descriptor d(id, typeDescr.getDisplayName(), typeDescr.getDocumentation());
                        attribute->getAttributeScript().setScriptVar(d, QVariant());
                    }
                }
                else {
                    QString id = dataTypePtr->getId().replace(QRegExp("[^a-zA-Z0-9_]"), "_");
                    if(id.at(0).isDigit()) {
                        id.prepend("_");
                    }
                    QString displayName = dataTypePtr->getDisplayName();
                    QString doc = dataTypePtr->getDocumentation();
                    attribute->getAttributeScript().setScriptVar(Descriptor(id,displayName,doc), QVariant());
                }
            }
        }

        QString attrVarName = attribute->getDisplayName();
        QString id = attribute->getId().replace(QRegExp("[^a-zA-Z0-9_]"), "_");
        if(id.at(0).isDigit()) {
            id.prepend("_");
        }
        attribute->getAttributeScript().
            setScriptVar(Descriptor(id, attrVarName, attribute->getDocumentation()), QVariant());
    }
}

void ActorCfgModel::selectIteration(int i) {
    listValues.clear();
    iterationIdx = i;
    reset();
}

void ActorCfgModel::setIterations(QList<Iteration>& lst) {
    iterations = lst;
    reset();
}

int ActorCfgModel::columnCount(const QModelIndex &) const { 
    if(scriptMode) {
        return 3; // key, value and script
    } else {
        return 2;
    }
}

int ActorCfgModel::rowCount( const QModelIndex & parent ) const {
    if(parent.isValid()) {
        return 0;
    }

    return attrs.isEmpty() || parent.isValid() ? 0 : attrs.size();
}

Qt::ItemFlags ActorCfgModel::flags( const QModelIndex & index ) const {
    int col = index.column();
    int row = index.row();
    switch(col) {
    case KEY_COLUMN:
        return Qt::ItemIsEnabled;
    case VALUE_COLUMN:
        return row < attrs.size() ? Qt::ItemIsEditable | Qt::ItemIsEnabled | Qt::ItemIsSelectable : Qt::ItemIsEnabled;
    case SCRIPT_COLUMN:
        {
            if(row < attrs.size()) {
                Attribute * currentAttribute = attrs.at(row);
                assert(currentAttribute != NULL);
                // FIXME: add support for all types in scripting
                if(currentAttribute->getAttributeType() != CoreDataTypes::STRING_TYPE() && currentAttribute->getAttributeType() != CoreDataTypes::NUM_TYPE()) {
                    return Qt::ItemIsEnabled;    
                } else {
                    return Qt::ItemIsEditable | Qt::ItemIsEnabled | Qt::ItemIsSelectable;
                }
            } else {
                return Qt::ItemIsEnabled;
            }
        }
    default:
        assert(false);
    }
    // unreachable code
    return Qt::NoItemFlags;
}

QVariant ActorCfgModel::headerData( int section, Qt::Orientation orientation, int role ) const {
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
        switch(section) {
        case KEY_COLUMN: 
            return WorkflowEditor::tr("Name");
        case VALUE_COLUMN: 
            return WorkflowEditor::tr("Value");
        case SCRIPT_COLUMN:
            return WorkflowEditor::tr("Script");
        default:
            assert(false);
        }
    }
    // unreachable code
    return QVariant();
}

bool ActorCfgModel::setAttributeValue( const Attribute * attr, QVariant & attrValue ) const{
    assert(attr != NULL);
    bool isDefaultVal = true;

    attrValue = attr->getAttributePureValue();

    if (iterations.isEmpty()) { //rare bug -> need to catch in debug
        assert(0);
        return isDefaultVal;

    }

    if (iterationIdx >= 0) {
        int x = iterationIdx;
        if (x >= iterations.size()) {
            //FIXME: handle error
            x = 0;
        }
        const Iteration& it = iterations.at(x);
        if (it.cfg.contains(subject->getId())) {
            const QVariantMap& params = it.cfg[subject->getId()];
            if (params.contains(attr->getId())) {
                attrValue = params.value(attr->getId());
                isDefaultVal = false;
            }
        }
    }
    return isDefaultVal;
}

static QString getStringForDisplayRole(const QVariant & value) {
    QString str = value.toString();
    if(!str.isEmpty() && str.at(0).isLower()) {
        return str.at(0).toUpper() + str.mid(1);
    }
    return str;
}

QVariant ActorCfgModel::data(const QModelIndex & index, int role ) const {
    const Attribute *currentAttribute = attrs.at(index.row());
    if (role == DescriptorRole) { // descriptor that will be shown in under editor. 'propDoc' in WorkflowEditor
        return qVariantFromValue<Descriptor>(*currentAttribute);
    }
    
    int col = index.column();
    switch(col) {
    case KEY_COLUMN:
        {
            switch (role) {
            case Qt::DisplayRole: 
                return currentAttribute->getDisplayName();
            case Qt::ToolTipRole: 
                return currentAttribute->getDocumentation();
            case Qt::FontRole:
                if (currentAttribute->isRequiredAttribute()) {
                    QFont fnt; 
                    fnt.setBold(true);
                    return QVariant(fnt);
                }
            default:
                return QVariant();
            }
        }
    case VALUE_COLUMN:
        {
            if (role == ConfigurationEditor::ItemListValueRole) {
                return listValues.value(currentAttribute->getId());
            }
            
            QVariant attributeValue;
            bool isDefaultVal = setAttributeValue(currentAttribute, attributeValue);
            ConfigurationEditor* confEditor = subject->getEditor();
            PropertyDelegate* propertyDelegate = confEditor ? confEditor->getDelegate(currentAttribute->getId()) : NULL;
            switch(role) {
            case Qt::DisplayRole: 
            case Qt::ToolTipRole:
                {
                    if(propertyDelegate) {
                        return propertyDelegate->getDisplayValue(attributeValue);
                    } else {
                        QString valueStr = getStringForDisplayRole(attributeValue);
                        if(!valueStr.isEmpty()) {
                            return valueStr;
                        } else {
                            return attributeValue;
                        }
                    }
                }
            case Qt::ForegroundRole:
                return isDefaultVal ? QVariant(QColor(Qt::gray)) : QVariant();
            case DelegateRole:
                return qVariantFromValue<PropertyDelegate*>(propertyDelegate);
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                return attributeValue;
            default:
                return QVariant();
            }
        }
    case SCRIPT_COLUMN:
        {
            // FIXME: add support for all types in scripting
            if(currentAttribute->getAttributeType() != CoreDataTypes::STRING_TYPE() && currentAttribute->getAttributeType() != CoreDataTypes::NUM_TYPE() ) {
                if( role == Qt::DisplayRole || role == Qt::ToolTipRole ) {
                    return QVariant(tr("N/A"));
                } else {
                    return QVariant();
                }
            }
            
            // for STRING type
            switch(role) {
            case Qt::DisplayRole:
            case Qt::ToolTipRole:
                return scriptDelegate ? 
                    scriptDelegate->getDisplayValue(qVariantFromValue<AttributeScript>(currentAttribute->getAttributeScript())) 
                    : QVariant();
            case Qt::ForegroundRole:
                return currentAttribute->getAttributeScript().isEmpty() ? QVariant(QColor(Qt::gray)) : QVariant();
            case DelegateRole:
                assert(scriptDelegate != NULL);
                return qVariantFromValue<PropertyDelegate*>(scriptDelegate);
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                return qVariantFromValue<AttributeScript>(currentAttribute->getAttributeScript());
            default:
                return QVariant();
            }
        }
    default:
        assert(false);
    }
    // unreachable code
    return QVariant();
}

bool ActorCfgModel::setData( const QModelIndex & index, const QVariant & value, int role ) {
    int col = index.column();
    Attribute* editingAttribute = attrs[index.row()];
    assert(editingAttribute != NULL);
    
    switch(col) {
    case VALUE_COLUMN:
        {
            switch(role) {
            case ConfigurationEditor::ItemListValueRole: 
                {
                    listValues.insert(editingAttribute->getId(), value); 
                    return true;
                }
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                {
                    const QString& key = editingAttribute->getId();
                    if (iterationIdx >= 0) {
                        int x = iterationIdx;
                        if (x >= iterations.size()) {
                            //FIXME: handle error
                            x = 0;
                        }
                        QVariantMap& cfg = iterations[x].cfg[subject->getId()];
                        QVariant old = cfg.contains(key) ? cfg.value(key) : editingAttribute->getAttributePureValue();
                        if (old != value) {
                            cfg.insert(key, value);
                            emit dataChanged(index, index);
                            uiLog.trace("committed property change");
                        }
                    } else {
                        if (editingAttribute->getAttributePureValue() != value) {
                            subject->setParameter(key, value);
                            emit dataChanged(index, index);
                            uiLog.trace("committed property change");
                        }
                    }
                    return true;
                }
            default:
                return false;
            }
        }
    case SCRIPT_COLUMN:
        {
            switch(role) {
            case Qt::EditRole:
            case ConfigurationEditor::ItemValueRole:
                {
                    AttributeScript attrScript = value.value<AttributeScript>();
                    editingAttribute->getAttributeScript().setScriptText(attrScript.getScriptText());
                    emit dataChanged(index, index);
                    uiLog.trace(QString("user script for '%1' attribute updated").arg(editingAttribute->getDisplayName()));
                    return true;
                }
            default:
                return false;
            }
        }
    default:
        assert(false);
    }
    
    // unreachable code
    return false;
}

void ActorCfgModel::changeScriptMode(bool _mode) {
    scriptMode = _mode;
}

} // U2
