/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QFile>

#include <U2Core/AppContext.h>
#include <U2Core/Log.h>
#include <U2Core/Settings.h>
#include <U2Core/Counter.h>
#include <U2Core/CMDLineRegistry.h>
#include <U2Core/CMDLineUtils.h>
#include <U2Core/CMDLineCoreOptions.h>
#include <U2Remote/SerializeUtils.h>

#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/WorkflowUtils.h>

#include "WorkflowDesignerPlugin.h"
#include "WorkflowCMDLineTasks.h"

#define WORKFLOW_CMDLINE_TASK_LOG_CAT "Workflow cmdline tasks"

namespace U2 {

static Logger log( WORKFLOW_CMDLINE_TASK_LOG_CAT );

/*******************************************
* WorkflowRunFromCMDLineBase
*******************************************/
WorkflowRunFromCMDLineBase::WorkflowRunFromCMDLineBase() 
: Task( tr( "Workflow run from cmdline" ), TaskFlag_NoRun ), schema(NULL), optionsStartAt(-1), loadTask(NULL) {
    GCOUNTER(cvar,tvar,"workflow_run_from_cmdline");
    
    CMDLineRegistry * cmdLineRegistry = AppContext::getCMDLineRegistry();

    // try to process schema without 'task' option (it can only be the first one)
    QStringList pureValues = CMDLineRegistryUtils::getPureValues();
    if( !pureValues.isEmpty() ) {
        QString schemaName = pureValues.first();
        processLoadSchemaTask( schemaName, 1 ); // because after program name
    }
    if( loadTask != NULL ) {
        addSubTask( loadTask );
        return;
    }

    // process schema with 'task' option
    int taskOptionIdx = CMDLineRegistryUtils::getParameterIndex( WorkflowDesignerPlugin::RUN_WORKFLOW );
    if(taskOptionIdx != -1) {
        processLoadSchemaTask( cmdLineRegistry->getParameterValue( WorkflowDesignerPlugin::RUN_WORKFLOW, taskOptionIdx ), taskOptionIdx );
    }
    if( loadTask == NULL ) {
        setError( tr( "no task to run" ) );
        return;
    }
    addSubTask( loadTask );
}

void WorkflowRunFromCMDLineBase::processLoadSchemaTask( const QString & schemaNameCandidate, int optionIdx ) {
    loadTask = prepareLoadSchemaTask( schemaNameCandidate );
    if( loadTask != NULL ) {
        schemaName = schemaNameCandidate;
        optionsStartAt = optionIdx + 1;
    }
}

LoadWorkflowTask * WorkflowRunFromCMDLineBase::prepareLoadSchemaTask( const QString & schemaName ) {
    QString pathToSchema = WorkflowUtils::findPathToSchemaFile( schemaName );
    if( pathToSchema.isEmpty() ) {
        log.error( tr( "Cannot find schema: %1" ).arg( schemaName ) );
        return NULL;
    }

    schema = new Schema();
    schema->setDeepCopyFlag(true);
    return new LoadWorkflowTask( schema, NULL, pathToSchema );
}

WorkflowRunFromCMDLineBase::~WorkflowRunFromCMDLineBase() {
    delete schema;
}

static Actor * findActorByParamAlias( Schema * schema, const QString & alias, QString & attrName ) {
    assert( schema != NULL );
    QList<Actor*> actors;
    foreach( Actor * actor, schema->getProcesses() ) {
        assert( actor != NULL );
        if( actor->getParamAliases().values().contains( alias ) ) {
            actors << actor;
        }
    }

    if ( actors.isEmpty() ) {
        return NULL;
    } else if( actors.size() > 1 ) {
        log.error( WorkflowRunFromCMDLineBase::tr( "%1 actors in schema have '%2' alias" ).arg( actors.size() ).arg( alias ) );
    }

    Actor * ret = actors.first();
    attrName = ret->getParamAliases().key( alias );
    return ret;
}

static void setSchemaCMDLineOptions( Schema * schema, int optionsStartAtIdx ) {
    assert( schema != NULL && optionsStartAtIdx > 0 );

    QList<StringPair> parameters = AppContext::getCMDLineRegistry()->getParameters();
    int sz = parameters.size();
    for( int i = optionsStartAtIdx; i < sz; ++i ) {
        const StringPair & param = parameters.at(i);
        if( param.first.isEmpty() ) { // TODO: unnamed parameters not supported yet
            continue;
        }

        QString paramAlias = param.first;
        QString paramName;
        Actor * actor = findActorByParamAlias( schema, paramAlias, paramName );
        if( actor == NULL ) {
            assert( paramName.isEmpty() );
            log.info( WorkflowRunFromCMDLineBase::tr( "alias '%1' not set in schema" ).arg( paramAlias ) );
            continue;
        }

        Attribute * attr = actor->getParameter( paramName );
        if( attr == NULL ) {
            log.error( WorkflowRunFromCMDLineBase::tr( "actor parameter '%1' not found" ).arg( paramName ) );
            continue;
        }

        DataTypeValueFactory * valueFactory = WorkflowEnv::getDataTypeValueFactoryRegistry()->
            getById( attr->getAttributeType()->getId() );
        if( valueFactory == NULL ) {
            log.error( WorkflowRunFromCMDLineBase::tr( "cannot parse value from '%1'" ).arg( param.second ) );
            continue;
        }

        ActorId id = actor->getId();
        bool isOk;
        QVariant value = valueFactory->getValueFromString( param.second, &isOk );
        if(!isOk){
            log.error( WorkflowRunFromCMDLineBase::tr( "Incorrect value for '%1', null or default value passed to schema" ).
                arg( param.first ));
            continue;
        }
        QList<Iteration> & iterations = schema->getIterations();
        QList<Iteration>::iterator it = iterations.begin();
        while( it != iterations.end() ) { // TODO: make different values for different iterations
            it->cfg[id].insert( paramName, value );
            ++it;
        }
    }
}

QList<Task*> WorkflowRunFromCMDLineBase::onSubTaskFinished( Task* subTask ) {
    assert( subTask != NULL );
    QList<Task*> res;

    propagateSubtaskError();
    if( hasErrors() || isCanceled() ) {
        return res;
    }
    assert( !hasErrors() ); // if error, we won't be here

    if( loadTask == subTask ) {
        Schema * schema = loadTask->getSchema();
        assert( schema != NULL );

        setSchemaCMDLineOptions( schema, optionsStartAt );
        if( schema->getDomain().isEmpty() ) {
            QList<QString> domainsId = WorkflowEnv::getDomainRegistry()->getAllIds();
            assert(!domainsId.isEmpty());
            if(!domainsId.isEmpty()) { schema->setDomain(domainsId.first()); }
        }
        
        QStringList l;
        bool good = WorkflowUtils::validate(*schema, l);
        if(!good) {
            QString schemaHelpStr = QString("\n\nsee 'ugene --help=%1' for details").arg(schemaName);
            setError("\n\n" + l.join("\n\n") + schemaHelpStr);
            return res;
        }
        
        res << getWorkflowRunTask();
    }
    return res;
}

/*******************************************
* WorkflowRunFromCMDLineTask
*******************************************/
Task * WorkflowRunFromCMDLineTask::getWorkflowRunTask() const {
    return new WorkflowRunTask(*schema, schema->getIterations());
}

/*******************************************
* WorkflowRemoteRunFromCMDLineTask
*******************************************/
WorkflowRemoteRunFromCMDLineTask::WorkflowRemoteRunFromCMDLineTask() {
    CMDLineRegistry * cmdlineReg = AppContext::getCMDLineRegistry();
    assert(cmdlineReg != NULL);
    QString filePath = cmdlineReg->getParameterValue(WorkflowDesignerPlugin::REMOTE_MACHINE);
    if( filePath.isEmpty() ) {
        stateInfo.setError(tr("%1 parameter expected, but not set").arg(WorkflowDesignerPlugin::REMOTE_MACHINE));
        return;
    }
    
    if( !SerializeUtils::deserializeRemoteMachineSettingsFromFile(filePath, &settings) ) {
        assert(settings == NULL);
        stateInfo.setError(tr("Cannot read remote machine settings from %2").arg(filePath));
        return;
    }
    assert(settings != NULL);
}

Task * WorkflowRemoteRunFromCMDLineTask::getWorkflowRunTask() const {
    assert(settings != NULL);
    return new RemoteWorkflowRunTask( settings, *schema, schema->getIterations());
}

} // U2
