/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "FindWorker.h"

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/DNASequence.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/AppContext.h>
#include <U2Core/L10n.h>
#include <U2Core/Log.h>
#include <U2Algorithm/FindAlgorithmTask.h>
#include <U2Core/FailTask.h>
#include <U2Core/TaskSignalMapper.h>

#include "CoreLib.h"

/* TRANSLATOR U2::LocalWorkflow::FindWorker */

namespace U2 {
namespace LocalWorkflow {

/***************************
 * FindWorkerFactory
 ***************************/
static const QString STRAND_ATTR("strand");
static const QString NAME_ATTR("result-name");
static const QString PATTERN_ATTR("pattern");
static const QString ERR_ATTR("max-mismatches-num");
static const QString ALGO_ATTR("allow-ins-del");
static const QString AMINO_ATTR("amino");

const QString FindWorkerFactory::ACTOR_ID("search");

void FindWorkerFactory::init() {

    QMap<Descriptor, DataTypePtr> m;
    m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
    //m[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
    DataTypePtr inSet(new MapDataType(Descriptor("regioned.sequence"), m));
    DataTypeRegistry* dr = WorkflowEnv::getDataTypeRegistry();
    assert(dr);
    dr->registerEntry(inSet);

    QList<PortDescriptor*> p; QList<Attribute*> a;
    {
        Descriptor ind(CoreLibConstants::IN_SEQ_PORT_ID, FindWorker::tr("Input data"), FindWorker::tr("An input sequence and set of regions to search in."));
        Descriptor oud(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID, FindWorker::tr("Pattern annotations"), FindWorker::tr("Found regions"));
        p << new PortDescriptor(ind, inSet, true);
        QMap<Descriptor, DataTypePtr> outM;
        outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
        p << new PortDescriptor(oud, DataTypePtr(new MapDataType("find.annotations", outM)), false, true);
    }
    
    //DNATranslation*     complementTT;
    //DNATranslation*     proteinTT;
    //LRegion             searchRegion;

    {
        Descriptor nd(NAME_ATTR, FindWorker::tr("Annotate as"), FindWorker::tr("Name of the result annotations marking found regions."));
        Descriptor pd(PATTERN_ATTR, FindWorker::tr("Pattern"), FindWorker::tr("A subsequence pattern to look for."));
        Descriptor ed(ERR_ATTR, FindWorker::tr("Max mismatches"), 
            FindWorker::tr("The search stringency measured in number of max allowed mismatching symbols to the pattern."));
        Descriptor ald(ALGO_ATTR, FindWorker::tr("Allow insertions/deletions"), 
            FindWorker::tr("Take into account possibility of insertions/deletions when searching. By default substitutions only considered."));
        Descriptor amd(AMINO_ATTR, FindWorker::tr("Search in translation"), FindWorker::tr("Translate a supplied nucleotide sequence to protein then search in the translated sequence."));
        Descriptor sd(STRAND_ATTR, L10N::searchIn(), FindWorker::tr("Which strands should be searched: direct, complement or both."));

        a << new Attribute(nd, CoreDataTypes::STRING_TYPE(), true, "misc_feature");
        a << new Attribute(pd, CoreDataTypes::STRING_TYPE(), true);
        a << new Attribute(ed, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(sd, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(ald, CoreDataTypes::BOOL_TYPE(), false, false);
        a << new Attribute(amd, CoreDataTypes::BOOL_TYPE(), false, false);
    }

    Descriptor desc(ACTOR_ID, FindWorker::tr("Find substrings"), FindWorker::tr("Finds regions of similarity to the specified string in each input sequence (nucleotide or protein one). "
        "<p>Under the hood is the well-known Smith-Waterman algorithm for performing local sequence alignment."));
    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    
    QVariantMap lenMap; lenMap["minimum"] = QVariant(0); lenMap["maximum"] = QVariant(INT_MAX);
    delegates[ERR_ATTR] = new SpinBoxDelegate(lenMap);
    
    QVariantMap strandMap; 
    strandMap[FindWorker::tr("both strands")] = QVariant(FindAlgorithmStrand_Both);
    strandMap[FindWorker::tr("direct strand")] = QVariant(FindAlgorithmStrand_Direct);
    strandMap[FindWorker::tr("complement strand")] = QVariant(FindAlgorithmStrand_Complement);
    delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);
    
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath( ":core/images/find_dialog.png" );
    proto->setPrompter(new FindPrompter());
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new FindWorkerFactory());
}


/***************************
 * FindPrompter
 ***************************/
QString FindPrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID));
    Actor* seqProducer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    Actor* annProducer = input->getProducer(BioActorLibrary::FEATURE_TABLE_SLOT().getId());
    
    QString seqName = seqProducer ? tr("In each sequence from <u>%1</u>,").arg(seqProducer->getLabel()) : "";
    QString annName = annProducer ? tr(" within a set of regions from <u>%1</u>").arg(annProducer->getLabel()) : "";

    FindAlgorithmSettings cfg;
    cfg.strand = FindAlgorithmStrand(getParameter(STRAND_ATTR).toInt());
    cfg.maxErr = getParameter(ERR_ATTR).toInt();
    cfg.insDelAlg = getParameter(ALGO_ATTR).toBool();
    QString pattern = getRequiredParam(PATTERN_ATTR);

    QString strandName;
    switch (cfg.strand) {
    case FindAlgorithmStrand_Both: strandName = FindWorker::tr("both strands"); break;
    case FindAlgorithmStrand_Direct: strandName = FindWorker::tr("direct strand"); break;
    case FindAlgorithmStrand_Complement: strandName = FindWorker::tr("complement strand"); break;
    }
    if (getParameter(AMINO_ATTR).toBool()) {
        strandName += tr(" of translated sequence");
    }

    QString resultName = getRequiredParam(NAME_ATTR);
    QString match = cfg.maxErr ? tr("matches with <u>no more than %1 errors</u>").arg(cfg.maxErr) : tr("exact matches");

    //FIXME mention search algorithm?
    QString doc = tr("%1 find pattern <u>%2</u> %3."
        "<br>Look for <u>%4</u> in <u>%5</u>."
        "<br>Output the list of found regions annotated as <u>%6</u>.")
        .arg(seqName)
        .arg(pattern)
        .arg(annName)
        .arg(match)
        .arg(strandName)
        .arg(resultName);
    
    return doc;
}

/***************************
 * FindWorker
 ***************************/
FindWorker::FindWorker(Actor* a) : BaseWorker(a), input(NULL), output(NULL) {
}

void FindWorker::init() {
    input = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
}

bool FindWorker::isReady() {
    return (input && input->hasMessage());
}

Task* FindWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    cfg.strand = FindAlgorithmStrand(actor->getParameter(STRAND_ATTR)->getAttributeValue<int>());
    cfg.maxErr = actor->getParameter(ERR_ATTR)->getAttributeValue<int>();
    cfg.insDelAlg = actor->getParameter(ALGO_ATTR)->getAttributeValue<bool>();
    resultName = actor->getParameter(NAME_ATTR)->getAttributeValue<QString>();
    cfg.pattern = actor->getParameter(PATTERN_ATTR)->getAttributeValue<QString>().toUpper().toAscii();
    QVariantMap qm = inputMessage.getData().toMap();
    
    DNASequence seq = qm.value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();
    if (!seq.isNull()) {
        FindAlgorithmTaskSettings config(cfg);
        config.sequence = QByteArray(seq.constData(), seq.length());
        if (config.strand != FindAlgorithmStrand_Direct /*&& seq.alphabet->getType() == DNAAlphabet_NUCL*/) {
            QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
                lookupTranslation(seq.alphabet, DNATranslationType_NUCL_2_COMPLNUCL);
            if (!compTTs.isEmpty()) {
                config.complementTT = compTTs.first();
            } else {
                config.strand = FindAlgorithmStrand_Direct;
            }
        }
        if (actor->getParameter(AMINO_ATTR)->getAttributeValue<bool>()) {
            DNATranslationType tt = (seq.alphabet->getType() == DNAAlphabet_NUCL) ? DNATranslationType_NUCL_2_AMINO : DNATranslationType_RAW_2_AMINO;
            QList<DNATranslation*> TTs = AppContext::getDNATranslationRegistry()->lookupTranslation(seq.alphabet, tt);
            if (!TTs.isEmpty()) config.proteinTT = TTs.first(); //FIXME let user choose or use hints ?
        }

        /*if (qm.contains(BioActorLibrary::FEATURE_TABLE_SLOT().getId())) {
            const QList<SharedAnnotationData>& atl = qVariantValue<QList<SharedAnnotationData> >(qm.value(BioActorLibrary::FEATURE_TABLE_SLOT().getId()));
            Task* t = new FindAllRegionsTask(config, atl);
            connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
            return t;
        } else {*/
            config.searchRegion.len = seq.length();
            Task* t = new FindAlgorithmTask(config);
            connect(new TaskSignalMapper(t), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
            return t;
        //}
    }
    QString err = tr("Null sequence supplied to FindWorker: %1").arg(seq.getName());
    return new FailTask(err);
}

void FindWorker::sl_taskFinished(Task* t) {
    FindAlgorithmTask* ft = qobject_cast<FindAlgorithmTask*>(t);
    FindAllRegionsTask* at = qobject_cast<FindAllRegionsTask*>(t);
    assert(ft || at);
    QList<FindAlgorithmResult> res = ft ? ft->popResults() : at->getResult();
    if (output) {
        QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(FindAlgorithmResult::toTable(res, resultName));
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
        algoLog.info(tr("Found %1 matches of pattern '%2'").arg(res.size()).arg(QString(cfg.pattern)));
    }
}

bool FindWorker::isDone() {
    return !input || input->isEnded();
}

void FindWorker::cleanup() {
}


/***************************
 * FindAllRegionsTask
 ***************************/
FindAllRegionsTask::FindAllRegionsTask(const FindAlgorithmTaskSettings& s, const QList<SharedAnnotationData>& l) :
Task(tr("FindAllRegionsTask"), TaskFlag_NoRun), cfg(s), regions(l) {}

void FindAllRegionsTask::prepare() {
    foreach(SharedAnnotationData sd, regions) {
        foreach(LRegion lr, sd->location) {
            cfg.searchRegion = lr;
            addSubTask(new FindAlgorithmTask(cfg));
        }
    }
}

QList<FindAlgorithmResult> FindAllRegionsTask::getResult() {
    QList<FindAlgorithmResult> lst;
    foreach(Task* t, getSubtasks()) {
        FindAlgorithmTask* ft = qobject_cast<FindAlgorithmTask*>(t);
        lst += ft->popResults();
    }
    return lst;
}

} //namespace LocalWorkflow
} //namespace U2
