/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MuscleSchemaTask.h"

#include <U2Core/BaseDocumentFormats.h>
#include <U2Core/AppContext.h>
#include <U2Core/IOAdapter.h>

namespace U2
{

/* class MuscleSchemaTask: public Task */

const QString MuscleSchemaTask::SCHEMA_NAME = "muscle";

MuscleSchemaTask::MuscleSchemaTask(MAlignmentObject *mao_, const MuscleTaskSettings &muscleSettings_) :
        Task(tr("MuscleSchemaTask"), TaskFlags_NR_FOSCOE),
        IO_FORMATID(BaseDocumentFormats::SRF),
        schemaParameters(),
        tempSubDir(),
        inFile(), outFile(),
        mao(mao_), muscleSettings(muscleSettings_),
        runTask(0),
        saveInputFileTask(0), openResultFileTask(0)
{
    assert(mao);
}

MuscleSchemaTask::~MuscleSchemaTask()
{
    SchemaForTaskUtils::removeTempSubDir(tempSubDir);
}

RunSchemaForTask* MuscleSchemaTask::prepareRunSchemaTask()
{
    return new RunSchemaForTask(SCHEMA_NAME, schemaParameters, inFile, outFile);
}

LoadDocumentTask* MuscleSchemaTask::prepareOpenResultFileTask()
{
    IOAdapterFactory *iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    assert(iof);

    return new LoadDocumentTask(IO_FORMATID, outFile, iof);
}

void MuscleSchemaTask::prepare()
{
    tempSubDir = SchemaForTaskUtils::createTempSubDdir(getTaskId());
    if (tempSubDir.isEmpty())
    {
        setError(tr("Unable to create temporary files"));
        return;
    }

    inFile = tempSubDir + "/in.srfa";
    outFile = tempSubDir + "/out.srfa";

    Document *doc = createInputDocument();
    assert(doc);

    saveInputFileTask = new SaveDocumentTask(doc, SaveDoc_Overwrite);
    addSubTask(saveInputFileTask);
}

QList<Task*> MuscleSchemaTask::onSubTaskFinished(Task *subTask)
{
    assert(subTask);
    QList<Task*> subTasks;

    propagateSubtaskError();
    if (hasErrors() || isCanceled())
    {
        return subTasks;
    }

    if (saveInputFileTask == subTask)
    {
        runTask = prepareRunSchemaTask();
        subTasks << runTask;
    }
    else if (runTask == subTask)
    {
        openResultFileTask = prepareOpenResultFileTask();
        subTasks << openResultFileTask;
    }
    else if (openResultFileTask == subTask)
    {
        Document *doc = openResultFileTask->getDocument();

        QList <GObject*> maos = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
        GObject *go = maos.first();

        MAlignmentObject *newMao = qobject_cast<MAlignmentObject*>(go->clone());
        mao->setMAlignment(newMao->getMAlignment());

        openResultFileTask->takeDocument()->unload();
    }

    return subTasks;
}


Document* MuscleSchemaTask::createInputDocument()
{
    DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(IO_FORMATID);
    assert(df);

    IOAdapterFactory *iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::LOCAL_FILE);
    assert(iof);

    QList<GObject*> objs;
    objs << mao->clone();

    assert(!inFile.isEmpty());
    Document *doc = new Document(df, iof, GUrl(inFile), objs);

    return doc;
}

QVariantMap convertMuscleSettings(const MuscleTaskSettings &muscleSettings)
{
    QVariantMap settings;

    /* // plugins_3rdparty/umuscle/src/MuscleWorker.cpp
    p << new PortDescriptor(ind, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), true /*input*//*);
    p << new PortDescriptor(oud, BioDataTypes::MULTIPLE_ALIGNMENT_TYPE(), false /*input*//*, true /*multi*//*);
    Descriptor mod(MODE_ATTR, MuscleWorker::tr("Mode"),
        MuscleWorker::tr("Selector of preset configurations, that give you the choice of optimizing accuracy, speed, or some compromise between the two. The default favors accuracy."));
    Descriptor sd(STABLE_ATTR, MuscleWorker::tr("Stable order"),
        MuscleWorker::tr("Do not rearrange aligned sequences (-stable switch of MUSCLE). "
        "<p>Otherwise, MUSCLE re-arranges sequences so that similar sequences are adjacent in the output file. This makes the alignment easier to evaluate by eye. "));
    a << new Attribute(mod, CoreDataTypes::NUM_TYPE(), false, 0);
    a << new Attribute(sd, CoreDataTypes::BOOL_TYPE(), false, true);
    */

    /* plugins_3rdparty/umuscle/src/MuscleTask.h
    class MuscleTaskSettings {
    public:
        MuscleTaskSettings() {reset();}
        void reset();

        MuscleTaskOp    op;

        int             maxIterations;
        unsigned long   maxSecs; // 0 - unlimited
        bool            stableMode;

        //used only for MuscleTaskOp_DoAlign
        bool            alignRegion;
        LRegion         regionToAlign;

        //used only for MuscleTaskOp_AddUnalignedToProfile and MuscleTaskOp_ProfileToProfile
        MAlignment      profile;

        //number of threads: 0 - auto, 1 - serial
        int nThreads;
        QString         inputFilePath;
    };
    */

    settings["muscle_stable"] = bool(muscleSettings.stableMode);
    settings["muscle_mode"] = int(muscleSettings.op);

    return settings;
}

}   // namespace U2
