/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <string.h>

#include <uglib.h>
#include <ug_stdio.h>
#include <ug_utils.h>
#include <ug_uri.h>
#include <ug_plugin_curl.h>


char*	ug_find_help_option (int argc, char** argv)
{
	char*	arg;
	int		arg_len;

	for (argc -= 1;  argc >= 0;  argc--) {
		arg = argv[argc];
		arg_len = strlen (arg);
#ifdef _WIN32
		// Check and remove some character (space,0x20) in tail of argument from command line.
		// This problem only happen in Windows platform.
		ug_str_clear_tail_charset (arg, arg_len, " \n");
#endif
		// check short_name: -h or -?
		if (arg_len < 2 || arg[0] != '-')
			continue;
		if (arg_len == 2 && (arg[1] == 'h' || arg[1] == '?'))
			return arg;
		// check long name: --help
		if (arg_len < 6 || arg[1] != '-')
			continue;
		if (strncmp (arg+2, "help", 4) == 0) {
			if (arg_len == 6 || arg[6] == '-')
				return arg;
		}
	}
	return NULL;
}

char*	ug_find_version_option (int argc, char** argv)
{
	char*	arg;
	int		arg_len;

	for (argc -= 1;  argc >= 0;  argc--) {
		arg = argv[argc];
		arg_len = strlen (arg);
#ifdef _WIN32
		// Check and remove some character (space,0x20) in tail of argument from command line.
		// This problem only happen in Windows platform.
		ug_str_clear_tail_charset (arg, arg_len, " \n");
#endif
		// check short_name: -V
		if (arg_len < 2 || arg[0] != '-')
			continue;
		if (arg_len == 2 && arg[1] == 'V')
			return arg;
		// check long name: --version
		if (arg_len != 9 || arg[1] != '-')
			continue;
		if (strncmp (arg+2, "version", 7) == 0)
			return arg;
	}
	return NULL;
}

// ------------------------------------------------------------------
// URI list functions
// get URIs from text file
GList*	ug_text_file_get_uris (const gchar* file_utf8, GError** error)
{
	GIOChannel*		channel;
	GList*			list;
	gchar*			string;
	gchar*			escaped;
	gsize			line_len;

	string = g_filename_from_utf8 (file_utf8, -1, NULL, NULL, NULL);
	channel = g_io_channel_new_file (string, "r", error);
	g_free (string);
	if (channel == NULL)
		return NULL;
	ug_io_channel_decide_encoding (channel);

	list = NULL;
	while (g_io_channel_read_line (channel, &string, NULL, &line_len, NULL) == G_IO_STATUS_NORMAL) {
		if (string == NULL)
			continue;
		string[line_len] = 0;		// clear '\n' in tail
		// check URI scheme
		if (ug_uri_scheme_len (string) == 0)
			g_free (string);
		else {
			// if URI is not valid UTF-8 string, escape it.
			if (g_utf8_validate (string, -1, NULL) == FALSE) {
				escaped = g_uri_escape_string (string,
						G_URI_RESERVED_CHARS_ALLOWED_IN_PATH, FALSE);
				g_free (string);
				string = escaped;
			}
			list = g_list_prepend (list, string);
		}
	}
	g_io_channel_unref (channel);
	return g_list_reverse (list);
}

// get URIs from text
GList*	ug_text_get_uris (const gchar* text, gint text_len)
{
	GList*		list;
	gchar*		escaped;
	gchar*		line;
	gint		line_len;
	gint		offset;

	if (text_len == -1)
		text_len = strlen (text);

	list = NULL;
	for (offset = 0;  offset < text_len;  offset += line_len +1) {
		line_len = ug_str_line_len (text, text_len, offset);
		line = g_strndup (text + offset, line_len);
		// check URI scheme
		if (ug_uri_scheme_len (line) == 0)
			g_free (line);
		else {
			// if URI is not valid UTF-8 string, escape it.
			if (g_utf8_validate (line, -1, NULL) == FALSE) {
				escaped = g_uri_escape_string (line,
						G_URI_RESERVED_CHARS_ALLOWED_IN_PATH, FALSE);
				g_free (line);
				line = escaped;
			}
			list = g_list_prepend (list, line);
		}
	}

	return g_list_reverse (list);
}

GList*	ug_uri_list_remove_scheme (GList* list, const gchar* scheme)
{
	GList*	link;
	gchar*	text;

	for (link = list;  link;  link = link->next) {
		text = g_uri_parse_scheme (link->data);
		if (text && strcmp (text, scheme) == 0) {
			g_free (link->data);
			link->data = NULL;
		}
		g_free (text);
	}
	return g_list_remove_all (list, NULL);
}


// ------------------------------------------------------------------
// Used by ug_io_channel_decide_encoding()
// BOM = Byte Order Mark
#define UG_BOM_UTF32BE			"\x00\x00\xFE\xFF"
#define UG_BOM_UTF32BE_LEN		4
#define UG_BOM_UTF32LE			"\xFF\xFE\x00\x00"
#define UG_BOM_UTF32LE_LEN		4
#define UG_BOM_UTF8				"\xEF\xBB\xBF"
#define	UG_BOM_UTF8_LEN			3
#define UG_BOM_UTF16BE			"\xFE\xFF"
#define	UG_BOM_UTF16BE_LEN		2
#define UG_BOM_UTF16LE			"\xFF\xFE"
#define	UG_BOM_UTF16LE_LEN		2

const char*	ug_io_channel_decide_encoding (GIOChannel* channel)
{
	gchar*		encoding;
	gchar		bom[4];
	guint		bom_len;

	// The internal encoding is always UTF-8.
	// set encoding NULL is safe to use with binary data.
	g_io_channel_set_encoding (channel, NULL, NULL);
	// read 4 bytes BOM (Byte Order Mark)
	if (g_io_channel_read_chars (channel, bom, 4, NULL, NULL) != G_IO_STATUS_NORMAL)
		return NULL;

	if (memcmp (bom, UG_BOM_UTF32BE, UG_BOM_UTF32BE_LEN) == 0) {
		bom_len = UG_BOM_UTF32BE_LEN;
		encoding = "UTF-32BE";
	}
	else if (memcmp (bom, UG_BOM_UTF32LE, UG_BOM_UTF32LE_LEN) == 0) {
		bom_len = UG_BOM_UTF32LE_LEN;
		encoding = "UTF-32LE";
	}
	else if (memcmp (bom, UG_BOM_UTF8, UG_BOM_UTF8_LEN) == 0) {
		bom_len = UG_BOM_UTF8_LEN;
		encoding = "UTF-8";
	}
	else if (memcmp (bom, UG_BOM_UTF16BE, UG_BOM_UTF16BE_LEN) == 0) {
		bom_len = UG_BOM_UTF16BE_LEN;
		encoding = "UTF-16BE";
	}
	else if (memcmp (bom, UG_BOM_UTF16LE, UG_BOM_UTF16LE_LEN) == 0) {
		bom_len = UG_BOM_UTF16LE_LEN;
		encoding = "UTF-16LE";
	}
	else {
		bom_len = 0;
		encoding = NULL;
//		encoding = "UTF-8";
	}
	// repositioned before set encoding. This flushes all the internal buffers.
	g_io_channel_seek_position (channel, bom_len, G_SEEK_SET, NULL);
	// The encoding can be set now.
	g_io_channel_set_encoding (channel, encoding, NULL);
	return encoding;
}

// ------------------------------------------------------------------
gboolean uget_class_init (void)
{
	// data
	ug_data_class_register (UgDataCommonClass);
	ug_data_class_register (UgDataProxyClass);
	ug_data_class_register (UgProgressClass);
	ug_data_class_register (UgDataHttpClass);
	ug_data_class_register (UgDataFtpClass);
	// message
	ug_data_class_register (UgMessageClass);
	// plug-ins
	ug_plugin_class_register (UgPluginCurlClass);

	return TRUE;
}

void uget_class_finalize (void)
{
	// data
	ug_data_class_unregister (UgDataCommonClass);
	ug_data_class_unregister (UgDataProxyClass);
	ug_data_class_unregister (UgProgressClass);
	ug_data_class_unregister (UgDataHttpClass);
	ug_data_class_unregister (UgDataFtpClass);
	// message
	ug_data_class_unregister (UgMessageClass);
	// plug-ins
	ug_plugin_class_unregister (UgPluginCurlClass);
}

// return  1: server start and push arguments to queue.
// return  0: server exist and send arguments to server.
// return -1: error
gint	uget_ipc_init (UgIpc* ipc, int argc, char** argv)
{
	guint		count;
	gboolean	workable;

	ug_ipc_init (ipc, NULL);
	// try to start server.
	if (ug_ipc_server_start (ipc)) {
		if (argc > 1)
			g_queue_push_tail (&ipc->queue, ug_arg_new (argc, argv, TRUE));
		return 1;
	}
	else {
		// connecting to server if server already exist.
		workable = ug_ipc_client_connect (ipc);
		// try to send command-line options
		for (count=0;  count<3;  count++) {
			if (workable == FALSE) {
				g_usleep (500 * 1000);
				workable = ug_ipc_client_connect (ipc);
				continue;
			}
			if (ug_ipc_ping (ipc) == FALSE) {
				g_usleep (500 * 1000);
				continue;
			}
			ug_ipc_send (ipc, argc, argv);
			return 0;
		}
	}

	return -1;
}

// ----------------------------------------------------------------------------
// UgetOption
//
void	uget_option_init  (UgetOption* uopt)
{
	uopt->context = g_option_context_new ("[URL 1] [URL 2] ... [URL n]");
	uopt->group   = g_option_group_new (NULL, NULL, NULL, NULL, NULL);
	uopt->list    = g_list_prepend (NULL, (gpointer) UgOptionMain);
	uopt->data    = UgOptionMain->data;

	g_option_group_add_entries (uopt->group, UgOptionMain->entry);
	g_option_context_set_main_group (uopt->context, uopt->group);
}

void	uget_option_add (UgetOption* uopt, const UgOption* option, GOptionGroup* group)
{
	GList*	link;

	if (option) {
		link = g_list_find (uopt->list, option);
		if (link)
			return;
	}

	if (option == NULL)
		g_option_context_add_group (uopt->context, group);
	else {
		if (group == NULL)
			g_option_group_add_entries (uopt->group, option->entry);
		else {
			g_option_group_add_entries (group, option->entry);
			g_option_context_add_group (uopt->context, group);
		}
		uopt->list = g_list_prepend (uopt->list, (gpointer) option);
	}
}

void	uget_option_help  (UgetOption* uopt, const char* progname, const char* help_option)
{
	gint		argc;
	gchar*		args[2];
	gchar**		argv;

	if (help_option == NULL)
		help_option = "-h";
	args[0] = (gchar*) progname;
	args[1] = (gchar*) help_option;
	argv = args;
	argc = 2;

	g_option_context_set_help_enabled (uopt->context, TRUE);
	g_option_context_parse (uopt->context, &argc, &argv, NULL);
}

// used by uget_option_parse()
static GList*	ug_arg_get_rest (GPtrArray* args)
{
	GList*		strings;
	gchar**		cur;
	gchar**		end;

	strings = NULL;
	cur  = (gchar**) args->pdata;
	end  = cur + args->len;
//        skip argv[0]
	for (cur = cur + 1;  cur < end;  cur++) {
		if ((*cur)[0] == '-')
			continue;
		strings = g_list_prepend (strings, g_strdup (*cur));
	}
	return g_list_reverse (strings);
}

GList*	uget_option_parse (UgetOption* uopt, GPtrArray* args)
{
	GPtrArray*		temp_args;
	UgDataset*		dataset;
	UgDataCommon*	common;
	GList*			downloads;
	GList*			strings;

	// setup
	g_option_context_set_help_enabled (uopt->context, FALSE);
	g_option_context_set_ignore_unknown_options (uopt->context, TRUE);
	// clear argument data
	g_list_foreach (uopt->list, (GFunc) ug_option_clear, NULL);

	temp_args = ug_arg_copy (args, FALSE);
	g_option_context_parse (uopt->context,
	                        (gint*)    &temp_args->len,
	                        (gchar***) &temp_args->pdata,
	                        NULL);
	// get URLs from file or command-line
	if (uopt->data->input_file)
		strings = ug_text_file_get_uris (uopt->data->input_file, NULL);
	else
		strings = NULL;
	strings = g_list_concat (strings, ug_arg_get_rest (temp_args));
	ug_arg_free (temp_args, FALSE);

	// get argument data and create jobs
	downloads = NULL;
	for (strings = g_list_last (strings);  strings;  strings = strings->prev) {
		dataset = ug_dataset_new ();
		common = ug_dataset_alloc_front (dataset, UgDataCommonClass);
		common->url = strings->data;
		common->keeping.url = TRUE;		// keep common->url no change
		g_list_foreach (uopt->list, (GFunc) ug_option_get, dataset);
		downloads = g_list_prepend (downloads, dataset);
		// if previous link is first one...
		if (strings->prev == NULL)
			break;
	}
	// free string list
	g_list_free (strings);

	return downloads;
}

