/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <UgUtils.h>
#include <UgString.h>
#include <UgData-download.h>
#include <UgCategory-gtk.h>
#include <UgDownloadWidget.h>

#include <glib/gi18n.h>

void	ug_download_widget_init (UgDownloadWidget* dwidget, GtkTreeModel* model)
{
	GtkTreeSelection*	selection;
	GtkScrolledWindow*	scroll;

	dwidget->self  = gtk_scrolled_window_new (NULL, NULL);
	dwidget->view  = ug_download_view_new ();
	dwidget->model = model;
	gtk_tree_view_set_model (dwidget->view, model);
	selection = gtk_tree_view_get_selection (dwidget->view);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	scroll = GTK_SCROLLED_WINDOW (dwidget->self);
	gtk_scrolled_window_set_shadow_type (scroll, GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (scroll, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scroll), GTK_WIDGET (dwidget->view));
	gtk_widget_show_all (dwidget->self);
}

void	ug_download_widget_finalize (UgDownloadWidget* dwidget)
{
	gtk_widget_destroy (dwidget->self);
	g_object_unref (dwidget->model);
}

void	ug_download_widget_set_filter (UgDownloadWidget* dwidget, GtkTreeModel* model, GtkTreeModelFilterVisibleFunc func, gpointer data)
{
	GtkTreeModel*		filter;

	filter = gtk_tree_model_filter_new (model, NULL);
	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (filter), func, data, NULL);

	gtk_tree_view_set_model (dwidget->view, filter);
	if (dwidget->model)
		g_object_unref (dwidget->model);
	dwidget->model = filter;
}

void	ug_download_widget_set_sortable (UgDownloadWidget* dwidget, GtkTreeModel* model, GtkTreeIterCompareFunc func, gpointer data)
{
	GtkTreeModel*		sort;
	GtkTreeSortable*	sortable;

	sort = gtk_tree_model_sort_new_with_model (model);
	sortable = GTK_TREE_SORTABLE (sort);
	gtk_tree_sortable_set_default_sort_func (sortable, func, data, NULL);

	gtk_tree_view_set_model (dwidget->view, sort);
	if (dwidget->model)
		g_object_unref (dwidget->model);
	dwidget->model = sort;
}

GList*	ug_download_widget_get_selected (UgDownloadWidget* dwidget)
{
	UgDataset*			dataset;
	GtkTreeSelection*	selection;
	GtkTreeModel*		model;
	GtkTreeIter			iter;
	GList*				list;
	GList*				link;

	selection = gtk_tree_view_get_selection (dwidget->view);
	list = gtk_tree_selection_get_selected_rows (selection, &model);
	for (link = list;  link;  link = link->next) {
		gtk_tree_model_get_iter (model, &iter, link->data);
		gtk_tree_model_get (model, &iter, 0, &dataset, -1);
		gtk_tree_path_free (link->data);
		link->data = dataset;
	}
	return list;
}

GList*	ug_download_widget_get_selected_indices (UgDownloadWidget* dwidget)
{
	GtkTreeSelection*	selection;
	GList*				list;
	GList*				link;
	gint				index;

	selection = gtk_tree_view_get_selection (dwidget->view);
	list = gtk_tree_selection_get_selected_rows (selection, NULL);
	for (link = list;  link;  link = link->next) {
		index = *gtk_tree_path_get_indices (link->data);
		gtk_tree_path_free (link->data);
		link->data = GINT_TO_POINTER (index);
	}
	return list;
}

gint	ug_download_widget_count_selected (UgDownloadWidget* dwidget)
{
	GtkTreeSelection*	selection;

	selection = gtk_tree_view_get_selection (dwidget->view);
	return gtk_tree_selection_count_selected_rows (selection);
}

UgDataset*	ug_download_widget_get_cursor (UgDownloadWidget* dwidget)
{
	UgDataset*		dataset;
	GtkTreePath*	path;
	GtkTreeIter		iter;
	gboolean		valid;

	gtk_tree_view_get_cursor (dwidget->view, &path, NULL);
	if (path) {
		valid = gtk_tree_model_get_iter (dwidget->model, &iter, path);
		gtk_tree_path_free (path);
		if (valid) {
			gtk_tree_model_get (dwidget->model, &iter, 0, &dataset, -1);
			return dataset;
		}
	}
	return NULL;
}

// ----------------------------------------------------------------------------
// GtkTreeView for UgDataset
//
#define UG_DOWNLOAD_ICON_NORMAL			GTK_STOCK_FILE
#define UG_DOWNLOAD_ICON_PAUSED			GTK_STOCK_MEDIA_PAUSE
#define UG_DOWNLOAD_ICON_ACTIVE			GTK_STOCK_MEDIA_PLAY
#define UG_DOWNLOAD_ICON_COMPLETED		GTK_STOCK_YES
#define UG_DOWNLOAD_ICON_ERROR			GTK_STOCK_DIALOG_ERROR
#define UG_DOWNLOAD_ICON_RECYCLED		GTK_STOCK_DELETE
#define UG_DOWNLOAD_ICON_FINISHED		GTK_STOCK_GOTO_LAST

static void col_set_icon (GtkTreeViewColumn *tree_column,
                          GtkCellRenderer   *cell,
                          GtkTreeModel      *model,
                          GtkTreeIter       *iter,
                          gpointer           data)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	const gchar*	stock_id;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	relation = UG_DATASET_RELATION_GTK (dataset);
	// set stock_id
	if (relation->hints & UG_HINT_ACTIVE)
		stock_id = UG_DOWNLOAD_ICON_ACTIVE;
	else if (relation->hints & UG_HINT_PAUSED)
		stock_id = UG_DOWNLOAD_ICON_PAUSED;
	else if (relation->hints & UG_HINT_ERROR)
		stock_id = UG_DOWNLOAD_ICON_ERROR;
	else if (relation->hints & UG_HINT_COMPLETED)
		stock_id = UG_DOWNLOAD_ICON_COMPLETED;
	else
		stock_id = UG_DOWNLOAD_ICON_NORMAL;

	g_object_set (cell, "stock-id", stock_id, NULL);
}

static void col_set_icon_all (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer   *cell,
                              GtkTreeModel      *model,
                              GtkTreeIter       *iter,
                              gpointer           data)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	const gchar*	stock_id;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	relation = UG_DATASET_RELATION_GTK (dataset);
	// set stock_id
	if (relation->hints & UG_HINT_RECYCLED)
		stock_id = UG_DOWNLOAD_ICON_RECYCLED;
	else if (relation->hints & UG_HINT_FINISHED)
		stock_id = UG_DOWNLOAD_ICON_FINISHED;
	else if (relation->hints & UG_HINT_ACTIVE)
		stock_id = UG_DOWNLOAD_ICON_ACTIVE;
	else if (relation->hints & UG_HINT_PAUSED)
		stock_id = UG_DOWNLOAD_ICON_PAUSED;
	else if (relation->hints & UG_HINT_ERROR)
		stock_id = UG_DOWNLOAD_ICON_ERROR;
	else if (relation->hints & UG_HINT_COMPLETED)
		stock_id = UG_DOWNLOAD_ICON_COMPLETED;
	else
		stock_id = UG_DOWNLOAD_ICON_NORMAL;

	g_object_set (cell, "stock-id", stock_id, NULL);
}

static void col_set_name (GtkTreeViewColumn *tree_column,
                          GtkCellRenderer   *cell,
                          GtkTreeModel      *model,
                          GtkTreeIter       *iter,
                          gpointer           data)
{
	UgDataset*		dataset;
	UgDataCommon*	common;
	gchar*			string = NULL;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	common = UG_DATASET_COMMON (dataset);

	if (common->name)
		string = common->name;
	else if (common->file)
		string = common->file;
	else
		string = _("unnamed");

	g_object_set (cell, "text", string, NULL);
}

static void col_set_complete (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer   *cell,
                              GtkTreeModel      *model,
                              GtkTreeIter       *iter,
                              gpointer           data)
{
	UgDataset*		dataset;
	UgProgress*		progress;
	gchar*			string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_dtoa_unit ((gdouble) progress->complete, 1, NULL);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_total (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_dtoa_unit ((gdouble) progress->total, 1, NULL);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_percent (GtkTreeViewColumn *tree_column,
                             GtkCellRenderer   *cell,
                             GtkTreeModel      *model,
                             GtkTreeIter       *iter,
                             gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress) {
//		string = g_strdup_printf ("%.*f%c", 1, progress->percent, '%');
		string = g_strdup_printf ("%.1f%c", progress->percent, '%');
		g_object_set (cell, "visible", 1, NULL);
		g_object_set (cell, "value", (gint) progress->percent, NULL);
		g_object_set (cell, "text", string, NULL);
		g_free (string);
	}
	else {
		g_object_set (cell, "visible", 0, NULL);
		g_object_set (cell, "value", 0, NULL);
		g_object_set (cell, "text", "", NULL);
	}
}

static void col_set_consume_time (GtkTreeViewColumn *tree_column,
                                  GtkCellRenderer   *cell,
                                  GtkTreeModel      *model,
                                  GtkTreeIter       *iter,
                                  gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_from_seconds ((guint) progress->consume_time, TRUE);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_remain_time (GtkTreeViewColumn *tree_column,
                                 GtkCellRenderer   *cell,
                                 GtkTreeModel      *model,
                                 GtkTreeIter       *iter,
                                 gpointer           data)
{
	UgDataset*	dataset;
	UgRelation*	relation;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);
	relation = UG_DATASET_RELATION (dataset);

	if (progress && relation && relation->plugin)
		string = ug_str_from_seconds ((guint) progress->remain_time, TRUE);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_speed (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*	dataset;
	UgRelation*	relation;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);
	relation = UG_DATASET_RELATION (dataset);

	if (progress && relation && relation->plugin)
		string = ug_str_dtoa_unit (progress->download_speed, 1, "/s");
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_retry (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*		dataset;
	UgDataCommon*	common;
	gchar*			string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	common = UG_DATASET_COMMON (dataset);

	if (common->retry_count != 0)
		string = g_strdup_printf ("%d", common->retry_count);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_category (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer   *cell,
                              GtkTreeModel      *model,
                              GtkTreeIter       *iter,
                              gpointer           data)
{
	UgDataset*		dataset;
	UgRelationGtk*	relation;
	gchar*			string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	relation = UG_DATASET_RELATION_GTK (dataset);

	if (relation && relation->category)
		string = relation->category->name;
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
}

static void col_set_url (GtkTreeViewColumn *tree_column,
                         GtkCellRenderer   *cell,
                         GtkTreeModel      *model,
                         GtkTreeIter       *iter,
                         gpointer           data)
{
	UgDataset*	dataset;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);

	g_object_set (cell, "text", UG_DATASET_COMMON (dataset)->url, NULL);
}

GtkTreeView*	ug_download_view_new (void)
{
	GtkTreeView*       tview;
	GtkTreeSelection*  selection;
	GtkCellRenderer*   renderer;
	GtkCellRenderer*   renderer_progress;
	GtkTreeViewColumn* column;

	tview = (GtkTreeView*)gtk_tree_view_new ();
	gtk_tree_view_set_fixed_height_mode (tview, TRUE);
	selection = gtk_tree_view_get_selection (tview);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	// column name
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Name"));
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_icon,
	                                         NULL, NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_name,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 180);
	gtk_tree_view_column_set_expand (column, TRUE);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column completed
	column = gtk_tree_view_column_new ();
	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "xalign", 1.0, NULL);
	gtk_tree_view_column_set_title (column, _("Complete"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_complete,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 80);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column total
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Size"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_total,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 80);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column percent
	column = gtk_tree_view_column_new ();
	renderer_progress = gtk_cell_renderer_progress_new ();
	gtk_tree_view_column_set_title (column, _("%"));
	gtk_tree_view_column_pack_start (column, renderer_progress, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer_progress,
	                                         col_set_percent,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 60);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column "Elapsed" for consuming time
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Elapsed"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_consume_time,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 65);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column "Left" for remaining time
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Left"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_remain_time,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 65);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// columns speed
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Speed"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_speed,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 90);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column retries
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Retry"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_retry,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 45);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column category
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Category"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_category,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 100);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column url
//	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("URL"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_url,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 300);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	gtk_widget_show (GTK_WIDGET (tview));
	return tview;
}

void	ug_download_view_use_all_icon (GtkTreeView* view, gboolean visible_all)
{
	GtkCellRenderer*   renderer;
	GtkTreeViewColumn* column;

	// clear
	column = gtk_tree_view_get_column (view, 0);
	gtk_tree_view_column_clear (column);
	// pack icon renderer
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	if (visible_all) {
		gtk_tree_view_column_set_cell_data_func (column, renderer,
				col_set_icon_all, NULL, NULL);
	}
	else {
		gtk_tree_view_column_set_cell_data_func (column, renderer,
				col_set_icon, NULL, NULL);
	}
	// pack text renderer
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column, renderer,
			col_set_name, NULL, NULL);
}

