#!/usr/bin/env python
#
# Copyright 2019 Ettus Research, a National Instruments Brand
#
# SPDX-License-Identifier: GPL-3.0-or-later
#
"""
Register map for LMX2572
"""

########################################################################
# Template for raw text data describing registers
# name addr[bit range inclusive] default optional enums
########################################################################

REGS_TMPL = """\
########################################################################
## address 0
########################################################################
powerdown               0[0]        0       normal_operation, power_down
reset                   0[1]        0       normal_operation, reset
muxout_ld_sel           0[2]        1       register_readback, lock_detect
fcal_en                 0[3]        1       invalid, enable
reg0_reserved0          0[4]        1
fcal_lpfd_adj           0[5:6]      0
fcal_hpfd_adj           0[7:8]      0
out_mute                0[9]        1       disabled, muted
reg0_reserved1          0[10]       0
add_hold                0[11]       0
reg0_reserved2          0[12:13]    0x2
vco_phase_sync_en       0[14]       0       normal_operation, phase_sync_mode
ramp_en                 0[15]       0       normal_operation, frequency_ramping
########################################################################
## address 1
########################################################################
cal_clk_div             1[0:2]      0
reg1_reserved0          1[3:15]     0x101
########################################################################
## address 2
########################################################################
reg2_reserved0          2[0:15]     0x500
########################################################################
## address 3
########################################################################
reg3_reserved0          3[0:15]     0x782
########################################################################
## address 4
########################################################################
reg4_reserved0          4[0:15]     0xA43
########################################################################
## address 5
########################################################################
reg5_reserved0          5[0:10]     0xC8
ipbuf_term              5[11]       0       normal_operation, internally_terminated
ipbuf_type              5[12]       1       differential, single_ended
reg5_reserved1          5[13:15]    1
########################################################################
## address 6
########################################################################
reg6_reserved0          6[0:10]     0x2
ldo_dly                 6[11:15]    0x19
########################################################################
## address 7
########################################################################
reg7_reserved0          7[0:13]     0xB2
out_force               7[14]       0       use_out_mute, no_mute
reg7_reserved1          7[15]       0
########################################################################
## address 8
########################################################################
reg8_reserved0          8[0:10]     0
vco_capctrl_force       8[11]       0       normal_operation, use_vco_capctrl
reg8_reserved1          8[12:13]    0x2
vco_daciset_force       8[14]       0       normal_operation, use_vco_daciset
reg8_reserved2          8[15]       0
########################################################################
## address 9
########################################################################
reg9_reserved0          9[0:11]     0x4
osc_2x                  9[12]       0       disabled, enabled
reg9_reserved1          9[13]       0
mult_hi                 9[14]       0       less_than_equal_to_100M, greater_than_100M
reg9_reserved2          9[15]       0
########################################################################
## address 10
########################################################################
reg10_reserved0         10[0:6]     0x78
mult                    10[7:11]    1
reg10_reserved1         10[12:15]   1
########################################################################
## address 11
########################################################################
reg11_reserved0         11[0:3]     0x8
pll_r                   11[4:11]    1
reg11_reserved1         11[12:15]   0xB
########################################################################
## address 12
########################################################################
pll_r_pre               12[0:11]    1
reg12_reserved0         12[12:15]   0x5
########################################################################
## address 13
########################################################################
reg13_reserved0          13[0:15]   0x4000
########################################################################
## address 14
########################################################################
reg14_reserved0         14[0:2]     0x0
cpg                     14[3:6]     0x8
reg14_reserved1         14[7:15]    0x30
########################################################################
## address 15
########################################################################
reg15_reserved0          15[0:15]   0x60E
########################################################################
## address 16
########################################################################
vco_daciset             16[0:8]     0x80
reg16_reserved0         16[9:15]    0
########################################################################
## address 17
########################################################################
vco_daciset_strt        17[0:8]     0x96
reg17_reserved0         17[9:15]    0
########################################################################
## address 18
########################################################################
reg18_reserved0          18[0:15]   0x64
########################################################################
## address 19
########################################################################
vco_capctrl             19[0:7]     0xB7
reg19_reserved0         19[8:15]    0x27
########################################################################
## address 20
########################################################################
reg20_reserved0         20[0:9]     0x48
vco_sel_force           20[10]      0       disabled, enabled
vco_sel                 20[11:13]   0x6
reg20_reserved1         20[14:15]   1
########################################################################
## address 21
########################################################################
reg21_reserved0         21[0:15]    0x409
########################################################################
## address 22
########################################################################
reg22_reserved0         22[0:15]    1
########################################################################
## address 23
########################################################################
reg23_reserved0         23[0:15]    0x7C
########################################################################
## address 24
########################################################################
reg24_reserved0         24[0:15]    0x71A
########################################################################
## address 25
########################################################################
reg25_reserved0         25[0:15]    0x624
########################################################################
## address 26
########################################################################
reg26_reserved0         26[0:15]    0x808
########################################################################
## address 27
########################################################################
reg27_reserved0         27[0:15]    0x2
########################################################################
## address 28
########################################################################
reg28_reserved0         28[0:15]    0x488
########################################################################
## address 29
########################################################################
reg29_reserved0         29[0:15]    0x18C6
########################################################################
## address 30
########################################################################
reg30_reserved0         30[0:15]    0x18C6
########################################################################
## address 31
########################################################################
reg31_reserved0         31[0:15]    0xC3E6
########################################################################
## address 32
########################################################################
reg32_reserved0         32[0:15]    0x5BF
########################################################################
## address 33
########################################################################
reg33_reserved0         33[0:15]    0x1E01
########################################################################
## address 34
########################################################################
pll_n_upper_3_bits      34[0:2]     0
reg34_reserved0         34[3:15]    0x2
########################################################################
## address 35
########################################################################
reg35_reserved0         35[0:15]    0x4
########################################################################
## address 36
########################################################################
pll_n_lower_16_bits     36[0:15]    0x28
########################################################################
## address 37
########################################################################
reg37_reserved0         37[0:7]     0x5
pfd_dly_sel             37[8:13]    0x2
reg37_reserved1         37[14]      0
mash_seed_en            37[15]      0       disabled, enabled
########################################################################
## address 38
########################################################################
pll_den_upper           38[0:15]    0xFFFF
########################################################################
## address 39
########################################################################
pll_den_lower           39[0:15]    0xFFFF
########################################################################
## address 40
########################################################################
mash_seed_upper         40[0:15]    0
########################################################################
## address 41
########################################################################
mash_seed_lower         41[0:15]    0
########################################################################
## address 42
########################################################################
pll_num_upper           42[0:15]    0
########################################################################
## address 43
########################################################################
pll_num_lower           43[0:15]    0
########################################################################
## address 44
########################################################################
mash_order              44[0:2]     0x2     interger_mode, first_order, second_order, third_order, fourth_order
reg44_reserved0         44[3:4]     0
mash_reset_n            44[5]       1       reset, normal_operation
outa_pd                 44[6]       0       normal_operation, power_down
outb_pd                 44[7]       1       normal_operation, power_down
outa_pwr                44[8:13]    0x22
reg44_reserved1         44[14:15]   0
########################################################################
## address 45
########################################################################
outb_pwr                45[0:5]     0x22
register45_reserved0    45[6:10]    0x18
outa_mux                45[11:12]   0       channel_divider, vco, not_used, high_impedance
register45_reserved1    45[13:15]   0x6
########################################################################
## address 46
########################################################################
outb_mux                46[0:1]     0       channel_divider, vco, sysref, high_impedance
register46_reserved0    46[2:15]    0x1FC
########################################################################
## address 47
########################################################################
reg47_reserved0         47[0:15]    0x300
########################################################################
## address 48
########################################################################
reg48_reserved0         48[0:15]    0x3E0
########################################################################
## address 49
########################################################################
reg49_reserved0         49[0:15]    0x4180
########################################################################
## address 50
########################################################################
reg50_reserved0         50[0:15]    0x80
########################################################################
## address 51
########################################################################
reg51_reserved0         51[0:15]    0x80
########################################################################
## address 52
########################################################################
reg52_reserved0         52[0:15]    0x420
########################################################################
## address 53
########################################################################
reg53_reserved0         53[0:15]    0
########################################################################
## address 54
########################################################################
reg54_reserved0         54[0:15]    0
########################################################################
## address 55
########################################################################
reg55_reserved0         55[0:15]    0
########################################################################
## address 56
########################################################################
reg56_reserved0         56[0:15]    0
########################################################################
## address 57
########################################################################
reg57_reserved0         57[0:15]    0
########################################################################
## address 58
########################################################################
reg58_reserved0         58[0:8]     1
inpin_fmt               58[9:11]    0       SYNC_EQUALS_SYSREFREQ_EQUALS_CMOS, SYNC_EQUALS_LVDS_SYSREFREQ_EQUAL_CMOS, SYNC_EQUALS_CMOS_SYSREFREQ_EQUAL_LVDS, SYNC_EQUALS_SYSREFREQ_EQUALS_LVDS, SYNC_EQUALS_SYSREFREQ_EQUALS_CMOS2, SYNC_EQUALS_LVDSFILTERED_SYSREFREQ_EQUALS_CMOS, SYNC_EQUALS_CMOS_SYSREFREQ_EQUALS_LVDSFILTERED, SYNC_EQUALS_SYSREFREQ_EQUALS_LVDSFILTERED
inpin_lvl               58[12:13]   0       vin_divide_by_4, vin, vin_divide_by_2, invalid
inpin_hyst              58[14]      0       disabled, enabled
inpin_ignore            58[15]      1
########################################################################
## address 59
########################################################################
ld_type                 59[0]       1       vcocal, vtune_and_vcocal
reg59_reserved0         59[1:15]    0
########################################################################
## address 60
########################################################################
ld_dly                  60[0:15]    0x3E8
########################################################################
## address 61
########################################################################
reg61_reserved0         61[0:15]    0xA8
########################################################################
## address 62
########################################################################
reg62_reserved0         62[0:9]     0xAF
dblbuf_en_0             62[10]      0       disabled, enabled
dblbuf_en_1             62[11]      0       disabled, enabled
dblbuf_en_2             62[12]      0       disabled, enabled
dblbuf_en_3             62[13]      0       disabled, enabled
dblbuf_en_4             62[14]      0       disabled, enabled
dblbuf_en_5             62[15]      0       disabled, enabled
########################################################################
## address 63
########################################################################
reg63_reserved0         63[0:15]    0
########################################################################
## address 64
########################################################################
reg64_reserved0         64[0:15]    0x1388
########################################################################
## address 65
########################################################################
reg65_reserved0         65[0:15]    0
########################################################################
## address 66
########################################################################
reg66_reserved0         6[0:15]    0x1F4
########################################################################
## address 67
########################################################################
reg67_reserved0         67[0:15]    0
########################################################################
## address 68
########################################################################
reg68_reserved0         68[0:15]    0x3E8
########################################################################
## address 69
########################################################################
mash_rst_count_upper    69[0:15]    0
########################################################################
## address 70
########################################################################
mash_rst_count_lower    70[0:15]    0xC350
########################################################################
## address 71
########################################################################
reg71_reserved0         71[0:1]     1
sysref_repeat           71[2]       0       master_mode, repeater_mode
sysref_en               71[3]       0       disabled, enabled
sysref_pulse            71[4]       0       disabled, enabled
sysref_div_pre          71[5:7]     0x4     divide_by_2=2, divide_by_4=4
reg71_reserved1         71[8:15]    0
########################################################################
## address 72
########################################################################
sysref_div              72[0:10]    1
reg72_reserved0         72[11:15]   0
########################################################################
## address 73
########################################################################
jesd_dac1_ctrl          73[0:5]     0x3F
jesd_dac2_ctrl          73[6:11]    0
reg73_reserved0         73[12:15]   0
########################################################################
## address 74
########################################################################
jesd_dac3_ctrl          74[0:5]     0
jesd_dac4_ctrl          74[6:11]    0
sysref_pulse_cnt        74[12:15]   0
########################################################################
## address 75
########################################################################
reg75_reserved0         75[0:5]     0
chdiv                   75[6:10]    0   divide_by_2=0, divide_by_4=1, divide_by_8=3,  divide_by_16=5, divide_by_32=7, divide_by_64=9, divide_by_128=12, divide_by_256=14
reg75_reserved1         75[11:15]   1
########################################################################
## address 76
########################################################################
reg76_reserved0         76[0:15]    0xC
########################################################################
## address 77
########################################################################
reg77_reserved0         77[0:15]    0
########################################################################
## address 78
########################################################################
reg78_reserved0         78[0]       0
vco_capctrl_strt        78[1:8]     0x32
quick_recal_en          78[9]       0
reg78_reserved1         78[10]      0
ramp_thresh_33rd        78[11]      0
reg78_reserved2         78[12:15]   0
########################################################################
## address 79
########################################################################
ramp_thresh_upper       79[0:15]    0
########################################################################
## address 80
########################################################################
ramp_thresh_lower       80[0:15]    0
########################################################################
## address 81
########################################################################
ramp_limit_high_33rd    81[0]       0
reg81_reserved0         81[1:15]    0
########################################################################
## address 82
########################################################################
ramp_limit_high_upper   82[0:15]    0
########################################################################
## address 83
########################################################################
ramp_limit_high_lower   83[0:15]    0
########################################################################
## address 84
########################################################################
ramp_limit_low_33rd     84[0]       0
reg84_reserved0         84[1:15]    0
########################################################################
## address 85
########################################################################
ramp_limit_low_upper    85[0:15]    0
########################################################################
## address 86
########################################################################
ramp_limit_low_lower    86[0:15]    0
########################################################################
## address 87
########################################################################
reg87_reserved0         87[0:15]    0
########################################################################
## address 88
########################################################################
reg88_reserved0         88[0:15]    0
########################################################################
## address 89
########################################################################
reg89_reserved0         89[0:15]    0
########################################################################
## address 90
########################################################################
reg90_reserved0         90[0:15]    0
########################################################################
## address 91
########################################################################
reg91_reserved0         91[0:15]    0
########################################################################
## address 92
########################################################################
reg92_reserved0         92[0:15]    0
########################################################################
## address 93
########################################################################
reg93_reserved0         93[0:15]    0
########################################################################
## address 94
########################################################################
reg94_reserved0         94[0:15]    0
########################################################################
## address 95
########################################################################
reg95_reserved0         95[0:15]    0
########################################################################
## address 96
########################################################################
reg96_reserved0         96[0:1]     0
ramp_burst_count        96[2:14]    0
ramp_burst_en           96[15]      0       disabled, enabled
########################################################################
## address 97
########################################################################
ramp_burst_trig         97[0:1]     0       ramp_transistion=0, trigger_a=1, trigger_b=2
reg97_reserved0         97[2]       0
ramp_triga              97[3:6]     0       disabled=0, rampclk_rising_edge=1, rampdir_rising_edge=2, always_triggered=4, rampclk_falling_edge=9, rampdir_falling_edge=10
ramp_trgb               97[7:10]    0       disabled=0, rampclk_rising_edge=1, rampdir_rising_edge=2, always_triggered=4, rampclk_falling_edge=9, rampdir_falling_edge=10
reg97_reserved1         97[11:14]   0
ramp0_rst               97[15]      0       disabled, reset
########################################################################
## address 98
########################################################################
ramp0_dly               98[0]       0
reg98_reserved0         98[1]       0
ramp0_inc_upper_14      98[15:2]    0
########################################################################
## address 99
########################################################################
ramp0_inc_lower         99[0:15]    0
########################################################################
## address 100
########################################################################
ramp0_len               100[0:15]   0
########################################################################
## address 101
########################################################################
ramp0_next_trig         101[0:1]    0       timeout_counter, trigger_a, trigger_b
reg101_reserved0        101[2:3]    0
ramp0_next              101[4]      0       ramp0, ramp1
ramp1_rst               101[5]      0       disabled, reset
ramp1_dly               101[6]      0
reg101_reserved1        101[7:15]   0
########################################################################
## address 102
########################################################################
ramp1_inc_upper_14      102[13:0]   0
reg102_reserved0        102[14:15]  0
########################################################################
## address 103
########################################################################
ramp1_inc_lower         103[0:15]   0
########################################################################
## address 104
########################################################################
ramp1_len               104[0:15]   0
########################################################################
## address 105
########################################################################
ramp1_next_trig         105[0:1]    0       timeout_counter, trigger_a, trigger_b
reg105_reserved0        105[2:3]    0
ramp1_next              105[4]      0       ramp0, ramp1
ramp_manual             105[5]      0       automatic_ramping, manual_ramping
ramp_dly_cnt            105[6:15]   0x111
########################################################################
## address 106
########################################################################
ramp_scale_count        106[0:2]    0x7
reg106_reserved0        106[3]      0
ramp_trig_cal           106[4]      0       disabled, enabled
reg106_reserved1        106[5:15]   0
########################################################################
## address 114
########################################################################
fsk_mode_sel            114[0:1]    0       unused=0,fsk_spi=2, fsk_spi_fast=3
fsk_spi_dev_sel         114[2:4]    0
fsk_spi_level           114[5:6]    0       disabled, 2fsk, 4fsk, 8fsk
reg114_reserved0        114[7:9]    0
fsk_en                  114[10]     0       disabled, enabled
reg114_reserved1        114[11:15]  0xF
########################################################################
## address 115
########################################################################
reg115_reserved0        115[0:2]    0
fsk_dev_scale           115[3:7]    0
reg115_reserved1        115[8:15]   0
########################################################################
## address 116
########################################################################
fsk_dev0                116[0:15]   0
########################################################################
## address 117
########################################################################
fsk_dev1                117[0:15]   0
########################################################################
## address 118
########################################################################
fsk_dev2                118[0:15]   0
########################################################################
## address 119
########################################################################
fsk_dev3                119[0:15]   0
########################################################################
## address 120
########################################################################
fsk_dev4                120[0:15]   0
########################################################################
## address 121
########################################################################
fsk_dev5                121[0:15]   0
########################################################################
## address 122
########################################################################
fsk_dev6                122[0:15]   0
########################################################################
## address 123
########################################################################
fsk_dev7                123[0:15]   0
########################################################################
## address 124
########################################################################
fsk_spi_dev             124[0:15]   0
########################################################################
## address 125
########################################################################
reg125                  125[0:15]   0x2288
"""
########################################################################
# Template for methods in the body of the struct
########################################################################

BODY_TMPL = """\
uint16_t get_reg(int addr){
    uint16_t reg = 0;
    switch(addr){
    % for addr in sorted(set(map(lambda r: r.get_addr(), regs))):
    case ${addr}:
        % for reg in filter(lambda r: r.get_addr() == addr, regs):
        reg |= (uint16_t(${reg.get_name()}) & ${reg.get_mask()}) << ${reg.get_shift()};
        % endfor
        break;
    % endfor
    }
    return reg;
}

std::set<uint8_t> get_ro_regs()
{
    return {107, 108, 109, 110, 111, 112, 113};
}

int get_num_regs()
{
    return 126;
}
"""

PY_BODY_TMPL = """\
def get_reg(self, addr):
    reg = 0
    % for addr in sorted(set(map(lambda r: r.get_addr(), regs))):
    <% if_state = 'if' if loop.index == 0 else 'elif' %>
    ${if_state} addr == ${addr}:
        % for reg in filter(lambda r: r.get_addr() == addr, regs):
        % if reg.get_enums():
        reg |= (self.${reg.get_name()}.value & ${reg.get_mask()}) << ${reg.get_shift()}
        % else:
        reg |= (self.${reg.get_name()} & ${reg.get_mask()}) << ${reg.get_shift()}
        % endif
        % endfor
    % endfor
    return reg

"""

if __name__ == "__main__":
    import common

    common.generate(
        name="lmx2572_regs",
        regs_tmpl=REGS_TMPL,
        body_tmpl=BODY_TMPL,
        py_body_tmpl=PY_BODY_TMPL,
        file=__file__,
    )
