/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.uima.jcas;

import java.util.ArrayList;

/**
 * Maintains a registry of JCas cover classes that have been loaded.  Methods on this
 * class are called from JCas cover classes (that is, classes generated by JCasGen).
 * They are not intended to be called directly from user-written code.
 */
public class JCasRegistry {
  private static ArrayList loadedJCasClasses = new ArrayList();
  
  /**
   * Registers a JCas cover class with this registry.  The registry will assign
   * it a unique index, which is then used by the cover-class to identify itself
   * to the JCas implementation.
   * @param aJCasCoverClass the class to register
   * @return the unique index value for this class.
   */
  public static synchronized int register(Class aJCasCoverClass) {
    loadedJCasClasses.add(aJCasCoverClass);
    return loadedJCasClasses.size() - 1;
  }
  
  /**
   * For a particular type, return true if that type should have run-time checking for use of fields
   * defined in the JCas Model which are not present in the CAS. If false, all fields
   * in the JCas must be in the CAS type system at instantiation time, or an exception is thrown;
   * this allows the runtime to skip this test.
   * <p>
   * This is reserved for future use; it currently always returns true.
   * @param fullyQualTypeName fully qualified type name
   * @return true if that type should have run-time checking for use of fields
   * defined in the JCas Model which are not present in the CAS. If false, all fields
   * in the JCas must be in the CAS type system at instantiation time, or an exception is thrown;
   * this allows the runtime to skip this test. 
   */
  public static boolean getFeatOkTst(String fullyQualTypeName) {
    return true;
  }  
  
  /**
   * Gets the number of cover classes that have been registered.
   * 
   * @return the number of registered JCas cover classes
   */
  public static synchronized int getNumberOfRegisteredClasses() {
    return loadedJCasClasses.size();
  }
  
  /**
   * Gets the JCas cover class for a given index.
   * 
   * @param aIndex the index
   * 
   * @return the JCas cover class that was assigned the value <code>aIndex</code> during its registration,
   *    <code>null</code> if none.
   */
  public static synchronized Class getClassForIndex(int aIndex) {
    if (aIndex >= 0 && aIndex < loadedJCasClasses.size())
      return (Class)loadedJCasClasses.get(aIndex);
    else
      return null;
  }
}
