/*
 * Copyright (c) 2003-2006 Erez Zadok
 * Copyright (c) 2003-2006 Charles P. Wright
 * Copyright (c) 2005-2006 Josef Sipek
 * Copyright (c) 2005      Arun M. Krishnakumar
 * Copyright (c) 2005-2006 David P. Quigley
 * Copyright (c) 2003-2004 Mohammad Nayyer Zubair
 * Copyright (c) 2003      Puja Gupta
 * Copyright (c) 2003      Harikesavan Krishnan
 * Copyright (c) 2003-2006 Stony Brook University
 * Copyright (c) 2003-2006 The Research Foundation of State University of New York
 *
 * For specific licensing information, see the COPYING file distributed with
 * this package.
 *
 * This Copyright notice must be kept intact and distributed with all sources.
 */
/*
 *  $Id: unionimap.c,v 1.9 2007/04/30 19:50:04 ezk Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "unionimap.h"

#if  !defined(HAVE_LIBUUID) || !defined(HAVE_UUID_UUID_H)
	#error Unionimap needs libuuid to compile
#endif


/**
 *	print_usage()
 *	Function used to print out usage information
 *	when an error is encountered
 */
void print_usage(void)
{
	fprintf(stderr,
		"unionimap version: $Id: unionimap.c,v 1.9 2007/04/30 19:50:04 ezk Exp $\n");
	fprintf(stderr, "Distributed with Unionfs " PACKAGE_VERSION "\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Usage: unionimap [-c] [-a ARG:] filename path\n");
	fprintf(stderr, " flag -c: create forward map with name filename\n");
	fprintf(stderr,
		" flag -a ARG: create reverse map with name filename and path and add it to forwardmap ARG\n");
	fprintf(stderr,
		" flag -d: if set once will dump the header of the map if twice it will also dump the contents\n");
	fprintf(stderr, " flag -h: prints this message then quits\n");
	fprintf(stderr,
		" please note that -c -a and -d are mutually exclusive\n");
}

int main(int argc, char **argv)
{
	int c = 0;		//Temp varliable to hold our character for the switch
	int cflag = 0, aflag = 0, dflag = 0;
	int err = 0;
	char *avalue = NULL;

	if (argc < 2) {
		print_usage();
		err = -EINVAL;
		goto out;
	}
	while ((c = getopt(argc, argv, "a:cd")) != -1) {
		switch (c) {
		case 'c':
			cflag = c;
			break;
		case 'a':
			aflag = 1;
			avalue = strdup(optarg);
			if (!avalue) {
				perror("strdup");
				exit(1);
			}
			break;
		case 'd':
			dflag++;
			break;
		case '?':
			if (isprint(optopt))
				fprintf(stderr, "Unknown option `-%c'.\n",
					optopt);
			else
				fprintf(stderr,
					"Unknown option character `\\x%x'.\n",
					optopt);
			print_usage();
			err = -EINVAL;
			goto out;
			break;
		default:
			print_usage();
			err = -EINVAL;
			goto out;
			break;
		}
	}
	if (optind == argc) {
		print_usage();
		err = -EINVAL;
		goto out;
	}
	if (!(cflag ^ aflag ^ dflag)) {
		print_usage();
		err = -EINVAL;
		goto out;
	}
	if (cflag) {
		err = unionfs_create_forwardmap(argv[optind]);
		if ( err ) {
			unlink(argv[optind]);
			fprintf(stderr, "Failed to create forward map %s: %s\n",
				argv[optind], strerror(errno));
		}
		goto out;
	} else if (aflag) {
		if (optind + 1 > argc) {
			print_usage();
			err = -EINVAL;
			goto out;
		}

		err = unionfs_create_reversemap(argv[optind], argv[optind + 1], avalue);
		if (err) {
			unlink(argv[optind]);
			fprintf(stderr, "Failed to create reverse map %s: %s\n",
				argv[optind], strerror(errno));
			goto out;
		}

	} else if (dflag) {
		err = unionfs_dump_map(argv[optind], dflag);
		if (err)
			goto out;
	}

      out:
	free(avalue);
	return err;
}

/*
 *
 * vim:shiftwidth=8
 * vim:tabstop=8
 *
 * For Emacs:
 * Local variables:
 * c-basic-offset: 8
 * c-comment-only-line-offset: 0
 * c-offsets-alist: ((statement-block-intro . +) (knr-argdecl-intro . 0)
 *              (substatement-open . 0) (label . 0) (statement-cont . +))
 * indent-tabs-mode: t
 * tab-width: 8
 * End:
 */
