/*
 * unity-webapps-binding-dispatcher.c
 * Copyright (C) Canonical LTD 2012
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <glib.h>

#include "npapi-headers/headers/npapi.h"
#include "npapi-headers/headers/npfunctions.h"

#include "unity-webapps-binding.h"
#include "unity-webapps-repository-binding.h"
#include "notify-binding.h"

#include "unity-webapps-binding-dispatcher.h"
#include "unity-npapi-common.h"
#include "unity-npapi-debug.h"


typedef char * CallableName;

typedef struct PluginCallableItem_t
{
  CallableName name;
  UnityWebAppsNpapiMethodPtr dispatch_func;
  
  // TODO add generic arg type verification
  
} PluginCallableItem;


#define DECLARE_BINDING(binding_function_prefix,binding_function_name) \
  {									\
    .name = #binding_function_name					\
  , .dispatch_func = (UnityWebAppsNpapiMethodPtr) binding_function_prefix##binding_function_name \
  }

#define DECLARE_UNITY_WEBAPPS_BINDING(binding_function_name) \
  DECLARE_BINDING(unity_webapps_binding_,binding_function_name)

#define DECLARE_NOTIFY_BINDING(binding_function_name) \
  DECLARE_BINDING(binding_notify_,binding_function_name)

static PluginCallableItem
plugin_callable_items [] = {
	
  DECLARE_UNITY_WEBAPPS_BINDING(context_new_lazy)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_prepare)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_new_sync)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_destroy)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_get_interest_id)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_get_name)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_get_domain)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(service_new)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(service_destroy_interest_for_context)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_add_icon)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_set_view_is_active)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_set_view_location)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_on_raise_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_on_close_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_set_preview_requested_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_add_application_actions)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_set_application_accept_data)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_remove_application_action)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_remove_application_actions)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_allow_domain)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_init)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_on_play_pause_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_on_previous_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_on_next_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_track)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_can_pause)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_can_play)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_can_go_next)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_can_go_previous)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_set_playback_state)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_get_can_pause)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_get_can_play)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_get_can_go_next)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_get_can_go_previous)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(music_player_get_playback_state)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(notification_show_notification)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_get_domain_allowed)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_get_domain_preauthorized)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_allow_domain)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_get_domain_dontask)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_is_integration_allowed)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(permissions_dontask_domain)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_set_count)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_clear_count)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_set_progress)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_clear_progress)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_set_urgent)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_add_action)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_remove_action)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_remove_actions)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_add_static_action)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(launcher_remove_static_actions)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_show_indicator)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_clear_indicator)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_clear_indicators)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_set_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_set_property)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_set_property_icon)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_add_action)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_get_presence)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(indicator_on_presence_changed_callback)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(context_set_homepage)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_new_default)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_prepare)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_resolve_url_as_json)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_get_resolved_application_status)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_get_userscript_contents)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_install_application)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_get_resolved_application_domain)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(application_repository_get_resolved_application_name)
  ,
  DECLARE_UNITY_WEBAPPS_BINDING(service_set_xid_for_browser_window_id)
  ,
  DECLARE_NOTIFY_BINDING(notification_init)
  ,
  DECLARE_NOTIFY_BINDING(notification_is_initted)
  ,
  DECLARE_NOTIFY_BINDING(notification_new)
  ,
  DECLARE_NOTIFY_BINDING(notification_show)
  ,
  DECLARE_NOTIFY_BINDING(notification_destroy)
};

static void for_each_callable_binding (bool (*f) (PluginCallableItem *));



////////////////////////////////////////////////////////////
// 
// public functions
// 
////////////////////////////////////////////////////////////

bool
webapps_binding_is_method_supported (const char * const name)
{
  bool isCallable = false;

  // TODO mmmh use a proper "type safe" name specific macro to avoid NULL comparaison
  //  on abstract type, same w/ strlen
  if (NULL == name || 0 == strlen (name))
    {
      return isCallable;
    }

  bool isfound (PluginCallableItem * callable)
  {
    // TODO should assert for NULL
    if (NULL == callable)
      {
	// TODO BAD
	return false;
      }

    // TODO check a safer version
    isCallable = (0 == g_strcmp0 (name, callable->name));

    // continue only is callable is false
    return ! isCallable;
  }
  
  NPAPI_LOG (INFO, "%s: %s", __func__, name);
  
  for_each_callable_binding (isfound);

  return isCallable;
}


NPVariant
webapps_binding_dispatch_method_call (NPP instance
				      , const char * const name
				      , int argcnt
				      , const NPVariant args[]
				      , void (*logger) (const char * const s))
{
  NPAPI_LOG (INFO, "webapps_binding_dispatch_method_call %s, count %d", name, argcnt);

  PluginCallableItem * item = NULL;

  // TODO mmmh use a proper "type safe" name specific macro to avoid NULL comparaison
  //  on abstract type, same w/ strlen
  if (NULL == name || 0 == strlen (name))
    {
      // TODO use exceptions
      NPVariant result;
      NULL_TO_NPVARIANT(result);
      return result;
    }

  bool findmethod (PluginCallableItem * currentItem)
  {
    // TODO should assert for NULL
    if (NULL == currentItem)
      {
	// TODO BAD
	return false;
      }

    // TODO check a safer version
    if (0 == g_strcmp0 (name, currentItem->name))
      {
	item = currentItem;
      }

    return item == NULL;
  }

  for_each_callable_binding (findmethod);

  if (NULL == item || NULL == item->dispatch_func)
    {
      // TODO exception
      NPVariant result;
      NULL_TO_NPVARIANT(result);

      return result;    	
    }

  // actuall call
  NPVariant
    result = (*item->dispatch_func) (instance, NULL, &args[0], argcnt);

  return result;
}



void webapps_binding_for_each (void (*f) (const char * const name, UnityWebAppsNpapiMethodPtr))
{
  g_return_if_fail (NULL != f);
  
  bool priv_for_each (PluginCallableItem * currentItem)
  {
    if (NULL != currentItem)
      {
	f (currentItem->name, currentItem->dispatch_func);
      }

    return true;
  }

  for_each_callable_binding (priv_for_each);
}



////////////////////////////////////////////////////////////
// 
// private
// 
////////////////////////////////////////////////////////////


/**
 * @param f function pointer. Gets a callable item for each registered element found.
 *          Should return false when the each look is supposed to be stopped (true otherwise).
 */
static void for_each_callable_binding (bool (*f) (PluginCallableItem *))
{
  g_return_if_fail (NULL != f);
  
  for (size_t i = 0
	 ; i < G_N_ELEMENTS(plugin_callable_items)
	 ; ++i)
    {
      if (false == f (&plugin_callable_items[i]))
        {
	  break;
        }
    }
}
