/*
 * Copyright © 2014 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SCREEN_STATE_HANDLER_
#define SCREEN_STATE_HANDLER_

#include "mir/input/event_filter.h"
#include "dbus_screen_observer.h"

#include <chrono>
#include <memory>
#include <mutex>

class DBusScreen;
class PowerdMediator;
enum class PowerStateChangeReason;
namespace usc { class Server; }

namespace mir
{
namespace time
{
class Alarm;
class Timer;
}
}

class ScreenStateHandler: public mir::input::EventFilter, public DBusScreenObserver
{
public:
    ScreenStateHandler(std::shared_ptr<usc::Server> const& server,
                       std::chrono::milliseconds power_off_timeout,
                       std::chrono::milliseconds dimmer_timeout);
    virtual ~ScreenStateHandler();

    //from EventFilter
    bool handle(MirEvent const& event) override;

    void enable_inactivity_timers(bool enable);
    void toggle_screen_power_mode(PowerStateChangeReason reason);

    void set_screen_power_mode(MirPowerMode mode, PowerStateChangeReason reason) override;
    void keep_display_on(bool on) override;
    void set_brightness(int brightness) override;
    void enable_auto_brightness(bool enable) override;
    void set_inactivity_timeouts(int power_off_timeout, int dimmer_timeout) override;
    
    void set_touch_visualization_enabled(bool enabled) override;

private:
    void set_screen_power_mode_l(MirPowerMode mode, PowerStateChangeReason reason);
    void configure_display_l(MirPowerMode mode, PowerStateChangeReason reason);

    void cancel_timers_l();
    void reset_timers_l();
    void enable_inactivity_timers_l(bool flag);

    void power_off_alarm_notification();
    void dimmer_alarm_notification();
    void long_press_alarm_notification();

    std::mutex guard;

    MirPowerMode current_power_mode;
    bool restart_timers;

    std::chrono::milliseconds power_off_timeout;
    std::chrono::milliseconds dimming_timeout;

    std::unique_ptr<PowerdMediator> powerd_mediator;
    std::shared_ptr<usc::Server> server;

    std::unique_ptr<mir::time::Alarm> power_off_alarm;
    std::unique_ptr<mir::time::Alarm> dimmer_alarm;

    std::unique_ptr<DBusScreen> dbus_screen;
};

#endif
