# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for unperish."""


import logging
import os
import StringIO
import sys
import unittest


import unperishlib


class StackTests(unittest.TestCase):

    def setUp(self):
        self.stack = unperishlib.Stack()

    def testIsInitiallyEmpty(self):
        self.failUnless(self.stack.is_empty())

    def testRaisesExceptionWhenEmptyAndTopCalled(self):
        self.failUnlessRaises(IndexError, self.stack.top)

    def testTopAfterPushReturnsTheRightValue(self):
        self.stack.push("pink")
        self.failUnlessEqual(self.stack.top(), "pink")

    def testPushPopOfOneValueReturnsIt(self):
        self.stack.push("pink")
        self.failUnlessEqual(self.stack.pop(), "pink")

    def testPushOfValueMakesStackNotBeEmpty(self):
        self.stack.push("pink")
        self.failIf(self.stack.is_empty())

    def testPushPopOfOneValueEmptiesStack(self):
        self.stack.push("pink")
        self.stack.pop()
        self.failUnless(self.stack.is_empty())

    def testPushPopOfTwoValueReturnsThemInReverseOrder(self):
        self.stack.push("pink")
        self.stack.push("pretty")
        returned = []
        returned.append(self.stack.pop())
        returned.append(self.stack.pop())
        self.failUnlessEqual(returned, ["pretty", "pink"])


class OperationTests(unittest.TestCase):

    def setUp(self):
        self.o = unperishlib.Operation()
        
    def tearDown(self):
        del self.o

    def testGetNameReturnsNone(self):
        self.failUnlessEqual(self.o.get_name(), None)

    def testGetRequiredOptionsReturnsEmptyList (self):
        self.failUnlessEqual(self.o.get_required_options(), [])

    def testGetProvidedOptionsReturnsEmptyList (self):
        self.failUnlessEqual(self.o.get_provided_options(), [])

    def testAddOptionsReturnsNone(self):
        self.failUnlessEqual(self.o.add_options(None), None)

    def testDoItReturnsNone(self):
        self.failUnlessEqual(self.o.do_it(None), None)


class PluginManagerTests(unittest.TestCase):

    def setUp(self):
        self.pm = unperishlib.PluginManager()
        
    def tearDown(self):
        del self.pm
        
    def testInitiallyIsNotLoaded(self):
        self.failIf(self.pm.has_loaded())
        
    def testInitiallyHasEmptyPluginDirectoryList(self):
        self.failUnlessEqual(self.pm.get_plugin_directories(), [])
        
    def testSetsPluginDirectoriesListCorrectly(self):
        list = ["pink", "pretty"]
        self.pm.set_plugin_directories(list)
        self.failUnlessEqual(self.pm.get_plugin_directories(), list)

    def testDoesNotComplainIfPluginDirectoriesDoNotExist(self):
        self.pm.set_plugin_directories(["this-dir-does-not-exist"])
        self.failUnlessEqual(self.pm.find_plugin_files(), [])

    def testFindsPluginFilesCorrectly(self):
        files = [os.path.join("testplugins", x) 
                 for x in os.listdir("testplugins")
                 if x.endswith(".py")]
        self.pm.set_plugin_directories(["testplugins"])
        self.failUnlessEqual(self.pm.find_plugin_files(), files)

    def testInitiallyHasEmptyOperationList(self):
        self.failUnlessEqual(self.pm.get_operations(), [])

    def testInitiallyHasEmptyPluginsList(self):
        self.failUnlessEqual(self.pm.get_plugins(), [])

    def testLoadPluginsFindsOperations(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.failIfEqual(self.pm.get_operations(), [])

    def testLoadPluginsFindsPlugins(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.failIfEqual(self.pm.get_plugins(), [])

    def testHasLoadedAfterLoadPlugins(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.failUnless(self.pm.has_loaded())

    def testHasNotLoadedAfterUnloadPlugins(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.pm.unload_plugins()
        self.failIf(self.pm.has_loaded())

    def testUnloadPluginsGetsRidOfOperations(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.pm.unload_plugins()
        self.failUnlessEqual(self.pm.get_operations(), [])

    def testUnloadPluginsGetsRidOfPlugins(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        self.pm.unload_plugins()
        self.failUnlessEqual(self.pm.get_plugins(), [])

    def testFindsNullPluginByName(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        oper = self.pm.get_operation("null")
        self.failUnlessEqual(oper.get_name(), "null")

    def testDoesNotFindPluginThatDoesNotExist(self):
        self.pm.set_plugin_directories(["testplugins"])
        self.pm.load_plugins()
        oper = self.pm.get_operation("invalid\0name")
        self.failUnlessEqual(oper, None)


class NullPluginTests(unittest.TestCase):

    def testProvidesNullOption(self):
        pm = unperishlib.PluginManager()
        pm.set_plugin_directories(["testplugins"])
        pm.load_plugins()
        oper = pm.get_operation("null")
        self.failUnless("null" in oper.get_provided_options())


class NeedNullPluginTests(unittest.TestCase):

    def testRequiresNullOption(self):
        pm = unperishlib.PluginManager()
        pm.set_plugin_directories(["testplugins"])
        pm.load_plugins()
        oper = pm.get_operation("neednull")
        self.failUnless("null" in oper.get_required_options())


class CommandLineParserTests(unittest.TestCase):

    def setUp(self):
        self.cli = unperishlib.CommandLineParser()
        
    def tearDown(self):
        del self.cli
        
    def testReturnsListOfOperationsCorrectly(self):
        options, opers = self.cli.parse_args(["pink", "pretty"])
        self.failUnlessEqual(opers, ["pink", "pretty"])

    def testDirectoryOptionHasDotAsDefault(self):
        options, opers = self.cli.parse_args([])
        self.failUnlessEqual(options.directory, ".")

    def testHandlesDirectoryOptionCorrectly(self):
        options, opers = self.cli.parse_args(["--directory=pink"])
        self.failUnlessEqual(options.directory, "pink")

    def testDoesNotIncludeInternalOptionsInHelpTextByDefault(self):
        self.failIf("Internal options:" in self.cli.format_help())

    def testDoesIncludeInternalOptionsInHelpTextWhenRequested(self):
        self.cli.show_internal_options()
        self.failUnless("Internal options:" in self.cli.format_help())

    def testDoesNotIncludeInternalOptionsInDashDashHelp(self):
        f = StringIO.StringIO()
        self.cli._parser.print_help(file=f)
        self.failIf("Internal options:" in f.getvalue())

    def testDoesIncludeInternalOptionsInDashDashHelpAll(self):
        f = StringIO.StringIO()
        self.cli.help_all(file=f)
        self.failUnless("Internal options:" in f.getvalue())


class UnknownOperationTests(unittest.TestCase):

    def testOperationNameIsInMessage(self):
        e = unperishlib.UnknownOperation("pink")
        self.failUnless("pink" in str(e))


class ApplicationTests(unittest.TestCase):

    def setUp(self):
        self.recorded = []
        self.app = unperishlib.Application(plugins=["testplugins"])
        self.real_run_operation = self.app.run_operation
        self.app.run_operation = self.record_operations
        
    def record_operations(self, options, oper):
        self.recorded.append(oper)
        self.real_run_operation(options, oper)

    def testSetsPluginDirsCorrectly(self):
        pm = self.app.get_plugin_manager()
        self.failUnlessEqual(pm.get_plugin_directories(), ["testplugins"])

    def testDoesNotLoadPluginManagerAutomatically(self):
        self.failIf(self.app.get_plugin_manager().has_loaded())

    def testNoNullOptionInitially(self):
        cli = self.app.get_command_line_parser()
        self.failIf(cli.has_option("--null"))

    def testAddsPluginOptions(self):
        options, _ = self.app.parse_args(["--null=null"])
        self.failUnlessEqual(options.null, "null")

    def testFindsNoMissingProvidersWhenNoneAreNeeded(self):
        options, _ = self.app.parse_args(["--null=null"])
        oper = self.app.get_plugin_manager().get_operation("neednull")
        self.failUnlessEqual(self.app.find_missing_providers(oper, options), 
                             (None, []))

    def testFindsProviderWhenOptionIsRequiredWithoutBeingExplicitlySet(self):
        options, _ = self.app.parse_args([])
        null = self.app.get_plugin_manager().get_operation("null")
        oper = self.app.get_plugin_manager().get_operation("neednull")
        self.failUnlessEqual(self.app.find_missing_providers(oper, options), 
                             ("null", [null]))

    def testReturnsEmptyListWhenNoProviderFoundForNullOption(self):
        options, _ = self.app.parse_args([])
        null = self.app.get_plugin_manager().get_operation("null")
        oper = self.app.get_plugin_manager().get_operation("neednull")
        null.provided_options = []
        self.failUnlessEqual(self.app.find_missing_providers(oper, options),
                             ("null", []))

    def testRunsNullWhenRequested(self):
        pm = self.app.get_plugin_manager()
        self.app.prepare_plugins()
        null = pm.get_operation("null")
        self.app.run_operation = self.record_operations
        self.app.run(["null"])
        self.failUnlessEqual(self.recorded, [null])

    def testRunsNullEvenWithoutExplicitlyLoadingPlugins(self):
        self.app.run(["null"])
        pm = self.app.get_plugin_manager()
        null = pm.get_operation("null")
        self.failUnlessEqual(self.recorded, [null])

    def testRunsNullAndNeednullWhenLatterRequested(self):
        self.app.run(["neednull"])
        self.failUnlessEqual([oper.get_name() for oper in self.recorded], 
                             ["null", "neednull"])

    def testRaisesExceptionWhenProviderNotFound(self):
        self.app.prepare_plugins()
        null = self.app.get_plugin_manager().get_operation("null")
        oper = self.app.get_plugin_manager().get_operation("neednull")
        null.provided_options = []
        self.failUnlessRaises(unperishlib.MissingOption,
                              self.app.run, ["neednull"])

    def testRaisesExceptionWhenOperationIsUnknown(self):
        self.failUnlessRaises(unperishlib.UnknownOperation,
                              self.app.run, ["pink"])

    def testRunsOperationOnlyOnce(self):
        self.app.prepare_plugins()
        null = self.app.get_plugin_manager().get_operation("null")
        self.app.run(["null", "null"])
        self.failUnlessEqual(null.count, 1)

    def testSetsVerboseFileToStdoutByDefault(self):
        self.failUnlessEqual(self.app.get_verbose_file(), sys.stdout)
        
    def testSetsVerboseFileToWhateverIfRequested(self):
        self.app.set_verbose_file("pink")
        self.failUnlessEqual(self.app.get_verbose_file(), "pink")

    def testDoesNotReportOperationsWhenVerboseIsNotUsed(self):
        f = StringIO.StringIO()
        self.app.set_verbose_file(f)
        self.app.prepare_plugins()
        self.app.run(["null"])
        self.failUnlessEqual(f.getvalue(), "")

    def testReportsOperationsWhenVerboseIsUsed(self):
        f = StringIO.StringIO()
        self.app.set_verbose_file(f)
        self.app.prepare_plugins()
        self.app.run(["--verbose", "null"])
        self.failUnlessEqual(f.getvalue(), "Executing null\n")


class CommandFailedTests(unittest.TestCase):

    def setUp(self):
        self.e = unperishlib.CommandFailed(["pink", "pretty"], 69, "black",
                                           "beautiful")

    def testCommandNameInMessage(self):
        self.failUnless("pink" in str(self.e))

    def testCommandArgInMessage(self):
        self.failUnless("pretty" in str(self.e))

    def testExitCodeInMessage(self):
        self.failUnless("69" in str(self.e))

    def testStdoutInMessage(self):
        self.failUnless("black" in str(self.e))

    def testStderrInMessage(self):
        self.failUnless("beautiful" in str(self.e))


class RunTests(unittest.TestCase):

    def testBinTrueSucceeds(self):
        self.failUnlessEqual(unperishlib.run(["/bin/true"]), "")
    
    def testBinFalseFails(self):
         self.failUnlessRaises(unperishlib.CommandFailed,
                               unperishlib.run, ["/bin/false"])
