/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//*****************************************************************************
//
// Class:	 UNSEventLogger
// Description:	 EventLogger for User Notification Service.
//
//*****************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "UNSEventLogger.h"
#include <string>

#define UNS_SERVICE_START				"Intel(R) User Notification Service started."
#define UNS_SERVICE_STOPPED				"Intel(R) User Notification Service stopped."
#define UNS_SUBSCRIPTION_ERROR			"Failed to subscribe to local Intel(R) AMT."
#define UNS_SUBSCRIPTION_SUCCESS		"Subscribed to local Intel(R) AMT."
#define UNS_GET_FQDN_ERROR				"Failed to resolve Intel(R) AMT fqdn."
#define UNS_CANCEL_SUBSCRIPTION_ERROR	"Failed to cancel alert subscription."
#define UNS_BIND_ERROR					"Failed to bind localhost."
#define UNS_SERVER_ERROR				"SOAP web server error."
#define UNS_GET_EAC_STATUS_ERROR		"Failed to get EAC Status."
#define UNS_GET_POSTURE_ERROR			"Failed to get posture."
#define UNS_GET_POSTURE_HASH_ERROR		"Failed to get posture hash."
#define UNS_WRITE_SHARED_DATA_ERROR		"Failed to write shared posture/hash."
#define UNS_HECI_COMMAND_ERROR			"HECI Command failed"
#define UNS_SET_AMT_STATUS_ERROR		"Failed to set AMT Status"

// Constructor

UNSEventLogger::UNSEventLogger()
{
	ResetFlags();
#ifdef _LINUX
	m_logger = new LinuxEventLog(EVENT_LOG_SOURCE_NAME);
#else
	m_logger = new WindowsEventLog(EVENT_LOG_LOG_NAME,EVENT_LOG_SOURCE_NAME,EVENT_LOG_CATEGORY_COUNT);
#endif
}

UNSEventLogger::~UNSEventLogger()
{
	delete m_logger;
}

// Log information event by id and category
void UNSEventLogger::LogInfoEvent(unsigned short	CategoryID,
			  	  unsigned long		EventID)
{
	Lock lock(m_StateSemaphore);
	if (m_logger != NULL) {
		m_logger->LogEvent(CategoryID , EventID, EVENTLOG_INFORMATION_TYPE);
	}
}

// Record service start message.
void UNSEventLogger::ServiceStart()
{
	DebugLog(UNS_SERVICE_START);
}

// Record service stop message.
void UNSEventLogger::ServiceStopped()
{
	DebugLog(UNS_SERVICE_STOPPED);
}

// Record subscription error message.
void UNSEventLogger::SubscriptionError()
{
	if (!m_SubscribeFail)
	{
		ResetFlags();
		ErrorLog(UNS_SUBSCRIPTION_ERROR);
		m_SubscribeFail = true;
	}
}

// Record subscription success message.
void UNSEventLogger::SubscriptionSuccess()
{
	if (!m_Subscribe)
	{
		ResetFlags();
		DebugLog(UNS_SUBSCRIPTION_SUCCESS);
		m_Subscribe = true;
	}
}

// Record get net params (FQDN) error message.
void UNSEventLogger::GetFqdnFailure()
{
	if (!m_GetFqdn)
	{
		ResetFlags();
		ErrorLog(UNS_GET_FQDN_ERROR);
		m_GetFqdn = true;
	}
}

// Record bind error message.
void UNSEventLogger::BindError()
{
	if (!m_Bind)
	{
		ResetFlags();
		ErrorLog(UNS_BIND_ERROR);
		m_Bind = true;
	}
}

// Recored failed to cancal alert subscription.
void UNSEventLogger::CancelSubscriptionError()
{
	if (!m_Cancel)
	{
		ResetFlags();
		WarningLog(UNS_CANCEL_SUBSCRIPTION_ERROR);
		m_Cancel = true;
	}
}

void UNSEventLogger::GetEACStatusFailedError()
{
	if (!m_GetEACStatus)
	{
		ResetFlags();
		WarningLog(UNS_GET_EAC_STATUS_ERROR);
		m_GetEACStatus = true;
	}
}

void UNSEventLogger::GetPostureFailedError()
{
	if (!m_GetPosture)
	{
		ResetFlags();
		ErrorLog(UNS_GET_POSTURE_ERROR);
		m_GetPosture = true;
	}
}

void UNSEventLogger::GetPostureHashFailedError()
{
	if (!m_GetPostureHash)
	{
		ResetFlags();
		ErrorLog(UNS_GET_POSTURE_HASH_ERROR);
		m_GetPostureHash = true;
	}
}


void UNSEventLogger::WriteSharedDataError()
{
	if (!m_WriteSharedData)
	{
		ResetFlags();
		ErrorLog(UNS_WRITE_SHARED_DATA_ERROR);
		m_WriteSharedData = true;
	}
}

void UNSEventLogger::HECICommandFailedError()
{
	if (!m_HECICommandFailed)
	{
		ResetFlags();
		ErrorLog(UNS_HECI_COMMAND_ERROR);
		m_HECICommandFailed = true;
	}
}

void UNSEventLogger::SetAMTStatusFailedError()
{
	if (!m_SetAMTStatusFailed)
	{
		ResetFlags();
		ErrorLog(UNS_SET_AMT_STATUS_ERROR);
		m_SetAMTStatusFailed = true;
	}
}


// Record Events for debugging.
void UNSEventLogger::DebugLog(const char *message)
{
	Lock lock(m_StateSemaphore);
	char szBuf[1024];
	sprintf(szBuf, "%.1014s\n", message);
	const char *lpStrings[] = {szBuf};
	unsigned int NumOfStrings	= 1;
	if (m_logger != NULL) {
		m_logger->LogEvent(UNS_GENERAL, TEXT_MESSAGE, EVENTLOG_INFORMATION_TYPE, lpStrings,NumOfStrings, 						NULL, 0);
	}
#ifndef NO_STDOUT_DEBUG_LOG
	printf("%s", szBuf);
#endif
}


// Record warning events
void UNSEventLogger::WarningLog(const char *message)
{
	Lock lock(m_StateSemaphore);
	char szBuf[1024];
	sprintf(szBuf, " [UNS] %.1014s\n", message);
	const char *lpStrings[] = {szBuf};
	unsigned int NumOfStrings	= 1;
	if (m_logger != NULL) {
		m_logger->LogEvent(UNS_GENERAL, UNS_WARNING_MESSAGE, EVENTLOG_WARNING_TYPE, lpStrings,NumOfStrings,
			NULL, 0);
	}
#ifndef NO_STDOUT_DEBUG_LOG
	printf("%s", szBuf);
#endif
}

// Record error events
void UNSEventLogger::ErrorLog(const char *message)
{
	Lock lock(m_StateSemaphore);
	char szBuf[1024];
	sprintf(szBuf, " [UNS] %.1014s\n", message);
	const char *lpStrings[] = {szBuf};
	unsigned int NumOfStrings	= 1;
	if (m_logger != NULL) {
		m_logger->LogEvent(UNS_GENERAL, UNS_ERROR_MESSAGE, EVENTLOG_ERROR_TYPE, lpStrings,NumOfStrings, NULL, 0);
	}
#ifndef NO_STDOUT_DEBUG_LOG
	printf("%s", szBuf);
#endif
}

// Reset all flags
void UNSEventLogger::ResetFlags()
{
	Lock lock(m_StateSemaphore);
	m_SubscribeFail = false;
	m_Subscribe		= false;
	m_GetFqdn		= false;
	m_Bind			= false;
	m_Cancel		= false;
	m_GetEACStatus	= false;
	m_GetPosture	= false;
	m_GetPostureHash = false;
	m_WriteSharedData = false;
	m_HECICommandFailed = false;
	m_SetAMTStatusFailed = false;
}

