/*
 * Copyright (c) 1988, Julian Onions <jpo@cs.nott.ac.uk>
 * Nottingham University 1987.
 *
 * Portions:
 * Copyright (c) 1995, Kazuhiro Fukumura <kazu-f@po.iijnet.or.jp>
 *
 * This source may be freely distributed, however I would be interested
 * in any changes that are made.
 *
 * This driver takes packets off the IP i/f and hands them up to a
 * user process to have it's wicked way with. This driver has it's
 * roots in a similar driver written by Phil Cockcroft (formerly) at
 * UCL. This driver is based much more on read/write/select mode of
 * operation though.
 * 
 * NOTE:
 * Linux device driver for the IP tunneling device written by Kazuhiro Fukumura.
 * Much of this driver was taken from Julian Onions' tunnel device driver,
 * so it bears his copyright.
 */

/*
 * USERLINK: userlink network driver for Linux.
 *
 * Modifies:
 *    96/01/01
 *	- dynamic allocation manabe@papilio.tutics.tut.ac.jp.
 *    96/01/14: 0.3
 *	- fix close problem manabe@papilio.tutics.tut.ac.jp.
 *    96/02/13: 0.5
 *	- compile with 1.2.x kernel manabe@papilio.tutics.tut.ac.jp.
 *    96/02/14: 0.6
 *	- work with 1.2.x kernel manabe@papilio.tutics.tut.ac.jp.
 *    96/02/19: 0.7
 *	- auto-alloc in register_chrdev() manabe@papilio.tutics.tut.ac.jp.
 *    96/05/01: 0.8
 *	- dequeue all non-sent skb manabe@papilio.tutics.tut.ac.jp.
 *    96/05/20: 0.9
 *	- use nqueue in old kernels manabe@papilio.tutics.tut.ac.jp.
 *    96/11/22: 0.10
 *	- count alloc_skb error as dropping packet
 *	- fix typo in #if LINUX_VERSION_CODE check
 *					shimaz-n@jed.uec.ac.jp.
 *    96/12/03: 0.11
 *	- removed 1.[23].x support code, added 2.1.x support code, instead
 *					manabe@papilio.tutics.tut.ac.jp.
 *    96/12/09: 0.90
 *	- added 'BASE' encoding mode based on
 *	  "New User Link Network Device Driver Draft Specification
 *	                  v0.1 draft, Nov 28 1996, Shimazaki Naoto"
 *					manabe@papilio.tutics.tut.ac.jp.
 *    97/01/21: 0.91
 *	- don't access to user space directly (fop_ul_write)
 *	  it causes segmentation violation in kernel space
 *					manabe@papilio.tutics.tut.ac.jp.
 *    97/04/20: 0.92
 *	- support new 2.1.x kernel
 *					yosshy@jedi.seg.kobe-u.ac.jp.
 *					manabe@papilio.ics.tut.ac.jp.
 *
 *    98/02/12: 0.96-2.0.0
 *	- call dev_close() from fop_ul_close()
 *					manabe@amitaj.or.jp.
 *
 * This program is based on Kazuhiro Fukumura's IP tunneling device
 * driver which is derived from Julian Onions' tunnel device driver.
 *
 */

static const char *version = VERSION"-2";

#include <linux/version.h>

#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>

#include <linux/in.h>
#include <linux/errno.h>
#include <linux/major.h>

#include <linux/netdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>

typedef unsigned char	u_int8_t;
typedef unsigned short int	u_int16_t;
typedef unsigned int	u_int32_t;

#include <linux/if_userlink.h>

#if LINUX_VERSION_CODE < 0x20100
#include <asm/segment.h>
#include <linux/mm.h>
#else
#include <linux/poll.h>
#include <asm/uaccess.h>
#endif

/*
#define	DEBUG
*/

#ifdef	DEBUG
static int prevjif=0;
#define	JIFDEBUG(a)\
{\
    printk("ul@%s: %ld(%ld)\n", a, jiffies, jiffies - prevjif);\
    prevjif=jiffies;\
}
#else
#define	JIFDEBUG(a)
#endif

#define	UL_MTU	1500
#define	UL_MAX_MTU	2000
#define	UL_NAME	"userlink"
#define	MAX_NQUEUE	300

static int ul_major=0;

static struct ullink {
    char name[IFNAMSIZ];
    __u32 minor;		/* minor number of char dev */
    __u32 rx_packets;		/* received packets */
    __u32 tx_packets;		/* transmitted packets */
    __u32 tx_errors;
    __u32 tx_dropped;
    __u32 rx_dropped;
    struct wait_queue *wait;
    struct device *dev;		/* pointer to network device structure */
    struct sk_buff_head	queue;
    struct ullink *next;
    __u16 max_mtu;
    __u8 version;		/* frame type */
} *ullHead;

#if 0
static struct ullink *
minor2ullink(int minor)
{
    struct ullink *ullp;

    for (ullp = ullHead; ullp != NULL; ullp = ullp->next)
	if (ullp->minor == minor) return(ullp);
    return(NULL);
}
#endif

static struct enet_statistics *
net_ul_stats(struct device *dev)
{
    struct ullink *ullp = (struct ullink *)dev->priv;
    static struct enet_statistics stats;

    stats.rx_packets = ullp->rx_packets;
    stats.tx_packets = ullp->tx_packets;
    stats.tx_errors = ullp->tx_errors;
    stats.tx_dropped = ullp->tx_dropped;
    stats.rx_dropped = ullp->rx_dropped;
    return(&stats);
}

static int
net_ul_dummy(struct device *dev)
{
    return(0);
}

static int
net_ul_header(struct sk_buff *skb, struct device *dev,
	      unsigned short type, void *daddr, void *saddr,
	      unsigned len)
{
    struct ullink *ullp = (struct ullink *)dev->priv;
    struct ul_fs_header_base *ulhp;

    if (ullp->version == UL_V_NONE) return(0);
    ulhp = (struct ul_fs_header_base *)
	skb_push(skb, sizeof(struct ul_fs_header_base));
    memset(ulhp, 0, sizeof(struct ul_fs_header_base));
    ulhp->version = ullp->version;
    ulhp->protocol = htons(type);
    return(sizeof(struct ul_fs_header_base));
}

static int
#if LINUX_VERSION_CODE < 0x20100
net_ul_rebuild(void *buff, struct device *dev, unsigned long raddr,
	       struct sk_buff *skb)
#else
net_ul_rebuild(struct sk_buff *skb)
#endif
{
    return(0);
}

static int
net_ul_tx(struct sk_buff *skb, struct device *dev)
{
    struct ullink *ullp = (struct ullink *)dev->priv;

JIFDEBUG("ul@net_ul_tx");
#ifdef DEBUG
printk("ul@net_ul_tx: ullp=%x\n", ullp);
#endif
    if (dev->tbusy) return(-EBUSY);

    if (skb == NULL) {
	dev_tint(dev);
	return(0);
    }

    if (skb->len > dev->mtu) {
	printk("%s: packet too big, %d.\n", dev->name, (int)skb->len);
	ullp->tx_dropped++;
	return(0);
    }
    if (skb_queue_len(&ullp->queue) == MAX_NQUEUE
	&& (skb = skb_dequeue(&ullp->queue)) != NULL) {
	dev_kfree_skb(skb, FREE_WRITE);
	ullp->tx_errors++;
    }
    if (skb_queue_len(&ullp->queue) < MAX_NQUEUE) {
#ifdef DEBUG
printk("net_ul_tx:queue_len=%d\n", skb_queue_len(&ullp->queue));
#endif
	skb_queue_tail(&ullp->queue, skb);
/*	ullp->tx_packets++;*/
/*	wake_up(&ullp->wait);*/
#ifdef DEBUG
printk("net_ul_tx:timeout0=%d\n", current->timeout);
#endif
	wake_up_interruptible(&ullp->wait);
#ifdef DEBUG
printk("net_ul_tx:timeout1=%d\n", current->timeout);
#endif
	return(0);
    }
    return(0);
}

static int
net_ul_ioctl(struct device *dev, struct ifreq *ifr, int cmd)
{
    struct ullink *ullp = (struct ullink *)dev->priv;
    struct ul_ifru_data uld;
    int ret;

    if (cmd != SIOCDEVPRIVATE) return(-EINVAL);
    if ((ret = verify_area(VERIFY_WRITE, ifr->ifr_data,
			   sizeof(struct ul_ifru_data))) != 0)
	return(ret);
#if LINUX_VERSION_CODE < 0x20100
    memcpy_fromfs(&uld, ifr->ifr_data, sizeof(struct ul_ifru_data));
#else
    copy_from_user(&uld, ifr->ifr_data, sizeof(struct ul_ifru_data));
#endif
    switch(uld.command) {
    case UL_IOCGET:
	uld.version = ullp->version;
	uld.max_mtu = ullp->max_mtu;
#if LINUX_VERSION_CODE < 0x20100
	memcpy_tofs(ifr->ifr_data, &uld, sizeof(struct ul_ifru_data));
#else
	copy_to_user(ifr->ifr_data, &uld, sizeof(struct ul_ifru_data));
#endif
	ret = 0;
	break;
    case UL_IOCSET:
	ullp->version = uld.version;
	if (uld.max_mtu) ullp->max_mtu = uld.max_mtu;
	if (uld.version == UL_V_NONE)
	    dev->hard_header_len = 0;
	else
	    dev->hard_header_len = sizeof(struct ul_fs_header_base);
	ret = 0;
	break;
    default:
	ret = -EINVAL;
    }
    return(ret);
}

static int
net_ul_mtu(struct device *dev, int new_mtu)
{
    struct ullink *ullp = (struct ullink *)dev->priv;

    if ((new_mtu < sizeof(struct ul_fs_header_base)) ||
	(new_mtu > ullp->max_mtu)) return(-EINVAL);
    dev->mtu = new_mtu;
    return(0);
}

/*
 * file operations
 */

static int
fop_ul_open(struct inode *inode, struct file *file)
{
    int i;
    register int minor = MINOR(inode->i_rdev);
    struct device *dev;
    struct ullink *ullp;

    for (ullp = ullHead; ullp != NULL; ullp = ullp->next) {
	if (ullp->minor == minor) return(-EBUSY);
    }
    ullp = kmalloc(sizeof(struct ullink), GFP_KERNEL);
    memset(ullp, 0, sizeof(struct ullink));
    if (ullHead) {
	ullp->next = ullHead;
	ullHead = ullp;
    } else {
	ullHead = ullp;
	ullp->next = NULL;
    }
    ullp->minor = minor;
    sprintf(ullp->name, "ul%d", minor);
    dev = ullp->dev = kmalloc(sizeof(struct device), GFP_KERNEL);
    memset(dev, 0, sizeof(struct device));
    skb_queue_head_init(&ullp->queue);
    ullp->max_mtu = UL_MAX_MTU;
    dev->mtu = UL_MTU;
    dev->init = net_ul_dummy;
    dev->open = net_ul_dummy;
    dev->stop = net_ul_dummy;
    dev->do_ioctl = net_ul_ioctl;
    dev->hard_start_xmit = net_ul_tx;
    dev->hard_header = net_ul_header;
    dev->rebuild_header = net_ul_rebuild;
    dev->change_mtu = net_ul_mtu;
    dev->type = ARPHRD_PPP;
    dev->get_stats = net_ul_stats;
    dev->priv = file->private_data = (void *)ullp;

    for (i = 0; i < DEV_NUMBUFFS; i++) {
	skb_queue_head_init(&dev->buffs[i]);
    }
    
    dev->flags = IFF_POINTOPOINT;
    dev->family = AF_INET;
/*    dev->pa_alen = sizeof(unsigned long);*/
    dev->pa_alen = sizeof(dev->pa_addr);
    ullp->dev->name = ullp->name;
    if ((i = register_netdev(ullp->dev))) {
	printk("%s: allocation failed.\n", dev->name);
	return(i);
    }
    dev->flags |= IFF_RUNNING | IFF_POINTOPOINT;
    dev->start = 1;
    dev->tbusy = 0;

    MOD_INC_USE_COUNT;
    return(0);
}

static int
fop_ul_close(struct inode *inode, struct file * file)
{
    struct device	*dev;
    struct sk_buff	*skb;
    struct ullink *ullp, *ull0;

    ullp = (struct ullink *)file->private_data;
    dev = ullp->dev;
    /*
     * junk all pending output
     */

    dev->start = 0;
    dev->tbusy = 1;
    if (dev->flags & IFF_UP) {
	dev->flags &= ~(IFF_UP | IFF_RUNNING);
	dev_close(dev);
    }
    while((skb = skb_dequeue(&ullp->queue)) != NULL) {
	dev_kfree_skb(skb, FREE_WRITE);
    }
    unregister_netdev(dev);

    if (ullp == ullHead) ullHead = ullp->next;
    else for (ull0 = ullHead; ull0 != NULL; ull0 = ull0->next) {
	if (ull0->next == ullp) {
	    ull0->next = ullp->next;
	    break;
	}
    }
    kfree(dev);
    kfree(ullp);
    MOD_DEC_USE_COUNT;
    return(0);
}

#if LINUX_VERSION_CODE < 0x20100
static int
fop_ul_write(struct inode *inode, struct file *file,
	     const char *buffer, int count)
#else
static long
fop_ul_write(struct inode *inode, struct file *file,
	     const char *buffer, unsigned long count)
#endif
{
    unsigned long xsize;
    struct ullink *ullp;
    struct sk_buff *skb;

    ullp = (struct ullink *)file->private_data;

    xsize = (ullp->version == UL_V_NONE)
	? count: count - sizeof(struct ul_fs_header_base);
    if (xsize < 0) return(-EINVAL);
    if ((skb = dev_alloc_skb(xsize)) == NULL) {
	printk("%s: memory squeeze, dropping packet.\n", ullp->name);
	ullp->rx_dropped++;
	return 0;
    }
    skb->dev = ullp->dev;
    if (ullp->version == UL_V_NONE) {
	skb->protocol = htons(ETH_P_IP);
    } else {
	struct ul_fs_header_base ulhb;

#if LINUX_VERSION_CODE < 0x20100
	memcpy_fromfs(&ulhb, buffer, sizeof(ulhb));
#else
	copy_from_user(&ulhb, buffer, sizeof(ulhb));
#endif
	skb->protocol = ulhb.protocol;
	buffer += sizeof(struct ul_fs_header_base);
    }
#if LINUX_VERSION_CODE < 0x20100
    memcpy_fromfs(skb_put(skb, xsize), buffer, xsize);
#else
    memcpy(skb_put(skb, xsize), buffer, xsize);
#endif
    skb->mac.raw=skb->data;

    netif_rx(skb);
    ullp->rx_packets++;
    JIFDEBUG("write_ret");
    return(count);
}

#if LINUX_VERSION_CODE < 0x20100
static int
fop_ul_read(struct inode *inode, struct file *file,
	    char *buffer, int count)
#else
static long
fop_ul_read(struct inode *inode, struct file *file,
	    char *buffer, unsigned long count)
#endif
{
    struct ullink *ullp;
    struct sk_buff	*skb;

    ullp = (struct ullink *)file->private_data;
    if (count < 0) return(-EINVAL);
    if ((skb = skb_dequeue(&ullp->queue)) == NULL) return(0);

    ullp->tx_packets++;

    count = (skb->len < count) ? skb->len : count;
#if LINUX_VERSION_CODE < 0x20100
    memcpy_tofs(buffer, skb->data, count);
#else
    memcpy(buffer, skb->data, count);
#endif

    dev_kfree_skb(skb, FREE_WRITE);
    JIFDEBUG("read_ret");
    return(count);
}

#if LINUX_VERSION_CODE < 0x20100
#define	poll_wait(a,b)	select_wait(a,b)
static int
fop_ul_poll(struct inode *inode, struct file *file,
	      int sel_type, select_table * wait)
{
    struct ullink *ullp;

    ullp = (struct ullink *)file->private_data;

    switch (sel_type) {
    case SEL_OUT:
	return(1);
    default:
	return(0);
    case SEL_IN:
	if (skb_queue_len(&ullp->queue) > 0) return(1);
	poll_wait(&ullp->wait, wait);
	return(0);
    }
}
#else
static unsigned int
fop_ul_poll(struct file *file, poll_table *wait)
{
    struct ullink *ullp;
    unsigned ret = POLLOUT|POLLWRNORM;

    ullp = (struct ullink *)file->private_data;

    JIFDEBUG("poll");
    poll_wait(&ullp->wait, wait);
    if (skb_queue_len(&ullp->queue) > 0) ret |= POLLIN|POLLRDNORM;
#ifdef	DEBUG
    printk("ul@poll: ret=%x\n", ret);
#endif
    return(ret);
}
#endif

static struct file_operations fops_ul = {
	NULL,		/* seek */
	fop_ul_read,
	fop_ul_write,
	NULL, 		/* readdir */
	fop_ul_poll,
	NULL,		/* ioctl */
	NULL,		/* mmap */
	fop_ul_open,
	fop_ul_close,
};

#ifdef MODULE
#define userlink_init init_module

void
cleanup_module(void)
{

    if (MOD_IN_USE) {
	printk(UL_NAME": device busy.\n");
	return;
    } else {
	unregister_chrdev(ul_major, UL_NAME);
/*	UL_DEBUG(1,printk(UL_NAME": successfully unregistered.\n"));*/
    }
}

#endif /* MODULE */

int
userlink_init(void)
{
    if ((ul_major = register_chrdev(ul_major, UL_NAME, &fops_ul)) < 0) {
	printk(UL_NAME": registration failed.\n");
	return(-EIO);
    } else {
	printk(UL_NAME" version %s registered major %d\n",
	       version, ul_major);
	return(0);
    }
}

/*
 * Local variables:
 * compile-command: "gcc -DMODULE -DMODVERSIONS -D__KERNEL__ -Wall -Wstrict-prototypes -O2 -g -fomit-frame-pointer -pipe -m486 -c userlink.c"
 * End:
 */
