/* usplash
 *
 * ubuntustudio-theme.c - definition of ubuntustudio theme
 *
 * Copyright © 2007 Tonic Artos <ghatanothoa@gmail.com>
 *           © 2007 Luke Yelavich <themuso@themuso.org>
 *           © 2006 Dennis Kaarsemaker <dennis@kaarsemaker.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <math.h>
#include <stdio.h>
#include <usplash-theme.h>
#include <usplash_backend.h>

#include "ubuntustudio-theme-vars.h"


#define TRUE 1
#define FALSE 0

extern struct usplash_pixmap pixmap_usplash_640_400;
extern struct usplash_pixmap pixmap_usplash_640_480;
extern struct usplash_pixmap pixmap_usplash_800_600;
extern struct usplash_pixmap pixmap_usplash_1024_768;
extern struct usplash_pixmap pixmap_usplash_1365_768;

extern struct usplash_pixmap pixmap_throbber_back_alt;
extern struct usplash_pixmap pixmap_throbber_fore_alt;
extern struct usplash_pixmap pixmap_throbber_back;
extern struct usplash_pixmap pixmap_throbber_fore;


struct pulse_rec {
	int position; /** current position - start at zero **/
	int bar_size; /** current size of pulse bar - start at current progress **/
	int region_size; /** most likely progress bar width **/
	float angle; /** angle in pi radians **/
	int period; /** ticks to complete a revolution **/
	int num_ticks; /** number of ticks pulse has run **/
};

struct usplash_theme usplash_theme_640_480;
struct usplash_theme usplash_theme_800_600;
struct usplash_theme usplash_theme_1024_768;
struct usplash_theme usplash_theme_1365_768;

void t_init(struct usplash_theme* theme);
void t_clear_progressbar(struct usplash_theme* theme);
void t_update_progress(struct usplash_theme* theme, int percentage);
void t_animate_step(struct usplash_theme* theme, int pulsating);

void draw_progress (struct usplash_theme * theme, int percent_progress, int boot_changed, int booting_up);
int smooth_progress (struct usplash_theme* theme, int boot_changed);
void harmonic_pulse (struct pulse_rec *pulse);
void draw_pulse (struct pulse_rec *pulse, struct usplash_theme* theme);

int progress_complete;

/*******************************
 *
 * THEME DEFINITIONS
 *
 *******************************/

struct usplash_theme usplash_theme = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_640_480,
    .ratio = USPLASH_16_9,

	/* Background and font */
	.pixmap = &pixmap_usplash_640_400,

	/* Palette indexes */
	.background = USPLASH_BACKGROUND,
  	.progressbar_background = USPLASH_PROGRESSBAR_BACKGROUND,
  	.progressbar_foreground = USPLASH_PROGRESSBAR_FOREGROUND,
	.text_background = USPLASH_TEXT_BACKGROUND,
	.text_foreground = USPLASH_TEXT_FOREGROUND,
	.text_success = USPLASH_TEXT_SUCCESS,
	.text_failure = USPLASH_TEXT_FAILURE,

	/* Progress bar position and size in pixels */
  	.progressbar_x = USPLASH_640_400_PROGRESSBAR_X,
  	.progressbar_y = USPLASH_640_400_PROGRESSBAR_Y,
  	.progressbar_width = USPLASH_640_400_PROGRESSBAR_WIDTH,
  	.progressbar_height = USPLASH_640_400_PROGRESSBAR_HEIGHT,

	/* Text box position and size in pixels */
  	.text_x = USPLASH_640_400_TEXT_X,
  	.text_y = USPLASH_640_400_TEXT_Y,
  	.text_width = USPLASH_640_400_TEXT_WIDTH,
  	.text_height = USPLASH_640_400_TEXT_HEIGHT,

	/* Text details */
  	.line_height = USPLASH_640_400_LINE_HEIGHT,
  	.line_length = USPLASH_640_400_LINE_LENGTH,
  	.status_width = USPLASH_640_400_STATUS_WIDTH,

    /* Functions */
    .init = t_init,
    .clear_progressbar = t_clear_progressbar,
    .draw_progressbar = t_update_progress,
    .animate_step = t_animate_step,
};

struct usplash_theme usplash_theme_640_480 = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_800_600,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_640_480,

	/* Palette indexes */
	.background = USPLASH_BACKGROUND,
	.progressbar_background = USPLASH_PROGRESSBAR_BACKGROUND,
	.progressbar_foreground = USPLASH_PROGRESSBAR_FOREGROUND, 
	.text_background = USPLASH_TEXT_BACKGROUND,
	.text_foreground = USPLASH_TEXT_FOREGROUND,
	.text_success = USPLASH_TEXT_SUCCESS,
	.text_failure = USPLASH_TEXT_FAILURE,

	/* Progress bar position and size in pixels */
	.progressbar_x = USPLASH_640_480_PROGRESSBAR_X,
	.progressbar_y = USPLASH_640_480_PROGRESSBAR_Y,
	.progressbar_width = USPLASH_640_480_PROGRESSBAR_WIDTH,
	.progressbar_height = USPLASH_640_480_PROGRESSBAR_HEIGHT,

	/* Text box position and size in pixels */
	.text_x = USPLASH_640_480_TEXT_X,
	.text_y = USPLASH_640_480_TEXT_Y,
	.text_width = USPLASH_640_480_TEXT_WIDTH,
	.text_height = USPLASH_640_480_TEXT_HEIGHT,

	/* Text details */
	.line_height = USPLASH_640_480_LINE_HEIGHT,
	.line_length = USPLASH_640_480_LINE_LENGTH,
	.status_width = USPLASH_640_480_STATUS_WIDTH,

    /* Functions */
    .init = t_init,
    .clear_progressbar = t_clear_progressbar,
    .draw_progressbar = t_update_progress,
    .animate_step = t_animate_step,
};

struct usplash_theme usplash_theme_800_600 = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_1024_768,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_800_600,

	/* Palette indexes */
	.background = USPLASH_BACKGROUND,
	.progressbar_background = USPLASH_PROGRESSBAR_BACKGROUND,
	.progressbar_foreground = USPLASH_PROGRESSBAR_FOREGROUND, 
	.text_background = USPLASH_TEXT_BACKGROUND,
	.text_foreground = USPLASH_TEXT_FOREGROUND,
	.text_success = USPLASH_TEXT_SUCCESS,
	.text_failure = USPLASH_TEXT_FAILURE,

	/* Progress bar position and size in pixels */
	.progressbar_x = USPLASH_800_600_PROGRESSBAR_X,
	.progressbar_y = USPLASH_800_600_PROGRESSBAR_Y,
	.progressbar_width = USPLASH_800_600_PROGRESSBAR_WIDTH,
	.progressbar_height = USPLASH_800_600_PROGRESSBAR_HEIGHT,

	/* Text box position and size in pixels */
	.text_x = USPLASH_800_600_TEXT_X,
	.text_y = USPLASH_800_600_TEXT_Y,
	.text_width = USPLASH_800_600_TEXT_WIDTH,
	.text_height = USPLASH_800_600_TEXT_HEIGHT,

	/* Text details */
	.line_height  = USPLASH_800_600_LINE_HEIGHT,
	.line_length  = USPLASH_800_600_LINE_LENGTH,
	.status_width = USPLASH_800_600_STATUS_WIDTH,

    /* Functions */
    .init = t_init,
    .clear_progressbar = t_clear_progressbar,
    .draw_progressbar = t_update_progress,
    .animate_step = t_animate_step,
};

struct usplash_theme usplash_theme_1024_768 = {
	.version = THEME_VERSION,
    .next = &usplash_theme_1365_768,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_1024_768,

	/* Palette indexes */
	.background             = USPLASH_BACKGROUND,
	.progressbar_background = USPLASH_PROGRESSBAR_BACKGROUND,
	.progressbar_foreground = USPLASH_PROGRESSBAR_FOREGROUND, 
	.text_background        = USPLASH_TEXT_BACKGROUND,
	.text_foreground        = USPLASH_TEXT_FOREGROUND,
	.text_success           = USPLASH_TEXT_SUCCESS,
	.text_failure           = USPLASH_TEXT_FAILURE,

	/* Progress bar position and size in pixels */
	.progressbar_x      = USPLASH_1024_768_PROGRESSBAR_X,
	.progressbar_y      = USPLASH_1024_768_PROGRESSBAR_Y,
	.progressbar_width  = USPLASH_1024_768_PROGRESSBAR_WIDTH,
	.progressbar_height = USPLASH_1024_768_PROGRESSBAR_HEIGHT,

	/* Text box position and size in pixels */
	.text_x      = USPLASH_1024_768_TEXT_X,
	.text_y      = USPLASH_1024_768_TEXT_Y,
	.text_width  = USPLASH_1024_768_TEXT_WIDTH,
	.text_height = USPLASH_1024_768_TEXT_HEIGHT,

	/* Text details */
	.line_height  = USPLASH_1024_768_LINE_HEIGHT,
	.line_length  = USPLASH_1024_768_LINE_LENGTH,
	.status_width = USPLASH_1024_768_STATUS_WIDTH,

    /* Functions */
    .init = t_init,
    .clear_progressbar = t_clear_progressbar,
    .draw_progressbar = t_update_progress,
    .animate_step = t_animate_step,
};

struct usplash_theme usplash_theme_1365_768 = {
	.version = THEME_VERSION,
    .next = NULL,
    .ratio = USPLASH_16_9,

	/* Background and font */
	.pixmap = &pixmap_usplash_1365_768,

	/* Palette indexes */
	.background             = USPLASH_BACKGROUND,
	.progressbar_background = USPLASH_PROGRESSBAR_BACKGROUND,
	.progressbar_foreground = USPLASH_PROGRESSBAR_FOREGROUND, 
	.text_background        = USPLASH_TEXT_BACKGROUND,
	.text_foreground        = USPLASH_TEXT_FOREGROUND,
	.text_success           = USPLASH_TEXT_SUCCESS,
	.text_failure           = USPLASH_TEXT_FAILURE,

	/* Progress bar position and size in pixels */
	.progressbar_x      = USPLASH_1365_768_PROGRESSBAR_X,
	.progressbar_y      = USPLASH_1365_768_PROGRESSBAR_Y,
	.progressbar_width  = USPLASH_1365_768_PROGRESSBAR_WIDTH,
	.progressbar_height = USPLASH_1365_768_PROGRESSBAR_HEIGHT,

	/* Text box position and size in pixels */
	.text_x      = USPLASH_1365_768_TEXT_X,
	.text_y      = USPLASH_1365_768_TEXT_Y,
	.text_width  = USPLASH_1365_768_TEXT_WIDTH,
	.text_height = USPLASH_1365_768_TEXT_HEIGHT,

	/* Text details */
	.line_height  = USPLASH_1365_768_LINE_HEIGHT,
	.line_length  = USPLASH_1365_768_LINE_LENGTH,
	.status_width = USPLASH_1365_768_STATUS_WIDTH,

    /* Functions */
    .init = t_init,
    .clear_progressbar = t_clear_progressbar,
    .draw_progressbar = t_update_progress,
    .animate_step = t_animate_step,
};

/*******************************
 *******************************/

void t_init(struct usplash_theme *theme) {
    int x, y;
    usplash_getdimensions(&x, &y);
    theme->progressbar_x = (x - theme->pixmap->width)/2 + theme->progressbar_x;
    theme->progressbar_y = (y - theme->pixmap->height)/2 + theme->progressbar_y;
}

void t_clear_progressbar(struct usplash_theme *theme) {
    usplash_put(theme->progressbar_x - 2, theme->progressbar_y, &pixmap_throbber_back_alt);
}

void t_update_progress(struct usplash_theme *theme, int percentage) {
	progress_complete = percentage * 100;
}

void t_animate_step(struct usplash_theme* theme, int pulsating) 
{
    static int booting_up = TRUE;
    static int boot_change = FALSE;
    static int new_pulse = TRUE;
    static struct pulse_rec pulse;
    
    if (pulsating) {
        if (new_pulse) {
		    pulse.num_ticks = 0;
		    pulse.position = 0;
            pulse.period = 75;
		    pulse.angle = 0;
		    pulse.bar_size = theme->progressbar_width / 6;
            pulse.region_size = theme->progressbar_width - theme->progressbar_width / 6;
		    new_pulse = FALSE;   
		}
        harmonic_pulse(&pulse);
        draw_pulse(&pulse, theme);
		
    } else {
    	new_pulse = TRUE;
    	if (booting_up && progress_complete < 0) {
    		boot_change = TRUE;
    	} else if (!booting_up && progress_complete > 0) {
    		boot_change = TRUE;
    	}

    	draw_progress(theme, smooth_progress(theme, boot_change), boot_change, booting_up);

    	if (boot_change) {
    		boot_change = FALSE;
    		booting_up = (booting_up) ? FALSE: TRUE;
    	}
    }
}

/******
 * draw_progress:
 * 
 * Draws progress bar at given progress amount.
 */
void draw_progress (struct usplash_theme * theme, int percent_progress, int boot_changed, int booting_up)
{
	int progress_amount;
	
	if (abs(percent_progress) == 10000) {
		progress_amount = theme->progressbar_width;
	} else {
		progress_amount = abs(percent_progress) / (10000 / theme->progressbar_width);
		if (progress_amount > theme->progressbar_width) {
			progress_amount = theme->progressbar_width;
		}
	}

	if (boot_changed) {
		if (booting_up) {
			usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_throbber_fore_alt);
		} else {
			usplash_put(theme->progressbar_x - 2, theme->progressbar_y, &pixmap_throbber_back_alt);
		}
		
	}
	
	if (percent_progress > 0) { /** starting up **/
		usplash_put_part(theme->progressbar_x, theme->progressbar_y,
                        progress_amount/2,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        0, 0);
		usplash_put_part(theme->progressbar_x + progress_amount/2,
                        theme->progressbar_y,
                        progress_amount - progress_amount/2,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        theme->progressbar_width - progress_amount + progress_amount/2, 0);
	} else if (percent_progress < 0) { /** shutting down **/
		usplash_put_part(theme->progressbar_x - 2, theme->progressbar_y,
                        progress_amount + 2,
                        pixmap_throbber_back_alt.height, &pixmap_throbber_back_alt,
                        0, 0);
        if (progress_amount < theme->progressbar_width) {
        	if ((theme->progressbar_width - progress_amount) < 10) {
				usplash_put_part(theme->progressbar_x + progress_amount,
                        theme->progressbar_y, 
                        (theme->progressbar_width - progress_amount)/2,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        0, 0);
                usplash_put_part(theme->progressbar_x + progress_amount + (theme->progressbar_width - progress_amount)/2,
                        theme->progressbar_y,
                        (theme->progressbar_width - progress_amount) - (theme->progressbar_width - progress_amount)/2,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        theme->progressbar_width - (theme->progressbar_width - progress_amount) + (theme->progressbar_width - progress_amount)/2, 0);
        	} else {
                usplash_put_part(theme->progressbar_x + progress_amount,
                        theme->progressbar_y, 5,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
	                    0, 0);
	        }
        }
	}
} /* End draw_progress */

/******
 * smooth_progress:
 * 
 * Smooths progress updates with some prediction. 
 * 
 * Param:
 * theme - current usplash theme struct.
 * 
 * Return:
 * int   - current smoothed progress values range from 0 - 10000.
 */
int smooth_progress (struct usplash_theme* theme, int boot_changed)
{
	static int partial_percent = 0;
	static int tick = 0;
	
	int time_remaining;
	int time_to_complete;
	int l_progress_complete;
	
	if (boot_changed) {
		partial_percent = 0;
		tick = 0;
	}
	
	l_progress_complete = abs(progress_complete);

	time_remaining = (tick / (float) (l_progress_complete + 1)) 
	                                         * (10000 - l_progress_complete);
	time_to_complete = (10000 - partial_percent) / 150;

	if (time_to_complete >= time_remaining) { /** need to finish **/
		partial_percent += 150;
	} else if (partial_percent - 3000 > l_progress_complete) { /** ahead **/
		partial_percent += 1;
	} else if (partial_percent - 2000 > l_progress_complete) {
		partial_percent += 5;
	} else if (partial_percent - 1000 > l_progress_complete) {
		partial_percent += 12;
	} else if (partial_percent + 2000 < l_progress_complete) { /** behind **/
		partial_percent += 100;
	} else if (partial_percent + 1000 < l_progress_complete) {
		partial_percent += 60;
	} else {
		partial_percent += 30;
	}
	
	if (partial_percent > 10000) {
		partial_percent = 10000;
	}
	
	tick += 1;

	if (progress_complete < 0) {
		return -partial_percent;
	} else {
		return partial_percent;
	}
} /* End smooth_progress */

/******
 * harmonic_pulse:
 * 
 * Harmonic motion of pulse bar.
 * 
 * Takes the current progress completed and shrinks/grows it to a pulse bar
 * while moving forward and back with harmonic motion.
 * 
 * Params:
 * pulse - record for pulse animation
 */
void harmonic_pulse (struct pulse_rec *pulse) 
{
	
	/** get position for harmonic motion **/
	pulse->angle += (float) 2 / pulse->period;
	if (pulse->num_ticks % pulse->period == 0) {
		pulse->angle = 0;
	}
	pulse->position = pulse->region_size / 2 - cosf(pulse->angle * M_PI)
	                        * pulse->region_size / 2;
	
	pulse->num_ticks += 1;
} /* End harmonic_pulse */

/******
 * draw_pulse:
 * 
 * Draws current pulse bar frame.
 * 
 * Params:
 * pulse - record for pulse animation
 */
void draw_pulse (struct pulse_rec *pulse, struct usplash_theme* theme) 
{
	/** draw background on left **/
	if (pulse->position > 0) {
        usplash_put_part(theme->progressbar_x - 2, theme->progressbar_y,
                        pulse->position + 2, pixmap_throbber_back_alt.height,
                        &pixmap_throbber_back_alt, 0, 0);
	}

    /** draw pulse bar **/
    /** left round **/
    usplash_put_part(theme->progressbar_x + pulse->position,
                        theme->progressbar_y, 5,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        0, 0);
	/** bar center **/
    usplash_put_part(theme->progressbar_x + pulse->position + 5,
                        theme->progressbar_y, pulse->bar_size - 10,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        5, 0);
    /** right round **/
    usplash_put_part(theme->progressbar_x + pulse->position
                        + pulse->bar_size - 5, theme->progressbar_y, 5,
                        pixmap_throbber_fore_alt.height, &pixmap_throbber_fore_alt,
                        211, 0);
	/** draw background on right **/
	if (theme->progressbar_width -pulse->position -pulse->bar_size > 0) {
        usplash_put_part(theme->progressbar_x + pulse->position
                        + pulse->bar_size,
                        theme->progressbar_y,
                        theme->progressbar_width - pulse->position
                        - pulse->bar_size,
                        pixmap_throbber_back_alt.height, &pixmap_throbber_back_alt,
                        pulse->position + pulse->bar_size + 2, 0);
    }
} /* End draw_pulse */
