/* usplash
 *
 * eft-theme.c - definition of eft theme
 *
 * Copyright © 2008 Tonic Artos (Russell Lomax) <ghatanothoa@gmail.com>
 *           © 2008 Cory K
 *           © 2006 Dennis Kaarsemaker <dennis@kaarsemaker.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <math.h>
#include <usplash-theme.h>
/* Needed for the custom drawing functions */
#include <usplash_backend.h>
#include <stdio.h>


#define TRUE 1
#define FALSE 0

extern struct usplash_pixmap pixmap_usplash_640_400, pixmap_usplash_640_480;
extern struct usplash_pixmap pixmap_usplash_800_600, pixmap_usplash_1024_768, pixmap_usplash_1365_768_scaled;
extern struct usplash_pixmap pixmap_bar_bg_1024, pixmap_bar_bg_800, pixmap_bar_bg_640;
extern struct usplash_pixmap pixmap_bar_fg_1024, pixmap_bar_fg_800, pixmap_bar_fg_640;
extern struct usplash_pixmap pixmap_throbber_back_16;
extern struct usplash_pixmap pixmap_throbber_fore_16;

struct pulse_rec {
	int position; /** current position - start at zero **/
	int bar_size; /** current size of pulse bar - start at current progress **/
	int region_size; /** most likely progress bar width **/
	float angle; /** angle in pi radians **/
	int period; /** ticks to complete a revolution **/
	int num_ticks; /** number of ticks pulse has run **/
};

void t_init_16(struct usplash_theme* theme);
void t_init_640(struct usplash_theme* theme);
void t_init_800(struct usplash_theme* theme);
void t_init_1024(struct usplash_theme* theme);
void t_clear_progressbar_16(struct usplash_theme* theme);
void t_clear_progressbar_640(struct usplash_theme* theme);
void t_clear_progressbar_800(struct usplash_theme* theme);
void t_clear_progressbar_1024(struct usplash_theme* theme);
void t_draw_progressbar(struct usplash_theme* theme, int percentage);
void t_draw_progressbar_16(struct usplash_theme* theme, int percentage);
void t_animate_step_16(struct usplash_theme* theme, int pulsating);
void t_animate_step_640(struct usplash_theme* theme, int pulsating);
void t_animate_step_800(struct usplash_theme* theme, int pulsating);
void t_animate_step_1024(struct usplash_theme* theme, int pulsating);

void draw_progress (struct usplash_theme * theme, int percent_progress, int boot_changed, int booting_up, struct usplash_pixmap * throbber_fore, struct usplash_pixmap * throbber_back);
int smooth_progress (struct usplash_theme* theme, int boot_changed);
void harmonic_pulse (struct pulse_rec *pulse);
void draw_pulse (struct pulse_rec *pulse, struct usplash_theme* theme, struct usplash_pixmap * throbber_fore, struct usplash_pixmap * throbber_back);

struct usplash_theme usplash_theme_640_480;
struct usplash_theme usplash_theme_800_600;
struct usplash_theme usplash_theme_1024_768;
struct usplash_theme usplash_theme_1365_768_scaled;

int progress_complete;

/* Theme definition */
struct usplash_theme usplash_theme = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_640_480,
    .ratio = USPLASH_16_9,

	/* Background and font */
	.pixmap = &pixmap_usplash_640_400,

	/* Palette indexes */
	.background             = 0,
  	.progressbar_background = 32,
  	.progressbar_foreground = 131,
	.text_background        = 0,
	.text_foreground        = 117,
	.text_success           = 189,
	.text_failure           = 55,

	/* Progress bar position and size in pixels */
  	.progressbar_x      = 212,
  	.progressbar_y      = 196,
  	.progressbar_width  = 216,
  	.progressbar_height = 8,

	/* Text box position and size in pixels */
  	.text_x      = 96,
  	.text_y      = 306,
  	.text_width  = 0,
  	.text_height = 0,

	/* Text details */
  	.line_height  = 15,
  	.line_length  = 32,
  	.status_width = 35,

    /* Functions */
    .init = t_init_16,
    .clear_progressbar = t_clear_progressbar_16,
    .draw_progressbar = t_draw_progressbar_16,
    .animate_step = t_animate_step_16,
};

struct usplash_theme usplash_theme_640_480 = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_800_600,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_640_480,

	/* Palette indexes */
	.background             = 0,
  	.progressbar_background = 32,
  	.progressbar_foreground = 131,
	.text_background        = 0,
	.text_foreground        = 117,
	.text_success           = 189,
	.text_failure           = 55,

	/* Progress bar position and size in pixels */
  	.progressbar_x      = 160,
  	.progressbar_y      = 296,
  	.progressbar_width  = 0,
  	.progressbar_height = 0,

	/* Text box position and size in pixels */
  	.text_x      = 120,
  	.text_y      = 390,
  	.text_width  = 360,
  	.text_height = 75,

	/* Text details */
  	.line_height  = 15,
  	.line_length  = 32,
  	.status_width = 35,

    /* Functions */
    .init = t_init_640,
    .clear_progressbar = t_clear_progressbar_640,
    .draw_progressbar = t_draw_progressbar,
    .animate_step = t_animate_step_640,
};

struct usplash_theme usplash_theme_800_600 = {
	.version = THEME_VERSION, /* ALWAYS set this to THEME_VERSION, 
                                 it's a compatibility check */
    .next = &usplash_theme_1024_768,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_800_600,

	/* Palette indexes */
	.background             = 0,
  	.progressbar_background = 32,
  	.progressbar_foreground = 131,
	.text_background        = 0,
	.text_foreground        = 117,
	.text_success           = 189,
	.text_failure           = 55,

	/* Progress bar position and size in pixels */
  	.progressbar_x      = 240,
  	.progressbar_y      = 352,
  	.progressbar_width  = 0,
  	.progressbar_height = 0,

	/* Text box position and size in pixels */
  	.text_x      = 220,
  	.text_y      = 480,
  	.text_width  = 360,
  	.text_height = 75,

	/* Text details */
  	.line_height  = 15,
  	.line_length  = 32,
  	.status_width = 35,

    /* Functions */
    .init = t_init_800,
    .clear_progressbar = t_clear_progressbar_800,
    .draw_progressbar = t_draw_progressbar,
    .animate_step = t_animate_step_800,
};

struct usplash_theme usplash_theme_1024_768 = {
	.version = THEME_VERSION,
    .next = &usplash_theme_1365_768_scaled,
    .ratio = USPLASH_4_3,

	/* Background and font */
	.pixmap = &pixmap_usplash_1024_768,

	/* Palette indexes */
	.background             = 0,
  	.progressbar_background = 32,
  	.progressbar_foreground = 131,
	.text_background        = 0,
	.text_foreground        = 117,
	.text_success           = 189,
	.text_failure           = 55,

	/* Progress bar position and size in pixels */
  	.progressbar_x      = 352,
  	.progressbar_y      = 445,
  	.progressbar_width  = 0,
  	.progressbar_height = 0,

	/* Text box position and size in pixels */
  	.text_x      = 322,
  	.text_y      = 633,
  	.text_width  = 380,
  	.text_height = 75,

	/* Text details */
  	.line_height  = 15,
  	.line_length  = 32,
  	.status_width = 35,

    /* Functions */
    .init = t_init_1024,
    .clear_progressbar = t_clear_progressbar_1024,
    .draw_progressbar = t_draw_progressbar,
    .animate_step = t_animate_step_1024,
};

struct usplash_theme usplash_theme_1365_768_scaled = {
	.version = THEME_VERSION,
    .next = NULL,
    .ratio = USPLASH_16_9,

	/* Background and font */
	.pixmap = &pixmap_usplash_1365_768_scaled,

	/* Palette indexes */
	.background             = 0,
  	.progressbar_background = 32,
  	.progressbar_foreground = 131,
	.text_background        = 0,
	.text_foreground        = 117,
	.text_success           = 189,
	.text_failure           = 55,

	/* Progress bar position and size in pixels */
  	.progressbar_x      = 352,
  	.progressbar_y      = 445,
  	.progressbar_width  = 0,
  	.progressbar_height = 0,

	/* Text box position and size in pixels */
  	.text_x      = 322,
  	.text_y      = 633,
  	.text_width  = 380,
  	.text_height = 75,

	/* Text details */
  	.line_height  = 15,
  	.line_length  = 32,
  	.status_width = 35,

    /* Functions */
    .init = t_init_1024,
    .clear_progressbar = t_clear_progressbar_1024,
    .draw_progressbar = t_draw_progressbar,
    .animate_step = t_animate_step_1024,
};

void t_init_16(struct usplash_theme *theme) {
    int x, y;
    usplash_getdimensions(&x, &y);
    theme->progressbar_x = x/2 - pixmap_throbber_fore_16.width/2;
    theme->progressbar_y = (y - 400)/2 + theme->progressbar_y;
}

void t_init_640(struct usplash_theme *theme) {
    int x, y;
    usplash_getdimensions(&x, &y);
    theme->progressbar_x = x/2 - pixmap_bar_fg_640.width/2;
    theme->progressbar_y = (y - 480)/2 + theme->progressbar_y;
}

void t_init_800(struct usplash_theme *theme) {
    int x, y;
    usplash_getdimensions(&x, &y);
    theme->progressbar_x = x/2 - pixmap_bar_fg_800.width/2;
    theme->progressbar_y = (y - 600)/2 + theme->progressbar_y;
    fprintf(stderr, "%d, %d.\n", x, y);
}

void t_init_1024(struct usplash_theme *theme) {
    int x, y;
    usplash_getdimensions(&x, &y);
    theme->progressbar_x = x/2 - pixmap_bar_fg_1024.width/2;
    theme->progressbar_y = (y - 768)/2 + theme->progressbar_y;
}

void t_clear_progressbar_16(struct usplash_theme *theme) {
    usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_throbber_back_16);
}

void t_draw_progressbar_16(struct usplash_theme *theme, int percentage) {
	progress_complete = percentage * 100;
}

void t_clear_progressbar_1024(struct usplash_theme *theme) {
    usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_1024);
}

void t_clear_progressbar_800(struct usplash_theme *theme) {
    usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_800);
}

void t_clear_progressbar_640(struct usplash_theme *theme) {
    usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_640);
}

void t_draw_progressbar(struct usplash_theme *theme, int percentage) {
	progress_complete = percentage * 100;
}

void t_animate_step_1024(struct usplash_theme* theme, int pulsating)  {
    static int booting_up = TRUE;
    static int boot_change = FALSE;
    static int new_pulse = TRUE;
    static struct pulse_rec pulse;
    
    if (pulsating) {
        if (new_pulse) {
		    pulse.num_ticks = 0;
		    pulse.position = 0;
            pulse.period = 150;
		    pulse.angle = 0;
		    pulse.bar_size = pixmap_bar_fg_1024.width / 6;
            pulse.region_size = pixmap_bar_fg_1024.width - pixmap_bar_fg_1024.width / 6;
		    new_pulse = FALSE;   
		}
        harmonic_pulse(&pulse);
        draw_pulse(&pulse, theme, &pixmap_bar_fg_1024, &pixmap_bar_bg_1024);
		
    } else {
    	if (!new_pulse) {
    		new_pulse = TRUE;
			usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_1024);
    	}
    	if (booting_up && progress_complete < 0) {
    		boot_change = TRUE;
    	} else if (!booting_up && progress_complete > 0) {
    		boot_change = TRUE;
    	}

    	draw_progress(theme, smooth_progress(theme, boot_change), boot_change, booting_up, &pixmap_bar_fg_1024, &pixmap_bar_bg_1024);

    	if (boot_change) {
    		boot_change = FALSE;
    		booting_up = (booting_up) ? FALSE: TRUE;
    	}
    }
}

void t_animate_step_800(struct usplash_theme* theme, int pulsating)  {
    static int booting_up = TRUE;
    static int boot_change = FALSE;
    static int new_pulse = TRUE;
    static struct pulse_rec pulse;
    
    if (pulsating) {
        if (new_pulse) {
		    pulse.num_ticks = 0;
		    pulse.position = 0;
            pulse.period = 150;
		    pulse.angle = 0;
		    pulse.bar_size = pixmap_bar_fg_800.width / 6;
            pulse.region_size = pixmap_bar_fg_800.width - pixmap_bar_fg_800.width / 6;
		    new_pulse = FALSE;   
		}
        harmonic_pulse(&pulse);
        draw_pulse(&pulse, theme, &pixmap_bar_fg_800, &pixmap_bar_bg_800);
		
    } else {
    	if (!new_pulse) {
    		new_pulse = TRUE;
			usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_800);
    	}
    	if (booting_up && progress_complete < 0) {
    		boot_change = TRUE;
    	} else if (!booting_up && progress_complete > 0) {
    		boot_change = TRUE;
    	}

    	draw_progress(theme, smooth_progress(theme, boot_change), boot_change, booting_up, &pixmap_bar_fg_800, &pixmap_bar_bg_800);

    	if (boot_change) {
    		boot_change = FALSE;
    		booting_up = (booting_up) ? FALSE: TRUE;
    	}
    }
}

void t_animate_step_640(struct usplash_theme* theme, int pulsating)  {
    static int booting_up = TRUE;
    static int boot_change = FALSE;
    static int new_pulse = TRUE;
    static struct pulse_rec pulse;
    
    if (pulsating) {
        if (new_pulse) {
		    pulse.num_ticks = 0;
		    pulse.position = 0;
            pulse.period = 150;
		    pulse.angle = 0;
		    pulse.bar_size = pixmap_bar_fg_640.width / 6;
            pulse.region_size = pixmap_bar_fg_640.width - pixmap_bar_fg_640.width / 6;
		    new_pulse = FALSE;   
		}
        harmonic_pulse(&pulse);
        draw_pulse(&pulse, theme, &pixmap_bar_fg_640, &pixmap_bar_bg_640);
		
    } else {
    	if (!new_pulse) {
    		new_pulse = TRUE;
			usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_bar_bg_640);
    	}
    	if (booting_up && progress_complete < 0) {
    		boot_change = TRUE;
    	} else if (!booting_up && progress_complete > 0) {
    		boot_change = TRUE;
    	}

    	draw_progress(theme, smooth_progress(theme, boot_change), boot_change, booting_up, &pixmap_bar_fg_640, &pixmap_bar_bg_640);

    	if (boot_change) {
    		boot_change = FALSE;
    		booting_up = (booting_up) ? FALSE: TRUE;
    	}
    }
}

void t_animate_step_16(struct usplash_theme* theme, int pulsating) {
    static int booting_up = TRUE;
    static int boot_change = FALSE;
    static int new_pulse = TRUE;
    static struct pulse_rec pulse;
    
    if (pulsating) {
        if (new_pulse) {
		    pulse.num_ticks = 0;
		    pulse.position = 0;
            pulse.period = 150;
		    pulse.angle = 0;
		    pulse.bar_size = pixmap_throbber_fore_16.width / 6;
            pulse.region_size = pixmap_throbber_fore_16.width - pixmap_throbber_fore_16.width / 6;
		    new_pulse = FALSE;   
		}
        harmonic_pulse(&pulse);
        draw_pulse(&pulse, theme, &pixmap_throbber_fore_16, &pixmap_throbber_back_16);
		
    } else {
    	if (!new_pulse) {
    		new_pulse = TRUE;
			usplash_put(theme->progressbar_x, theme->progressbar_y, &pixmap_throbber_back_16);
    	}
    	if (booting_up && progress_complete < 0) {
    		boot_change = TRUE;
    	} else if (!booting_up && progress_complete > 0) {
    		boot_change = TRUE;
    	}

    	draw_progress(theme, smooth_progress(theme, boot_change), boot_change, booting_up, &pixmap_throbber_fore_16, &pixmap_throbber_back_16);

    	if (boot_change) {
    		boot_change = FALSE;
    		booting_up = (booting_up) ? FALSE: TRUE;
    	}
    }
}

/******
 * draw_progress:
 * 
 * Draws progress bar at given progress amount.
 */
void draw_progress (struct usplash_theme * theme, int percent_progress, int boot_changed, int booting_up, struct usplash_pixmap * throbber_fore, struct usplash_pixmap * throbber_back)
{
	int progress_amount;
	
	if (abs(percent_progress) == 10000) {
		progress_amount = throbber_fore->width;
	} else {
		progress_amount = abs(percent_progress) / (10000 / throbber_fore->width);
		if (progress_amount > throbber_fore->width) {
			progress_amount = throbber_fore->width;
		}
	}

	if (boot_changed) {
		if (booting_up) {
			usplash_put(theme->progressbar_x, theme->progressbar_y, throbber_fore);
		} else {
			usplash_put(theme->progressbar_x, theme->progressbar_y, throbber_back);
		}
		
	}
	
	if (percent_progress > 0) { /** starting up **/
		usplash_put_part(theme->progressbar_x, theme->progressbar_y,
                        progress_amount,
                        throbber_fore->height, throbber_fore,
                        0, 0);
	} else if (percent_progress < 0) { /** shutting down **/
		usplash_put_part(theme->progressbar_x, theme->progressbar_y,
                        progress_amount,
                        throbber_back->height, throbber_back,
                        0, 0);
	}
} /* End draw_progress */

/******
 * smooth_progress:
 * 
 * Smooths progress updates with some prediction. 
 * 
 * Param:
 * theme - current usplash theme struct.
 * 
 * Return:
 * int   - current smoothed progress values range from 0 - 10000.
 */
int smooth_progress (struct usplash_theme* theme, int boot_changed)
{
	static int partial_percent = 0;
	static int tick = 0;
	
	int time_remaining;
	int time_to_complete;
	int l_progress_complete;
	
	if (boot_changed) {
		partial_percent = 0;
		tick = 0;
	}
	
	l_progress_complete = abs(progress_complete);

	time_remaining = (tick / (float) (l_progress_complete + 1)) 
	                                         * (10000 - l_progress_complete);
	time_to_complete = (10000 - partial_percent) / 150;

	if (time_to_complete >= time_remaining) { /** need to finish **/
		partial_percent += 150;
	} else if (partial_percent - 3000 > l_progress_complete) { /** ahead **/
		partial_percent += 1;
	} else if (partial_percent - 2000 > l_progress_complete) {
		partial_percent += 5;
	} else if (partial_percent - 1000 > l_progress_complete) {
		partial_percent += 12;
	} else if (partial_percent + 2000 < l_progress_complete) { /** behind **/
		partial_percent += 100;
	} else if (partial_percent + 1000 < l_progress_complete) {
		partial_percent += 60;
	} else {
		partial_percent += 30;
	}
	
	if (partial_percent > 10000) {
		partial_percent = 10000;
	}
	
	tick += 1;

	if (progress_complete < 0) {
		return -partial_percent;
	} else {
		return partial_percent;
	}
} /* End smooth_progress */

/******
 * harmonic_pulse:
 * 
 * Harmonic motion of pulse bar.
 * 
 * Takes the current progress completed and shrinks/grows it to a pulse bar
 * while moving forward and back with harmonic motion.
 * 
 * Params:
 * pulse - record for pulse animation
 */
void harmonic_pulse (struct pulse_rec *pulse) 
{
	
	/** get position for harmonic motion **/
	pulse->angle += (float) 2 / pulse->period;
	if (pulse->num_ticks % pulse->period == 0) {
		pulse->angle = 0;
	}
	pulse->position = pulse->region_size / 2 - cosf(pulse->angle * M_PI)
	                        * pulse->region_size / 2;
	
	pulse->num_ticks += 1;
} /* End harmonic_pulse */

/******
 * draw_pulse:
 * 
 * Draws current pulse bar frame.
 * 
 * Params:
 * pulse - record for pulse animation
 */
void draw_pulse (struct pulse_rec *pulse, struct usplash_theme* theme, struct usplash_pixmap * throbber_fore, struct usplash_pixmap * throbber_back) 
{
	/** draw background on left **/
	if (pulse->position > 0) {
        usplash_put_part(theme->progressbar_x, theme->progressbar_y,
                        pulse->position, throbber_back->height,
                        throbber_back, 0, 0);
	}
    usplash_put_part(theme->progressbar_x + pulse->position,
                        theme->progressbar_y, pulse->bar_size,
                        throbber_fore->height, throbber_fore,
                        pulse->position, 0);
	/** draw background on right **/
	if (throbber_fore->width -pulse->position -pulse->bar_size > 0) {
        usplash_put_part(theme->progressbar_x + pulse->position
                        + pulse->bar_size,
                        theme->progressbar_y,
                        throbber_fore->width - pulse->position
                        - pulse->bar_size,
                        throbber_back->height, throbber_back,
                        pulse->position + pulse->bar_size, 0);
    }
} /* End draw_pulse */
