/* usplash
 *
 * Copyright © 2006, 2007 Canonical Ltd.
 * Copyright © 2006 Dennis Kaarsemaker <dennis@kaarsemaker.net>
 * Copyright © 2005 Matthew Garrett <mjg59@srcf.ucam.org>
 *
 * Console font handling from kbd, whose COPYING file states:
 *   Copyright (C) 1992 Rickard E. Faith.
 *   Copyright (C) 1993 Risto Kankkunen.
 *   Copyright (C) 1993 Eugene G. Crosser.
 *   Copyright (C) 1994 H. Peter Anvin.
 *   Copyright (C) 1994-1999 Andries E. Brouwer.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <linux/vt.h>
#include <linux/limits.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/kd.h>

#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <signal.h>
#include <poll.h>
#include <termios.h>
#include <time.h>

#include "usplash_backend.h"
#include "usplash_bogl_backend.h"
#ifdef SVGA
#include "usplash_svga_backend.h"
#endif
#include "usplash.h"
#include "usplash-theme.h"

#include "libusplash.h"

/*
 * This shouldn't be needed any more now that animation takes place
 * outside the alarm handler.  However, removing it seems to introduce
 * an Intel driver X initialization problem (LP: #327230).  ?!
 */
sigset_t sigs;
#define blocksig() do{ sigprocmask(SIG_BLOCK, &sigs, NULL); } while(0)
#define unblocksig() do{ sigprocmask(SIG_UNBLOCK, &sigs, NULL); } while(0)

/* Prototypes of non-static functions */
/* Helpers for video implementations */
void usplash_restore_console(void);
void usplash_save_font(void);
void usplash_restore_font(void);

/* Prototypes of static functions */
static void ensure_console(void);
static int get_font(int *width, int *height, int *count, unsigned char *buf);
static int set_font(int width, int height, int count, unsigned char *buf);
static void draw_newline(void);
static void draw_chars(const char *string, size_t len);

/* Theme being used */
struct usplash_theme *theme;

/* Distance to themed area from the top-left corner of the screen */
int left_edge, top_edge;

/* Current text output position (i.e. where the cursor would be) */
static int text_position;

/* Coordinates of the text box */
static int text_x1, text_x2, text_y1, text_y2;

/* Size of the screen */
int usplash_xres, usplash_yres;

/* Virtual terminal we switched away from */
static int saved_vt = 0;
static int saved_vt_fd = -1;

/* Virtual terminal we switched to */
static int new_vt = 0;

/* Saved console font */
static struct {
	int width;
	int height;
	int charcount;
	unsigned char *data;
} saved_font = { 0, 0, 0, NULL };

/* Number of seconds to wait for a command before exiting */
static int timeout = 15;

/* Are we verbose or not? */
static int verbose = 0;
static int verbose_default = 0;

/* /dev/console */
static int console_fd = -1;

/* pipe to trigger animation steps */
static int alarm_fds[2];
int animation_needed_fd;
static int default_pulsate_mode = 0;

struct usplash_funcs {
	const char *name;
	int (*usplash_setfont) (void *font);
	int (*usplash_getfontwidth) (char c);
	int (*usplash_init) (void);
	int (*usplash_set_resolution) (int x, int y);
	void (*usplash_set_palette) (int ncols,
				     unsigned char palette[][3]);
	void (*usplash_clear) (int x1, int y1, int x2, int y2, int colour);
	void (*usplash_move) (int sx, int sy, int dx, int dy, int w,
			      int h);
	void (*usplash_text) (int x, int y, const char *s, int len, int fg,
			      int bg);
	void (*usplash_done) ();
	void (*usplash_getdimensions) (int *x, int *y);
	void (*usplash_put) (int x, int y, void *pointer);
	void (*usplash_put_part) (int x, int y, int w, int h,
				  void *pointer, int x0, int y0);
};

#ifdef SVGA
struct usplash_funcs usplash_svga_funcs = {
	.name = "svga",
	.usplash_setfont = usplash_svga_setfont,
	.usplash_getfontwidth = usplash_svga_getfontwidth,
	.usplash_init = usplash_svga_init,
	.usplash_set_resolution = usplash_svga_set_resolution,
	.usplash_set_palette = usplash_svga_set_palette,
	.usplash_clear = usplash_svga_clear,
	.usplash_move = usplash_svga_move,
	.usplash_text = usplash_svga_text,
	.usplash_done = usplash_svga_done,
	.usplash_getdimensions = usplash_svga_getdimensions,
	.usplash_put = usplash_svga_put,
	.usplash_put_part = usplash_svga_put_part,
};
#endif

struct usplash_funcs usplash_bogl_funcs = {
	.name = "bogl",
	.usplash_setfont = usplash_bogl_setfont,
	.usplash_getfontwidth = usplash_bogl_getfontwidth,
	.usplash_init = usplash_bogl_init,
	.usplash_set_resolution = usplash_bogl_set_resolution,
	.usplash_set_palette = usplash_bogl_set_palette,
	.usplash_clear = usplash_bogl_clear,
	.usplash_move = usplash_bogl_move,
	.usplash_text = usplash_bogl_text,
	.usplash_done = usplash_bogl_done,
	.usplash_getdimensions = usplash_bogl_getdimensions,
	.usplash_put = usplash_bogl_put,
	.usplash_put_part = usplash_bogl_put_part,
};

static struct usplash_funcs *usplash_operations;

const char *usplash_backend_name(void)
{
	return usplash_operations->name;
}

int usplash_setfont(void *font)
{
	return usplash_operations->usplash_setfont(font);
}

int usplash_getfontwidth(char c)
{
	return usplash_operations->usplash_getfontwidth(c);
}

int usplash_init()
{
	return usplash_operations->usplash_init();
}

int usplash_set_resolution(int x, int y)
{
	return usplash_operations->usplash_set_resolution(x, y);
}

void usplash_set_palette(int ncols, unsigned char palette[][3])
{
	usplash_operations->usplash_set_palette(ncols, palette);
}

void usplash_clear(int x1, int y1, int x2, int y2, int colour)
{
	usplash_operations->usplash_clear(x1, y1, x2, y2, colour);
}

void usplash_move(int sx, int sy, int dx, int dy, int w, int h)
{
	usplash_operations->usplash_move(sx, sy, dx, dy, w, h);
}

void usplash_text(int x, int y, const char *s, int len, int fg, int bg)
{
	usplash_operations->usplash_text(x, y, s, len, fg, bg);
}

void usplash_done()
{
	usplash_operations->usplash_done();

	close(alarm_fds[1]);
	close(alarm_fds[0]);
	animation_needed_fd = alarm_fds[0] = alarm_fds[1] = -1;
}

void usplash_getdimensions(int *x, int *y)
{
	usplash_operations->usplash_getdimensions(x, y);
}

void usplash_put(int x, int y, void *pointer)
{
	usplash_operations->usplash_put(x, y, pointer);
}

void usplash_put_part(int x, int y, int w, int h, void *pointer, int x0,
		      int y0)
{
	usplash_operations->usplash_put_part(x, y, w, h, pointer, x0, y0);
}

void usplash_setup_funcs()
{
#ifdef SVGA
	/* Check which set of functions we should be using */
	int fd;
	fd = open("/dev/fb0", O_RDWR | O_NOCTTY);
	if (fd < 0) {
		usplash_operations = &usplash_svga_funcs;
		return;
	}
	close(fd);
#endif
	usplash_operations = &usplash_bogl_funcs;
}

void ensure_console(void)
{
	if (console_fd == -1) {
		console_fd = open("/dev/console", O_RDWR | O_NOCTTY);
		if (console_fd == -1) {
			fprintf(stderr,
				"usplash: cannot open /dev/console: %s\n",
				strerror(errno));
			exit(1);
		}
	}
}

void switch_console(int vt, int vt_fd)
{
	char saved_vtname[10];
	struct vt_stat state;

	ensure_console();
	ioctl(console_fd, VT_GETSTATE, &state);

	saved_vt = state.v_active;
	assert((saved_vt >= 0) && (saved_vt < 10));
	sprintf(saved_vtname, "/dev/tty%d", saved_vt);
	/* This may fail when restoring the console before exit, since the
	 * initramfs has gone away; but that's OK.
	 */
	saved_vt_fd = open(saved_vtname, O_RDWR | O_NOCTTY);
	new_vt = vt;

	ioctl(vt_fd, VT_ACTIVATE, vt);
	/* Note that, when using SVGA, we may be interrupted around here by
	 * a signal and never come back. See __svgalib_releasevt_signal.
	 */
	ioctl(vt_fd, VT_WAITACTIVE, vt);

	close(STDIN_FILENO);
	dup2(vt_fd, 0);
}

void usplash_restore_console(void)
{
	struct vt_stat state;

	if (saved_vt != 0 && saved_vt_fd != -1) {
		ensure_console();
		ioctl(console_fd, VT_GETSTATE, &state);

		/* Switch back if we're still on the console we switched to */
		if (state.v_active == new_vt)
			switch_console(saved_vt, saved_vt_fd);
	}
}

/* This is a nasty bodge. gdm's Upstart job sometimes seems to exit before
 * the X server has got round to shutting down and giving us back a normal
 * VT. Ideally we'd use VT_WAITINACTIVE but that doesn't exist. Since
 * shutdown speed isn't so critical and since this doesn't block shutdown,
 * we just busy-wait ...
 */
void wait_normal_vt(void)
{
	char vtname[11];
	struct vt_stat state;
	int fd;
	long vc_mode;

	ensure_console();
	ioctl(console_fd, VT_GETSTATE, &state);

	assert((state.v_active >= 0) && (state.v_active < 100));
	sprintf(vtname, "/dev/tty%d", state.v_active);
	fd = open(vtname, O_RDWR | O_NOCTTY);

	for (;;) {
		struct timespec wait;

		ioctl(fd, KDGETMODE, &vc_mode);
		if (vc_mode != KD_GRAPHICS)
			break;
		wait.tv_sec = 0;
		wait.tv_nsec = 100000000; /* 0.1 seconds */
		nanosleep(&wait, NULL);
	}
}

/* introduced in Linux 2.1.111 */
#ifndef KDFONTOP
#define KDFONTOP 0x4B72
struct console_font_op {
	unsigned int op;
	unsigned int flags;
	unsigned int width, height;
	unsigned int charcount;
	unsigned char *data;
};

#define KD_FONT_OP_SET 0
#define KD_FONT_OP_GET 1
#endif /* KDFONTOP */

/* may be called with buf==NULL if we only want info */
int get_font(int *width, int *height, int *count, unsigned char *buf)
{
	struct console_font_op cfo;

	ensure_console();

	cfo.op = KD_FONT_OP_GET;
	cfo.flags = 0;
	cfo.width = cfo.height = 32;
	cfo.charcount = *count;
	cfo.data = buf;
	if (ioctl(console_fd, KDFONTOP, &cfo) == 0) {
		*count = cfo.charcount;
		if (height)
			*height = cfo.height;
		if (width)
			*width = cfo.width;
		return 0;
	}

	/* don't bother supporting older font ioctls (GIO_FONTX, GIO_FONT);
	 * usplash is vanishingly unlikely to be used on such old kernels
	 */
	fprintf(stderr, "usplash: can't get console font: %s\n",
		strerror(errno));
	return -1;
}

int set_font(int width, int height, int count, unsigned char *buf)
{
	struct console_font_op cfo;

	ensure_console();

	cfo.op = KD_FONT_OP_SET;
	cfo.flags = 0;
	cfo.width = width;
	cfo.height = height;
	cfo.charcount = count;
	cfo.data = buf;
	if (ioctl(console_fd, KDFONTOP, &cfo) == 0)
		return 0;
	/* if the count is not 256 or 512, round up and try again */
	if (errno == EINVAL && width == 8 && count != 256 && count < 512) {
		int ct = ((count > 256) ? 512 : 256);
		unsigned char *mybuf = malloc(32 * ct);
		int ret;

		if (!mybuf) {
			fprintf(stderr, "usplash: out of memory\n");
			return -1;
		}
		memset(mybuf, 0, 32 * ct);
		memcpy(mybuf, buf, 32 * count);
		cfo.data = mybuf;
		cfo.charcount = ct;
		ret = ioctl(console_fd, KDFONTOP, &cfo);
		free(mybuf);
		if (ret == 0)
			return 0;
	}

	/* don't bother supporting older font ioctls (PIO_FONTX, PIO_FONT);
	 * usplash is vanishingly unlikely to be used on such old kernels
	 */
	fprintf(stderr, "usplash: can't set console font: %s\n",
		strerror(errno));
	return -1;
}

void usplash_save_font(void)
{
	int width, height, count = 0;
	int size;
	unsigned char *data;

	if (get_font(&width, &height, &count, NULL) != 0 || count == 0)
		return;
	/* size calculation from linux/drivers/char/vt.c:con_font_get() */
	size = (width + 7) / 8 * 32 * count;
	data = malloc(size);
	if (!data) {
		fprintf(stderr, "usplash: out of memory\n");
		return;
	}
	if (get_font(&width, &height, &count, data) != 0)
		return;

	saved_font.width = width;
	saved_font.height = height;
	saved_font.charcount = count;
	saved_font.data = data;
}

void usplash_restore_font(void)
{
	if (!saved_font.data)
		return;
	set_font(saved_font.width, saved_font.height,
		 saved_font.charcount, saved_font.data);
}

int usplash_theme_width(struct usplash_theme* theme)
{
	return theme->theme_width ? theme->theme_width : theme->pixmap->width;
}

int usplash_theme_height(struct usplash_theme* theme)
{
	return theme->theme_height ? theme->theme_height : theme->pixmap->height;
}

int usplash_sanity_check_theme(struct usplash_theme* theme)
{
	USPLASH_THEME_ASSERT(theme->theme_width && theme->text_x + theme->text_width > theme->theme_width);
	USPLASH_THEME_ASSERT(theme->theme_height && theme->text_y + theme->text_height > theme->theme_height);

	USPLASH_THEME_ASSERT(theme->theme_width && theme->progressbar_x + theme->progressbar_width > theme->theme_width);
	USPLASH_THEME_ASSERT(theme->theme_height && theme->progressbar_y + theme->progressbar_height > theme->theme_height);
	return 0;
}

int usplash_setup(int xres, int yres, int v)
{
	int ret;
	short ncolors;
	void *theme_handle;
	struct usplash_theme *htheme;
	int maxarea;
	usplash_ratio ratio;
	unsigned char *palette;
	struct termios t;

	usplash_setup_funcs();
	ensure_console();

	verbose = v;
        verbose_default = verbose;
	theme_handle = dlopen(USPLASH_THEME, RTLD_LAZY);
	if (theme_handle) {
		theme = dlsym(theme_handle, "usplash_theme");
		if ((theme == NULL) || (theme->version != THEME_VERSION)) {
			dlclose(theme_handle);
			theme = &testcard_theme;
		}
	} else {
		theme = &testcard_theme;
	}

	/* If xres or yres is 0, try reading resolution of the framebuffer */
	if (xres == 0 || yres == 0) {
		FILE *vsz;

		vsz = fopen("/sys/class/drm/card0/device/graphics/fb0/virtual_size", "r");
		if (! vsz)
			vsz = fopen("/sys/class/graphics/fb0/virtual_size", "r");
		if (vsz) {
			xres = yres = 0;
			fscanf(vsz, "%d,%d", &xres, &yres);
			fclose(vsz);
		}
	}

	/* If xres or yres is still 0, use resolution from the config file. */
	if (xres == 0 || yres == 0) {
		FILE *conf;
		char line[1024];
		conf = fopen("/etc/usplash.conf", "r");
		if (conf) {
			xres = yres = 0;
			while (!feof(conf) && 
			       fgets(line, sizeof(line), conf)) {
				sscanf(line, "xres = %d", &xres);
				sscanf(line, "yres = %d", &yres);
			}
		} else {
		  /* Pick the first one, which by convention is
			   the lowest resolution one. */
			xres = usplash_theme_width(theme);
			yres = usplash_theme_height(theme);
		}
	}

	ret = usplash_set_resolution(xres, yres);
	if (ret)
		return ret;
	ret = usplash_init();
	if (ret)
		return ret;
	/* usplash_init might have changed the resolution */
	usplash_getdimensions(&xres, &yres);
	usplash_xres = xres;
	usplash_yres = yres;

	/* Select theme from linked list */
	htheme = NULL;
	maxarea = 0;
	ratio =
	    (float) xres / (float) yres >
	    1.55 ? USPLASH_16_9 : USPLASH_4_3;

	while (theme) {
		if (usplash_theme_height(theme) <= yres
		    && usplash_theme_width(theme) <= xres
		    && usplash_theme_height(theme) * usplash_theme_width(theme) > maxarea
		    && theme->ratio == ratio
		    && usplash_sanity_check_theme(theme) == 0) {
			maxarea = usplash_theme_height(theme) * usplash_theme_width(theme);
			htheme = theme;
		}
		theme = theme->next;
	}
	theme = htheme;
	if (!theme) {
		fprintf(stderr,
			"usplash: No usable theme found for %dx%d\n",
			xres, yres);
		return 1;
	}

	ncolors = theme->pixmap->ncols;
	if (theme->init)
		theme->init(theme);

	left_edge = (usplash_xres - usplash_theme_width(theme)) / 2;
	top_edge = (usplash_yres - usplash_theme_height(theme)) / 2;
	text_x1 = left_edge + theme->text_x;
	text_y1 = top_edge + theme->text_y;
	text_x2 = text_x1 + theme->text_width;
	text_y2 = text_y1 + theme->text_height;
	text_position = text_x1;

	if (theme->font)
		usplash_setfont(theme->font);

	/* clear palette (will be faded in later) */
	palette = calloc (theme->pixmap->ncols, 3);
	if (!palette) {
	    fprintf (stderr, "usplash: Could not allocate palette\n");
	    exit (1);
	}
	usplash_set_palette(theme->pixmap->ncols, palette);
	free (palette);

	alarm_fds[0] = alarm_fds[1] = -1;
	pipe(alarm_fds);
	animation_needed_fd = alarm_fds[0];
	if (fcntl(animation_needed_fd, F_SETFL, O_NONBLOCK)<0) {
		perror("fcntl");
	}

	if (fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK) < 0)
		perror("fcntl");
	/* configure unbuffered input compatible with select */
	tcgetattr(STDIN_FILENO, &t);
	t.c_cc[VMIN] = 1;
	t.c_cc[VTIME] = 0;
	t.c_lflag &= ~ICANON;
	tcsetattr(STDIN_FILENO, TCSANOW, &t);

	sigemptyset(&sigs);
	sigaddset(&sigs, SIGALRM);
	return 0;
}

size_t strncspn(const char *s, size_t n, const char *reject)
{
	register size_t l;

	for (l = 0; l < n; l++)
		if (strchr(reject, s[l]))
			break;

	return l;
}

void clear_screen(void)
{
	blocksig();
	if (theme->clear_screen)
		theme->clear_screen(theme);
	else {
		usplash_clear(0, 0, usplash_xres, usplash_yres,
			      theme->background);
		usplash_put(left_edge + theme->pixmap_x, top_edge +
			theme->pixmap_y, theme->pixmap);
	}
	unblocksig();
}

static
void dim_logo(int percentage, int restore_palette)
{ 
	unsigned char palette[256][3];
	int i, j;

	for (i = theme->pixmap->ncols - 1; i >= 0; --i)
		for (j = 0; j < 3; ++j)
			palette[i][j] = theme->pixmap->palette[i][j] * percentage / 100;
	usplash_set_palette(theme->pixmap->ncols, palette);

	usplash_put(left_edge + theme->pixmap_x, top_edge +
		theme->pixmap_y, theme->pixmap);

	if (restore_palette)
		usplash_set_palette(theme->pixmap->ncols, theme->pixmap->palette);
}

void fade_logo(int on, int step)
{
	int pct;
	struct timespec wait;
	struct timespec remaining;

	if (on && theme->clear_screen) {
		usplash_set_palette(theme->pixmap->ncols, theme->pixmap->palette);
		theme->clear_screen(theme);
		return;
	}

	wait.tv_sec = 0;
	wait.tv_nsec = 20000000;
	if (on) {
		for (pct = 0; pct <= 100; pct += step) {
			dim_logo(pct, 0);
			blocksig();
			nanosleep(&wait, &remaining);
			unblocksig();
		}
	} else {
		for (pct = 100; pct >= 0; pct -= step) {
			dim_logo(pct, 0);
			blocksig();
			nanosleep(&wait, &remaining);
			unblocksig();
		}
	}
}

void pulse_logo(int num_cycles)
{
	static int cycle = 0;

	int pct;

	/* in the first half of the cycle, we pulsate from 100% to 50%, and back
	   to 100% in the second half */
	if (cycle < num_cycles/2)
		pct = 100 - (cycle*100/num_cycles);
	else
		pct = cycle*100/num_cycles;

	dim_logo(pct, 1);

	if (++cycle >= num_cycles)
	    cycle = 0;
}

void clear_progressbar(void)
{
	if (theme->clear_progressbar)
		theme->clear_progressbar(theme);
	else {
		int x1, y1, x2, y2;

		x1 = left_edge + theme->progressbar_x;
		y1 = top_edge + theme->progressbar_y;

		x2 = x1 + theme->progressbar_width;
		y2 = y1 + theme->progressbar_height;

		usplash_clear(x1, y1, x2, y2,
			      theme->progressbar_background);
	}
}

void draw_progressbar(int percentage)
{
	if (percentage > 100 || percentage < -100)
		return;

	blocksig();
	if (theme->draw_progressbar)
		theme->draw_progressbar(theme, percentage);
	else {
		int x1, y1, x2, y2, xx, bg, fg;

		if (percentage < 0) {
			bg = theme->progressbar_foreground;
			fg = theme->progressbar_background;
			percentage = -percentage;
		} else {
			bg = theme->progressbar_background;
			fg = theme->progressbar_foreground;
		}

		x1 = left_edge + theme->progressbar_x;
		y1 = top_edge + theme->progressbar_y;

		x2 = x1 + theme->progressbar_width;
		y2 = y1 + theme->progressbar_height;

		xx = x1 + ((theme->progressbar_width * percentage) / 100);

		usplash_clear(x1, y1, xx, y2, fg);
		usplash_clear(xx, y1, x2, y2, bg);
	}
	unblocksig();
}


void clear_text(void)
{
	blocksig();
	if (theme->clear_text)
		theme->clear_text(theme);
	else {
		int x1, y1, x2, y2;

		x1 = left_edge + theme->text_x;
		y1 = top_edge + theme->text_y;

		x2 = x1 + theme->text_width;
		y2 = y1 + theme->text_height;

		usplash_clear(x1, y1, x2, y2, theme->text_background);
	}
	unblocksig();
}

void draw_text_urgent(const char *string, size_t len)
{
	static int inited = 0;
	int ov = verbose;

	verbose = 1;

	if (inited == 0 && ov == 0)
		clear_text();
	inited = 1;

	draw_text(string, len);
	verbose = ov;
}

/* Moves all text one line up and positions cursor at beginning of line */
static void draw_newline(void)
{
	/* Move existing text up */
	usplash_move(text_x1,
		     top_edge + theme->text_y + theme->line_height,
		     text_x1, top_edge + theme->text_y, theme->text_width,
		     theme->text_height - theme->line_height);

	usplash_clear(text_x1, text_y2 - theme->line_height,
		      text_x2, text_y2, theme->text_background);

	/* Reset "cursor" position */
	text_position = text_x1;
}

/* Continues to draw text at current position */
static void draw_chars(const char *string, size_t len)
{
	int i, slen;
	size_t drawn;

	drawn = 0;
	while (drawn < len) {
		/* See how many characters we can draw on this line */
		slen = 0;
		for (i = 0; i + drawn < len; i++) {
			slen +=
			    usplash_getfontwidth(*(string + i + drawn));
			if (text_position + slen > text_x2)
				break;
		}

		/* Ok, draw them and move on to the next line */
		usplash_text(text_position, text_y2 - theme->line_height,
			     string + drawn, i,
			     theme->text_foreground,
			     theme->text_background);

		text_position += slen;
		drawn += i;
		if (drawn == len)
			break;
		else
			draw_newline();
	}
}

/* Adds a newline and draws one line of text */
void draw_text(const char *string, size_t len)
{
	if (!verbose)
		return;

	blocksig();
	if (theme->draw_text) {
		theme->draw_text(theme, string, len);
	} else {
		draw_newline();
		draw_chars(string, len);
	}
	unblocksig();
}

void flush_stdin()
{
	struct pollfd input_fd = {
		.fd = STDIN_FILENO,
		.events = POLLIN,
	};

	while((poll( &input_fd, 1 , 100) > 0)
	      && (getchar () != EOF))
		;
}

char get_timeout_char(int inputtimeout)
{
	char charinput = -1;
	struct pollfd fds[2];
	fds[0].fd = STDIN_FILENO;
	fds[0].events = POLLIN;
	fds[1].fd = animation_needed_fd;
	fds[1].events = POLLIN;
        struct termios t, orig_t;

        /* configure unbuffered input */
        tcgetattr(STDIN_FILENO, &t);
        orig_t = t;
        t.c_cc[VMIN] = 0;
        t.c_cc[VTIME] = 0;
        t.c_lflag &= ~ICANON;
        tcsetattr(STDIN_FILENO, TCSANOW, &t);

	while( inputtimeout )
	{
		if( poll( fds, 2 , 100) > 0 )
		{
			if ( fds[0].revents & POLLIN ) {
				if( read(STDIN_FILENO,&charinput,1) < 0 )
					charinput = '\0';
				inputtimeout = 1; // like "break", but unblocksig-safe
			}
			if ( fds[1].revents & POLLIN ) {
				process_pending_animation_steps();
			}
		}
		if (inputtimeout > 0) inputtimeout--;
	}
	tcsetattr(STDIN_FILENO, TCSANOW, &orig_t);
	return charinput;
}


int usplash_timeout_get_string(char *inputbuf, int length, int quiet,int inputtimeout)
{
	char input;
	int i;
	
	/* Get user input */
	for (i = 0; i < length - 1; i++) {
		input = get_timeout_char(inputtimeout);
		if ( input == -1 )
			break;
		if (input == '\n' || input == '\r' || input == '\0')
			break;

                /* backspace */
                if (input == '\x7F') {
                    if (i > 0) {
                            int p, w;

                            w = usplash_getfontwidth(quiet == 1 ? '*' : inputbuf[i-1]);
                            text_position -= w;
                            p = text_position;
                            draw_chars("  ", 2);
                            text_position = p;
                            --i;
                    }
                    --i;
                    continue;
                }
		
		if (quiet == 2) {
			i--;
			continue;
		}

		inputbuf[i] = input;

		if (quiet)
			input = '*';

		draw_chars(&input, 1);
	}
	inputbuf[i] = '\0';
	return i;
}

int handle_input(const char *string, const size_t len, const int quiet)
{
	return handle_timeout_input(string,len,quiet,-1);
}

static int _open_out_fifo()
{
	int fifo_fd = -1;
	int sec;
	int deci;
	struct timespec needed;
	struct timespec remaining;

	/* We wait for timeout seconds for someone to read the user input */
	for (sec = 0; timeout == 0 || sec < timeout; sec++) {
		/* allow deci-second precision to avoid visible delays */
		for (deci = 0; deci < 10; deci++) {
			needed.tv_sec = 0;
			needed.tv_nsec = 100000000;
			fifo_fd = open(USPLASH_OUTFIFO, O_WRONLY | O_NONBLOCK | O_NOCTTY);
			if (fifo_fd < 0) {
				/* protect ourselves from signals */
				while (nanosleep(&needed,&remaining) < 0) {
					if (errno != EINTR)
						return -1;
					needed = remaining;
				}
			}
			else
				return fifo_fd;
		}
	}

	return fifo_fd;
}

/* handle input with a timeout in seconds. timeout = -1 means no timeout */
int handle_timeout_input(const char *string, const size_t len, const int quiet,int inputtimeout)
{
	ssize_t wlen;
	int fifo_outfd = -1;
	int reset_verbose = 0;
	char inputbuf[PIPE_BUF];
	int err = 0;

	/* need to be verbose for string output */
	if (!verbose) {
		reset_verbose = 1;
		verbose = 1;
	}

	/* draw the prompt */
	draw_text(string, len);

	/* Get the user input */
	usplash_timeout_get_string(inputbuf, PIPE_BUF, quiet, inputtimeout);

	/* Reset the verbose flag */
	if (reset_verbose)
		verbose = 0;

	fifo_outfd = _open_out_fifo();

	if (fifo_outfd < 0) {
		err = 1;
		goto out;
	}

	wlen = write(fifo_outfd, inputbuf, strlen(inputbuf) + 1);
	if (wlen < 0)
		err = 1;

out:
	if (fifo_outfd >= 0)
		close(fifo_outfd);
	memset(inputbuf, 0, PIPE_BUF);
	return err;
}

/* Non-blocking check for a single character key press; Return ASCII code to
 * FIFO, or an empty string if there is no pending key press. */
int handle_input_char()
{
        char ch;
	int fifo_outfd = -1;
	int err = 0;

        ch = get_timeout_char(1);

	fifo_outfd = _open_out_fifo();

	if (fifo_outfd < 0) {
		err = 1;
		goto out;
	}

        if (ch != -1 && ch != '\0')
            if (write(fifo_outfd, &ch, 1) < 1)
		err = 1;

out:
	if (fifo_outfd >= 0)
		close(fifo_outfd);
	return err;
}

int handle_verbose (int mode)
{
    int err = 0;
    switch(mode) {
        case 2:
            verbose = verbose_default;
            break;
        case 1:
            verbose = 1;
            break;
        case 0:
            verbose = 0;
            break;
        default:
            fprintf(stderr, "usplash: invalid verbosity mode %i\n", mode);
            err = 1;
    }

    return err;
}

void draw_status(const char *string, size_t len, int mode)
{
	if (!verbose)
		return;

	blocksig();
	if (theme->draw_status) {
		theme->draw_status(theme, string, len, mode);
	} else {
		int x1, y1, fg;

		if (mode < 0)
			fg = theme->text_failure;
		else if (mode > 0)
			fg = theme->text_success;
		else
			fg = theme->text_foreground;

		x1 = text_x2 - theme->status_width;
		y1 = text_y2 - theme->line_height;

		usplash_clear(x1, y1, text_x2, text_y2,
			      theme->text_background);
		usplash_text(x1, y1, string, len, fg,
			     theme->text_background);
	}
	unblocksig();
}

void set_pulsating(int pulsate_enabled)
{
	default_pulsate_mode = pulsate_enabled;
}

void request_animate_step()
{
	static char dummy = '\0';
	write(alarm_fds[1], &dummy, 1);
}

/* catch up on any pending animation steps */
void process_pending_animation_steps()
{
	char dummy;
	int bytes;
	for (;;) {
		bytes = read(animation_needed_fd, &dummy, 1);
		if (bytes == 1) {
			 animate_step(-1);
		}
		else if (bytes < 0) {
			if (errno == EINTR) continue;
			/* Treat expected EAGAIN and unexpected other errno
			   values the same way: stop the animation cycle. */
			return;
		}
		else {
			/* Something unexpected has happened: EOF, large read */
			return;
		}
	}
}

/*
  pulsating =  1  throbber should be drawn
  pulsating =  0  throbber should not be drawn
  pulsating = -1  fall back to pulsate value set via set_pulsating()
*/
void animate_step(int pulsating)
{
	static int pulsate_step = 0;
	static int num_steps = 37;
	int x1, y1, x2, y2;

	/* fall back to externally defined pulsate mode */
	if (pulsating < 0) {
		pulsating = default_pulsate_mode;
	}

	if (theme->animate_step)
		theme->animate_step(theme, pulsating);
	else {
		if (pulsating) {
			clear_progressbar();

			if (pulsate_step < 19)
				x1 = left_edge + theme->progressbar_x +
				    (theme->progressbar_width / 20) *
				    pulsate_step;
			else
				x1 = left_edge + theme->progressbar_x +
				    (theme->progressbar_width / 20) * (36 -
								       pulsate_step);

			y1 = top_edge + theme->progressbar_y;

			x2 = x1 + (theme->progressbar_width / 10);
			y2 = y1 + theme->progressbar_height;
			usplash_clear(x1, y1, x2, y2,
				      theme->progressbar_foreground);

			pulsate_step = (pulsate_step + 1) % num_steps;
		}
	}
}
