/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "view.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include <visu_tools.h>
#include <visu_configFile.h>
#include <visu_data.h>
#include <coreTools/toolMatrix.h>

/* Global value used for precision. */
static float precisionOfRendering;
/* Global value used as default when creating a new OpenGLView. */
static float anglesDefault[3] = {40., -50., 0.};
static float translatDefault[2] = {0.5, 0.5};
static float grossDefault = 1.;
static float perspDefault = 5.;




/* Change the position of the observer. */
static void modelize(OpenGLView *view);
/* Change the OpenGL viewport. */
static void project(OpenGLView *view);


/* Change the value of the theta and phi angles. */
int openGLViewSet_thetaPhiOmega(OpenGLView *view, float valueTheta,
				  float valuePhi, float valueOmega, int mask)
{
  float valT, valP, valO;
  int idem;

  g_return_val_if_fail(view && view->camera, 0);
  
  idem = 1;
  if (mask & MASK_THETA)
    {
      valT = valueTheta;
      while (valT < -180.)
	valT += 360.;
      while (valT > 180.)
	valT -= 360.;

      if (view->camera->theta != valT)
	{
	  idem = 0;
	  view->camera->theta = valT;
	}
    }
  if (mask & MASK_PHI)
    {
      valP = valuePhi;
      while (valP < -180.)
	valP += 360.;
      while (valP > 180.)
	valP -= 360.;

      if (view->camera->phi != valP)
	{
	  idem = 0;
	  view->camera->phi = valP;
	}
    }
  if (mask & MASK_OMEGA)
    {
      valO = valueOmega;
      while (valO < -180.)
	valO += 360.;
      while (valO > 180.)
	valO -= 360.;

      if (view->camera->omega != valO)
	{
	  idem = 0;
	  view->camera->omega = valO;
	}
    }
  if (idem)
    return 0;

  modelize(view);
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLThetaPhi_signal_id, */
/* 		 0 , NULL); */

  return 1;
}

/* Change the value of the camera position. */
int openGLViewSet_XsYs(OpenGLView *view, float valueX, float valueY, int mask)
{
  float valX, valY;
  int idem;
  
  g_return_val_if_fail(view && view->camera, 0);
  
  idem = 1;
  if (mask & MASK_XS)
    {
      valX = valueX;
      if (valX < -3.)
	valX = -3.;
      while (valX > 3.)
	valX = 3.;

      if (view->camera->xs != valX)
	{
	  idem = 0;
	  view->camera->xs = valX;
	}
    }
  if (mask & MASK_YS)
    {
      valY = valueY;
      while (valY < -3.)
	valY = -3.;
      while (valY > 3.)
	valY = 3.;

      if (view->camera->ys != valY)
	{
	  idem = 0;
	  view->camera->ys = valY;
	}
    }
  if (idem)
    return 0;

/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLXsYs_signal_id, */
/* 		 0 , NULL); */
  project(view);

  return 1;
}

/* Change the value of the camera zoom value. */
int openGLViewSet_gross(OpenGLView *view, float value)
{
  float val;
  
  g_return_val_if_fail(view && view->camera, 0);
  
  val = value;
  if (val < 0.02)
    val = 0.02;
  else if (val > 999.)
    val = 999.;

  if (view->camera->gross == val)
    return 0;

  view->camera->gross = val;
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLGross_signal_id, */
/* 		 0 , NULL); */
  project(view);
  
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
/* 		 0 , NULL); */
  return 1;
}
/* Change the value of the camera perspective value. */
int openGLViewSet_persp(OpenGLView *view, float value)
{
  float val;
  
  g_return_val_if_fail(view && view->camera, 0);
  
  val = value;
  if (val < 1.1)
    val = 1.1;
  else if (val > 1000.)
    val = 1000.;

  if (view->camera->d_red == val)
    return 0;

  view->camera->d_red = val;
  view->camera->gr = val / (val - 1.0);
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLDRed_signal_id, */
/* 		 0 , NULL); */
/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
/* 		 0 , NULL); */
  project(view);
  modelize(view);
  return 1;
}
int OpenGLViewSet_windowSize(OpenGLView *view, guint width, guint height)
{
  DBG_fprintf(stderr, "OpenGL View : set viewport size (%dx%d) for view %p.\n",
	      width, height, (gpointer)view);

  if (!view)
    {
      glViewport(0, 0, width, height);
      return 0;
    }
  if (view->window->width != width || view->window->height != height)
    {
      view->window->width = width;
      view->window->height = height;
      glViewport(0, 0, view->window->width, view->window->height);
      project(view);
      /*       g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLWidthHeight_signal_id, */
      /* 		     0 , NULL); */
      /*       g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
      /* 		     0 , NULL); */
      return 1;
    }
  return 0;
}


static void modelize(OpenGLView *view)
{
  double theta_rad;
  double phi_rad; 
  double sth, cth, sph, cph, com, som;
  double eyex, eyey, eyez;
  double distance;   
  double upx, upy, upz;

  g_return_if_fail(view && view->camera && view->box);
 
  theta_rad = view->camera->theta * PI180;
  phi_rad   = view->camera->phi   * PI180; 

  distance = view->camera->d_red * view->box->extens;

  sth = sin(theta_rad);
  cth = cos(theta_rad);      
  sph = sin(phi_rad);
  cph = cos(phi_rad);
  com = cos(view->camera->omega * PI180);
  som = sin(view->camera->omega * PI180);

  /* La matrice de rotation est la suivante pour passer
     des coordonnes transformes aux coordonnes de l'cran :
     /cph.cth -sph cph.sth\
     |sph.cth  cph sph.sth|
     \   -sth   0      cth/
     Ainsi la camra qui est situ en (0,0,Distance) dans le repre transform
     devient dans le repre de l'cran : */
  eyex = distance*sth*cph;
  eyey = distance*sth*sph;
  eyez = distance*cth;
   
  /* Vecteur donnant la direction verticale.
     Dans le repre transform il est (-1,0,0). */
  upx = -cth*cph*com + sph*som;
  upy = -cth*sph*com - cph*som;
  upz = sth*com;
   
  glMatrixMode(GL_MODELVIEW); 
  glLoadIdentity();
  gluLookAt(eyex, eyey, eyez, 0.0, 0.0, 0.0, upx, upy, upz);
}

static void project(OpenGLView *view)
{ 
  double x, y, xmin, xmax, ymin, ymax, fact, rap;
  double rap_win;
 
  g_return_if_fail(view && view->camera && view->box && view->window);
 
  view->window->near = (view->camera->d_red - 1.0) * view->box->extens;
  view->window->far  = (view->camera->d_red + 1.0) * view->box->extens;

/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLNearFar_signal_id, */
/* 		 0 , NULL); */

  fact = view->window->near / view->camera->gross / view->camera->d_red;
  rap = 2. * view->window->near / (view->camera->d_red - 1.);
  rap_win = (1.0*view->window->height)/view->window->width;
  x = (0.5 - view->camera->xs) * rap;
  if (rap_win < 1.)
    x /= rap_win;
  xmin = x - fact;
  xmax = x + fact;
  y = (0.5 - view->camera->ys) * rap;
  if (rap_win > 1.)
    y /= rap_win;
  ymin = y - fact;
  ymax = y + fact;
  view->window->left   = xmin;
  view->window->bottom = ymin;
    
  if ( 1. > rap_win )
    {
      view->window->top   = ymax;
      fact                = (ymax - ymin) / rap_win;
      view->window->left  = 0.5 * (xmin + xmax - fact);
      view->window->right = 0.5 * (xmin + xmax + fact);
    }
  else if ( 1. < rap_win )
    {
      view->window->right  = xmax;
      fact                 = (xmax - xmin) * rap_win;
      view->window->bottom = 0.5 * (ymin + ymax - fact);
      view->window->top    = 0.5 * (ymin + ymax + fact);
    }
  else
    {
      view->window->right  = xmax;
      view->window->top    = ymax;
    }

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(view->window->left, view->window->right, view->window->bottom,
	    view->window->top, view->window->near, view->window->far);
  glMatrixMode(GL_MODELVIEW);
}
void openGLViewCompute_matrixAndView(OpenGLView* view)
{
  g_return_if_fail(view);

  DBG_fprintf(stderr, "OpenGL View : modelize and project view %p.\n", (gpointer)view);
  modelize(view);
  project(view);
}

OpenGLView* OpenGLViewNew()
{
  OpenGLView *view;

  view         = g_malloc(sizeof(OpenGLView));
  view->camera = g_malloc(sizeof(OpenGLCamera));
  view->box    = g_malloc(sizeof(OpenGLBox));
  view->window = g_malloc(sizeof(OpenGLWindow));

  view->window->width  = 100.;
  view->window->height = 100.;
  view->window->near   = 0.;
  view->window->far    = 0.;

  view->camera->theta = anglesDefault[0];
  view->camera->phi   = anglesDefault[1];
  view->camera->omega = anglesDefault[2];
  view->camera->xs    = translatDefault[0];
  view->camera->ys    = translatDefault[1];
  view->camera->gross = grossDefault;
  view->camera->d_red = perspDefault;
  view->camera->gr    = view->camera->d_red / (view->camera->d_red - 1.);

  DBG_fprintf(stderr, "OpenGL view : create a new view (%p).\n", (gpointer)view);
  DBG_fprintf(stderr, " | theta = %g\n", view->camera->theta);
  DBG_fprintf(stderr, " | phi   = %g\n", view->camera->phi);
  DBG_fprintf(stderr, " | omega = %g\n", view->camera->omega);
  DBG_fprintf(stderr, " | dx-dy = %g %g\n", view->camera->xs, view->camera->ys);
  DBG_fprintf(stderr, " | gross = %g\n", view->camera->gross);
  DBG_fprintf(stderr, " | persp = %g\n", view->camera->d_red);
  DBG_fprintf(stderr, " | width x height = %d x %d\n", view->window->width, view->window->height);
  return view;
}
void OpenGLViewFree(OpenGLView *view)
{
  g_return_if_fail(view);

  if (view->camera)
    g_free(view->camera);
  if (view->window)
    g_free(view->window);
  if (view->box)
    g_free(view->box);
  g_free(view);
}
OpenGLView* OpenGLViewCopy(OpenGLView *view)
{
  OpenGLView *viewCopy;
  int i;

  g_return_val_if_fail(view, (OpenGLView*)0);

  viewCopy         = g_malloc(sizeof(OpenGLView));
  viewCopy->camera = g_malloc(sizeof(OpenGLCamera));
  viewCopy->box    = g_malloc(sizeof(OpenGLBox));
  viewCopy->window = g_malloc(sizeof(OpenGLWindow));

  viewCopy->window->width  = view->window->width;
  viewCopy->window->height = view->window->height;
  viewCopy->window->near   = view->window->near;
  viewCopy->window->far    = view->window->far;
  viewCopy->window->left   = view->window->left;
  viewCopy->window->right  = view->window->right;
  viewCopy->window->bottom = view->window->bottom;
  viewCopy->window->top    = view->window->top;

  viewCopy->camera->theta = view->camera->theta;
  viewCopy->camera->phi   = view->camera->phi;
  viewCopy->camera->omega = view->camera->omega;
  viewCopy->camera->xs    = view->camera->xs;
  viewCopy->camera->ys    = view->camera->ys;
  viewCopy->camera->gross = view->camera->gross;
  viewCopy->camera->d_red = view->camera->d_red;
  viewCopy->camera->gr    = view->camera->gr;
  
  viewCopy->box->extens = view->box->extens;
  viewCopy->box->dxxs2  = view->box->dxxs2;
  viewCopy->box->dyys2  = view->box->dyys2;
  viewCopy->box->dzzs2  = view->box->dzzs2;
  for (i = 0; i < 3; i++)
    {
      viewCopy->box->p1[i]  = view->box->p1[i];
      viewCopy->box->p2[i]  = view->box->p2[i];
      viewCopy->box->p3[i]  = view->box->p3[i];
      viewCopy->box->p4[i]  = view->box->p4[i];
      viewCopy->box->p5[i]  = view->box->p5[i];
      viewCopy->box->p6[i]  = view->box->p6[i];
      viewCopy->box->p7[i]  = view->box->p7[i];
      viewCopy->box->p8[i]  = view->box->p8[i];
    }

  return viewCopy;
}

/* This is a function to get the number of facettes advised
   by the server (according to its policy on rendering)
   to draw an object according to a given dimension. */
int OpenGLViewGet_numberOfFacettes(OpenGLView *view, float dimension)
{
  int rsize;     
  int nlat;
#define NLAT_MIN 12
#define NLAT_MAX 50
#define RSIZE_MIN  10
#define RSIZE_MAX 250

#define NLAT_V_MIN 0
#define NLAT_V_MAX (NLAT_MIN)
#define RSIZE_V_MIN  0
#define RSIZE_V_MAX (RSIZE_MIN)

#define NLAT_MINI 3
#define NLAT_MAXI 100

  static float fac = -1.0f, fac_v = -1.0f;

  g_return_val_if_fail(view && view->camera && view->window && view->box, -1);

/*   if (!visuCamera->gr || !visuCamera->gross) */
/*     return -1; */
/*   if (dimension <= 0.) */
/*     return -1; */

  /* calculate once fac and fac_v!... */
  if(fac < 0.0f) {
    fac = ((float)(NLAT_MAX - NLAT_MIN))/(RSIZE_MAX - RSIZE_MIN);
    fac_v = ((float)(NLAT_V_MAX - NLAT_V_MIN))/(RSIZE_V_MAX - RSIZE_V_MIN);
  }

  rsize = (int)(view->window->width * (dimension /
				       (2.0 * view->box->extens / view->camera->gr /
					view->camera->gross)));
  
  if(rsize < RSIZE_MIN) {
    nlat = (int)(NLAT_V_MIN + fac_v * (rsize - RSIZE_V_MIN));
    if(nlat < NLAT_MINI) nlat = NLAT_MINI;
  }
  else if(rsize > RSIZE_MAX) {
    nlat = NLAT_MAX;
  }
  else {
    nlat = (int)(NLAT_MIN + fac * (rsize - RSIZE_MIN));
  }

  nlat = (int)((float)nlat * precisionOfRendering);
  if (nlat < NLAT_MINI)
    nlat = NLAT_MINI;
  if (nlat > NLAT_MAXI)
    nlat = NLAT_MAXI;
   
  return nlat;
}
/* This function change the value of the parameter precisionOfRendering. */
int OpenGLViewSet_precision(float value)
{
  if (value <= 0. || value == precisionOfRendering)
    return 0;
  
  precisionOfRendering = value;

/*   g_signal_emit (visu, VISU_GET_CLASS (visu)->OpenGLFacetteChanged_signal_id, */
/* 		 0 , NULL); */
  return 1;
}
/* This function retrieve the value of the parameter precisionOfRendering. */
float OpenGLViewGet_precision()
{
  return precisionOfRendering;
}
float OpenGLViewGet_fileUnitPerPixel(OpenGLView *view)
{
  float deltaH, deltaV;

  g_return_val_if_fail(view, 0.);

  deltaH = view->window->right - view->window->left;
  deltaV = view->window->top - view->window->bottom;
  if (deltaH < deltaV)
    return deltaH / (float)view->window->width;
  else
    return deltaV / (float)view->window->height;
}

void OpenGLViewGet_screenAxes(OpenGLView *view, float xAxis[3], float yAxis[3])
{
  double cth, sth, cph, sph, com, som;
  float matPhi[3][3], matTheta[3][3], matOmega[3][3];
  float matRes[3][3], matRes2[3][3];
  float axis[3];

  g_return_if_fail(view);

  cth = cos(view->camera->theta * PI180);
  sth = sin(view->camera->theta * PI180);
  cph = cos(view->camera->phi * PI180);
  sph = sin(view->camera->phi * PI180);
  com = cos(view->camera->omega * PI180);
  som = sin(view->camera->omega * PI180);

  matPhi[0][0] = cph;
  matPhi[1][0] = sph;
  matPhi[2][0] = 0.;
  matPhi[0][1] = -sph;
  matPhi[1][1] = cph;
  matPhi[2][1] = 0.;
  matPhi[0][2] = 0.;
  matPhi[1][2] = 0.;
  matPhi[2][2] = 1.;

  matTheta[0][0] = cth;
  matTheta[1][0] = 0.;
  matTheta[2][0] = -sth;
  matTheta[0][1] = 0.;
  matTheta[1][1] = 1.;
  matTheta[2][1] = 0.;
  matTheta[0][2] = sth;
  matTheta[1][2] = 0.;
  matTheta[2][2] = cth;

  matOmega[0][0] = com;
  matOmega[1][0] = som;
  matOmega[2][0] = 0.;
  matOmega[0][1] = -som;
  matOmega[1][1] = com;
  matOmega[2][1] = 0.;
  matOmega[0][2] = 0.;
  matOmega[1][2] = 0.;
  matOmega[2][2] = 1.;

  matrix_productMatrix(matRes, matTheta, matOmega);
  matrix_productMatrix(matRes2, matPhi, matRes);

  axis[0] = 0.;
  axis[1] = 1.;
  axis[2] = 0.;
  matrix_productVector(xAxis, matRes2, axis);

  axis[0] = -1.;
  axis[1] = 0.;
  axis[2] = 0.;
  matrix_productVector(yAxis, matRes2, axis);
}

void openGLViewRotate_box(OpenGLView *view, float dTheta, float dPhi, float angles[2])
{
  g_return_if_fail(view && angles);

  angles[0] = view->camera->theta + dTheta;
  angles[1] = view->camera->phi + dPhi;
}
void openGLViewRotate_camera(OpenGLView *view, float dTheta, float dPhi, float angles[3])
{
  double cth, sth, cph, sph, com, som;
  double cdth, sdth, cdph, sdph;
  double Theta, Phi, Omega;
  #define RADTODEG 57.29577951
  float MinRprime[3], MinR[3];
  float Mspherical[3];
  float matPhi[3][3], matTheta[3][3], matOmega[3][3], matdPhi[3][3], matdTheta[3][3];
  float matPhiPrime[3][3], matThetaPrime[3][3];
  float matRprime2R[3][3];
  float matRes[3][3], matRes2[3][3];

  g_return_if_fail(view && angles);

  cth = cos(view->camera->theta * PI180);
  sth = sin(view->camera->theta * PI180);
  cph = cos(view->camera->phi * PI180);
  sph = sin(view->camera->phi * PI180);
  com = cos(view->camera->omega * PI180);
  som = sin(view->camera->omega * PI180);

  cdth = cos(dTheta * PI180);
  sdth = sin(dTheta * PI180);
  cdph = cos(dPhi * PI180);
  sdph = sin(dPhi * PI180);


  matPhi[0][0] = cph;
  matPhi[1][0] = sph;
  matPhi[2][0] = 0.;
  matPhi[0][1] = -sph;
  matPhi[1][1] = cph;
  matPhi[2][1] = 0.;
  matPhi[0][2] = 0.;
  matPhi[1][2] = 0.;
  matPhi[2][2] = 1.;

  matTheta[0][0] = cth;
  matTheta[1][0] = 0.;
  matTheta[2][0] = -sth;
  matTheta[0][1] = 0.;
  matTheta[1][1] = 1.;
  matTheta[2][1] = 0.;
  matTheta[0][2] = sth;
  matTheta[1][2] = 0.;
  matTheta[2][2] = cth;

  matOmega[0][0] = com;
  matOmega[1][0] = som;
  matOmega[2][0] = 0.;
  matOmega[0][1] = -som;
  matOmega[1][1] = com;
  matOmega[2][1] = 0.;
  matOmega[0][2] = 0.;
  matOmega[1][2] = 0.;
  matOmega[2][2] = 1.;

  matdPhi[0][0] = 1.;
  matdPhi[1][0] = 0.;
  matdPhi[2][0] = 0.;
  matdPhi[0][1] = 0.;
  matdPhi[1][1] = cdph;
  matdPhi[2][1] = -sdph;
  matdPhi[0][2] = 0.;
  matdPhi[1][2] = sdph;
  matdPhi[2][2] = cdph;

  matdTheta[0][0] = cdth;
  matdTheta[1][0] = 0.;
  matdTheta[2][0] = -sdth;
  matdTheta[0][1] = 0.;
  matdTheta[1][1] = 1.;
  matdTheta[2][1] = 0.;
  matdTheta[0][2] = sdth;
  matdTheta[1][2] = 0.;
  matdTheta[2][2] = cdth;

  matrix_productMatrix(matRes, matdPhi, matdTheta);
  matrix_productMatrix(matRes2, matOmega, matRes);
  matrix_productMatrix(matRes, matTheta, matRes2);
  matrix_productMatrix(matRprime2R, matPhi, matRes);

  MinRprime[0] = 0.;
  MinRprime[1] = 0.;
  MinRprime[2] = 1.;
  matrix_productVector(MinR, matRprime2R, MinRprime);
/*   fprintf(stderr, "M : %f %f %f -> %f\n", MinR[0], MinR[1], MinR[2], */
/* 	  MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]); */

/*   cartesian_to_spherical(Mspherical, MinR); */
  Mspherical[0] = sqrt(MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]);
  if (MinR[1] == 0 && MinR[0] == 0)
    {
      Mspherical[1] = (MinR[2] > 0.)?0.:180.;
      Mspherical[2] = view->camera->phi;
    }
  else
    {
      Mspherical[1] = acos(MinR[2] / Mspherical[0]) * RADTODEG;
      if (MinR[0] == 0.)
	Mspherical[2] = (MinR[1] > 0.)?90.:-90.;
      else
	{
	  Mspherical[2] = atan(MinR[1] / MinR[0]) * RADTODEG;
	  if (MinR[0] < 0.)
	    Mspherical[2] += 180.;
	}
    }
/*   fprintf(stderr, "avant %f %f\n", Mspherical[1], Mspherical[2]); */
  while (Mspherical[1] - view->camera->theta < -90.)
    Mspherical[1] += 360.;
  while (Mspherical[1] - view->camera->theta > 90.)
    Mspherical[1] -= 360.;
  while (Mspherical[2] - view->camera->phi < -90.)
    Mspherical[2] += 360.;
  while (Mspherical[2] - view->camera->phi > 90.)
    Mspherical[2] -= 360.;
/*   fprintf(stderr, "aprs %f %f\n", Mspherical[1], Mspherical[2]); */

  Theta = Mspherical[1];
  Phi = Mspherical[2];

/*   fprintf(stderr, "%f %f, %f %f\n", view->camera->theta, view->camera->phi, Theta, Phi); */
/*   fprintf(stderr, "%f %f, %f %f\n", dTheta, dPhi, Theta - view->camera->theta, Phi -  view->camera->phi); */

  cth = cos(Theta * PI180);
  sth = sin(Theta * PI180);
  cph = cos(Phi * PI180);
  sph = sin(Phi * PI180);

  matPhiPrime[0][0] = cph;
  matPhiPrime[1][0] = -sph;
  matPhiPrime[2][0] = 0.;
  matPhiPrime[0][1] = sph;
  matPhiPrime[1][1] = cph;
  matPhiPrime[2][1] = 0.;
  matPhiPrime[0][2] = 0.;
  matPhiPrime[1][2] = 0.;
  matPhiPrime[2][2] = 1.;

  matThetaPrime[0][0] = cth;
  matThetaPrime[1][0] = 0.;
  matThetaPrime[2][0] = sth;
  matThetaPrime[0][1] = 0.;
  matThetaPrime[1][1] = 1.;
  matThetaPrime[2][1] = 0.;
  matThetaPrime[0][2] = -sth;
  matThetaPrime[1][2] = 0.;
  matThetaPrime[2][2] = cth;

  matrix_productMatrix(matRes2, matPhiPrime, matRprime2R);
  matrix_productMatrix(matRes, matThetaPrime, matRes2);

  MinRprime[0] = 0.;
  MinRprime[1] = 1.;
  MinRprime[2] = 0.;
  matrix_productVector(MinR, matRes, MinRprime);
/*   fprintf(stderr, "vect u : %f %f %f -> %f\n", MinR[0], MinR[1], MinR[2], */
/* 	  MinR[0]*MinR[0] + MinR[1]*MinR[1] + MinR[2]*MinR[2]); */
  Omega = acos(MinR[1]) * RADTODEG;
  if (MinR[0] > 0.)
    Omega = -Omega;
  while (Omega - view->camera->omega < -90.)
    Omega += 360.;
  while (Omega - view->camera->omega > 90.)
    Omega -= 360.;

/*   fprintf(stderr, "Theta phi omega : %f %f %f\n", Theta, Phi, Omega); */
  angles[0] = Theta;
  angles[1] = Phi;
  angles[2] = Omega;
}



/* This is a positive float that enable to increase or
   decrease the rendering load by modifying the
   number of facettes. */
#define FLAG_PARAMETER_OPENGL_DETAILS   "opengl_details"
#define DESC_PARAMETER_OPENGL_DETAILS   "Give a value to the quality of rendering (100 is normal) ; positive integer"
#define PARAMETER_OPENGL_DETAILS_DEFAULT 100
gboolean readOpenGLPrecision(gchar **lines, int nbLines,
			     int position, GString *errorMessage);
gboolean exportParametersOpenGLView(GString *data, int *nbLinesWritten,
				    VisuData *dataObj);

#define FLAG_RESOURCE_OPENGL_ANGLES "opengl_theta_phi_omega"
#define DESC_RESOURCE_OPENGL_ANGLES "2 real values (degrees) for user orientation with respect to sample"
gboolean readOpenGLThetaPhi(gchar **lines, int nbLines,
			    int position, GString *errorMessage);
#define FLAG_RESOURCE_OPENGL_TRANSLAT "opengl_xs_ys"
#define DESC_RESOURCE_OPENGL_TRANSLAT "2 real values for image position with respect to [0.0, 1.0]x[0.0, 1.0] window"
gboolean readOpenGLXsYs(gchar **lines, int nbLines,
			int position, GString *errorMessage);
#define FLAG_RESOURCE_OPENGL_GROSS "opengl_gross"
#define DESC_RESOURCE_OPENGL_GROSS "gross factor (must be real > 0.0)"
gboolean readOpenGLGross(gchar **lines, int nbLines,
			 int position, GString *errorMessage);
#define FLAG_RESOURCE_OPENGL_PERSP "opengl_d_red"
#define DESC_RESOURCE_OPENGL_PERSP "reduced perspective distance (must be real > 1.0)"
gboolean readOpenGLPersp(gchar **lines, int nbLines,
			 int position, GString *errorMessage);
gboolean exportResourcesOpenGLView(GString *data, int *nbLinesWritten,
				   VisuData *dataObj);


void OpenGLViewInit()
{
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "OpenGl View : initialization.\n");
  /* Parameters */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OPENGL_DETAILS,
					  DESC_PARAMETER_OPENGL_DETAILS,
					  1, readOpenGLPrecision);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersOpenGLView);

  OpenGLViewSet_precision((float)PARAMETER_OPENGL_DETAILS_DEFAULT / 100.);

  /* Resources */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_ANGLES,
					  DESC_RESOURCE_OPENGL_ANGLES,
					  1, readOpenGLThetaPhi);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_TRANSLAT,
					  DESC_RESOURCE_OPENGL_TRANSLAT,
					  1, readOpenGLXsYs);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_GROSS,
					  DESC_RESOURCE_OPENGL_GROSS,
					  1, readOpenGLGross);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_OPENGL_PERSP,
					  DESC_RESOURCE_OPENGL_PERSP,
					  1, readOpenGLPersp);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesOpenGLView);

}

gboolean readOpenGLPrecision(gchar **lines, int nbLines,
			     int position, GString *errorMessage)
{
  int val;
  int res;

  res = sscanf(lines[0],"%d", &val);
  if (res != 1)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d,"
					       " 1 integer value (v > 0) must appear"
					       " after the %s markup.\n"),
			       position, FLAG_PARAMETER_OPENGL_DETAILS);
      OpenGLViewSet_precision((float)PARAMETER_OPENGL_DETAILS_DEFAULT / 100.);
      return FALSE;
    }

  OpenGLViewSet_precision((float)val / 100.);
  return TRUE;
}
gboolean exportParametersOpenGLView(GString *data, int *nbLinesWritten,
				    VisuData *dataObj)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OPENGL_DETAILS);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OPENGL_DETAILS,
	  (int)(precisionOfRendering * 100.));
  *nbLinesWritten = 3;
  return TRUE;
}
gboolean readOpenGLThetaPhi(gchar **lines, int nbLines,
			    int position, GString *errorMessage)
{
  int res;
  float val[3];
  GList *tmpLst;
  
  res = sscanf(lines[0],"%f %f %f", &val[0], &val[1], &val[2]);
  if (res != 3)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 2 floating "
					       "point values must appear"
					       " after the %s markup.\n"),
			       position, FLAG_RESOURCE_OPENGL_ANGLES);
      /* Don't change current values. */
      return FALSE;
    }
  anglesDefault[0] = val[0];
  anglesDefault[1] = val[1];
  anglesDefault[2] = val[2];
  tmpLst = visuDataGet_allObjects();
  while (tmpLst)
    {
      visuDataSet_angleOfView(VISU_DATA(tmpLst->data), val[0], val[1], val[2],
			      MASK_THETA | MASK_PHI | MASK_OMEGA);
      tmpLst = g_list_next(tmpLst);
    }

  return TRUE;
}
gboolean readOpenGLXsYs(gchar **lines, int nbLines,
			int position, GString *errorMessage)
{
  int res;
  float val[2];
  GList *tmpLst;
  
  res = sscanf(lines[0],"%f %f", &val[0], &val[1]);
  if (res != 2)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 2 floating "
					       "point values must appear"
					       " after the %s markup.\n"),
			       position, FLAG_RESOURCE_OPENGL_TRANSLAT);
      /* Don't change current values. */
      return FALSE;
    }
  translatDefault[0] = val[0];
  translatDefault[1] = val[1];
  tmpLst = visuDataGet_allObjects();
  while (tmpLst)
    {
      visuDataSet_positionOfView(VISU_DATA(tmpLst->data), val[0], val[1],
				 MASK_XS | MASK_YS);
      tmpLst = g_list_next(tmpLst);
    }

  return TRUE;
}
gboolean readOpenGLGross(gchar **lines, int nbLines,
			 int position, GString *errorMessage)
{
  int res;
  float val;
  GList *tmpLst;
  
  res = sscanf(lines[0],"%f", &val);
  if (res != 1)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 1 floating "
					       "point value (0. < v <= 10)"
					       " must appear after the %s"
					       " markup.\n"), position,
			       FLAG_RESOURCE_OPENGL_GROSS);
      /* Don't change current values. */
      return FALSE;
    }
  grossDefault = val;
  tmpLst = visuDataGet_allObjects();
  while (tmpLst)
    {
      visuDataSet_zoomOfView(VISU_DATA(tmpLst->data), val);
      tmpLst = g_list_next(tmpLst);
    }

  return TRUE;
}
gboolean readOpenGLPersp(gchar **lines, int nbLines,
			 int position, GString *errorMessage)
{
  int res;
  float val;
  GList *tmpLst;
  
  res = sscanf(lines[0],"%f", &val);
  if (res != 1)
    {
      if (errorMessage)
	g_string_append_printf(errorMessage, _("WARNING! Parse error at line %d, 1 floating "
					       "point value (0. < v <= 10)"
					       " must appear after the %s"
					       " markup.\n"), position,
			       FLAG_RESOURCE_OPENGL_PERSP);
      /* Don't change current values. */
      return FALSE;
    }
  perspDefault = val;
  tmpLst = visuDataGet_allObjects();
  while (tmpLst)
    {
      visuDataSet_perspectiveOfView(VISU_DATA(tmpLst->data), val);
      tmpLst = g_list_next(tmpLst);
    }

  return TRUE;
}
gboolean exportResourcesOpenGLView(GString *data, int *nbLinesWritten,
				   VisuData *dataObj)
{
  GList *tmpLst;
  OpenGLView *view;

  tmpLst = visuDataGet_allObjects();
  if (tmpLst)
    view = visuDataGet_openGLView(VISU_DATA(tmpLst->data));
  else
    view = (OpenGLView*)0;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_ANGLES);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_ANGLES);
  if (view)
    g_string_append_printf(data, "    %9.3f %9.3f %9.3f\n",
			   view->camera->theta, view->camera->phi, view->camera->omega);
  else
    g_string_append_printf(data, "    %9.3f %9.3f %9.3f\n",
			   anglesDefault[0], anglesDefault[1], anglesDefault[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_TRANSLAT);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_TRANSLAT);
  if (view)
    g_string_append_printf(data, "    %9.3f %9.3f\n",
			   view->camera->xs, view->camera->ys);
  else
    g_string_append_printf(data, "    %9.3f %9.3f\n",
			   translatDefault[0], translatDefault[1]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_GROSS);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_GROSS);
  if (view)
    g_string_append_printf(data, "    %9.3f\n", view->camera->gross);
  else
    g_string_append_printf(data, "    %9.3f\n", grossDefault);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_OPENGL_PERSP);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_OPENGL_PERSP);
  if (view)
    g_string_append_printf(data, "    %9.3f\n", view->camera->d_red);
  else
    g_string_append_printf(data, "    %9.3f\n", perspDefault);
  g_string_append_printf(data, "\n");
  *nbLinesWritten = 10;

  return TRUE;
}

