/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef TOOLFILEFORMAT_H
#define TOOLFILEFORMAT_H

#include <glib.h>

G_BEGIN_DECLS

/**
 * FileFormat_struct:
 * @fileType: a #GList, containing file patterns, for example (*.jpg; *.jpeg; *.jpe) ;
 * @description: a short text to describe this file format. It should not be bigger
 *                than 30 characters ;
 * @labelString: a private field, it is a concatenation
 *                of "description ({patterns},)" ;
 * @properties: a private field to store some properties, see fileFormatAdd_property*()
 *              and fileFormatGet_property*().
 *
 * This structure is used to describe a file format, for example JPEG image.
 */
struct FileFormat_struct
{
  /* This is the list of file patterns, for example (*.jpg; *.jpeg; *.jpe). */
  GList *fileType;
  
  /* This is a short text to describe this file format.
     It should not be bigger than 30 characters. */
  gchar *description;

  /* This is a private field, it is a concatenation
     of "description ({patterns},)". */
  gchar *labelString;

  /* This is a private field. It enables to add some properties
     to a file format. See fileFormatAdd_property*() and fileFormatGet_property*()
     to control them. */
  GList *properties;
};
/**
 * FileFormat:
 *
 * A short name to identify #FileFormat_struct structures.
 */
typedef struct FileFormat_struct FileFormat;

#define VISU_FILEFORMAT_ERROR fileFormatGet_quark()
GQuark fileFormatGet_quark();
/**
 * FileFormatErrorFlag:
 * @FILEFORMAT_ERROR_METHOD: Error from the loading method.
 * @FILEFORMAT_ERROR_FILE: Error when opening.
 * @FILEFORMAT_ERROR_FORMAT: Wrongness in format.
 * @FILEFORMAT_ERROR_UNKNOWN_FORMAT: the file is not recognised.
 *
 * These are flags used when reading a file with a loading method associated to a file format.
 */
typedef enum
  {
    FILEFORMAT_ERROR_METHOD,   /* Error from the rendering method. */
    FILEFORMAT_ERROR_FILE,     /* Error when opening. */
    FILEFORMAT_ERROR_FORMAT,   /* Wrongness in format. */
    FILEFORMAT_ERROR_UNKNOWN_FORMAT
  } FileFormatErrorFlag;

void fileFormatInit();

/**
 * fileFormatNew:
 * @descr: a short string to label a new file format.
 * @patterns: a set of patterns to identify files of this format.
 *
 * Allocate a new #FileFormat. The @patterns argument is copied in
 * the #FileFormat object and can be freed safely after the call to this method.
 *
 * Returns: a newly allocated FileFormat, or NULL if something goes wrong.
 */
FileFormat* fileFormatNew(gchar* descr, gchar** patterns);
/**
 * fileFormatFree:
 * @format: a #FileFormat to free.
 *
 * Free all the memory used by @format. If @format is NULL, the method
 * writes a warning on the eror output.
 */
void fileFormatFree(FileFormat *format);
/**
 * fileFormatGet_label:
 * @format: a #FileFormat.
 *
 * This method gives a label describing the file format.
 *
 * Returns: a string made by the description and all the paterns of the
 * given format, given in parentethis. This string should not be freed.
 */
gchar* fileFormatGet_label(FileFormat *format);
/**
 * fileFormatUse_match:
 * @format: a #FileFormat ;
 * @filename: a string to match.
 *
 * This method try to match the given string to one of the patterns of
 * the #FileFormat @format.
 *
 * Returns: TRUE if the given string matchs the file format.
 */
gboolean fileFormatUse_match(FileFormat *format, gchar*filename);

/**
 * FileFormatProperty_struct:
 *
 * Opaque structure to store some #FileFormat properties.
 */
struct FileFormatProperty_struct;
/**
 * FileFormatProperty:
 *
 * Short definition for a #FileFormatProperty_struct object.
 */
typedef struct FileFormatProperty_struct FileFormatProperty;
/**
 * FileFormatPropertyTypeId:
 * @fileFormat_type_integer: the property is an integer value ;
 * @fileFormat_type_boolean: the property is a boolean value ;
 * @fileFormat_type_string: the property is a string ;
 * @fileFormat_nb_type: give the number of supported property types.
 *
 * Type for stored properties.
 */
typedef enum
  {
    fileFormat_type_integer,
    fileFormat_type_boolean,
    fileFormat_type_string,
    fileFormat_nb_type
  } FileFormatPropertyTypeId;
/**
 * fileFormatGet_propertiesList:
 * @format: a #FileFormat.
 *
 * Use this method to get a list of stored properties for the given #FileFormat.
 * The GList is composed of #FileFormatProperty objects.
 * 
 * Returns: a GList* that is read-only.
 */
GList* fileFormatGet_propertiesList(FileFormat *format);
/**
 * fileFormatGet_propertyLabelUTF8:
 * @property: a pointer to the property to get the name from.
 * 
 * Use this method to get a printable label for the given property.
 *
 * Returns: a string in UTF8. This string is read-only.
 */
gchar* fileFormatGet_propertyLabelUTF8(FileFormatProperty *property);
/**
 * fileFormatGet_propertyName:
 * @property: a pointer to the property to get the name from.
 * 
 * Use this method to get the string used when saving for the given property.
 *
 * Returns: a string. This string is read-only.
 */
gchar* fileFormatGet_propertyName(FileFormatProperty *property);
/**
 * fileFormatGet_propertyType:
 * @property: a pointer to the property to get the type from.
 * 
 * Use this method to get the type of the given property.
 *
 * Returns: an integer (see #FileFormatPropertyTypeId).
 */
FileFormatPropertyTypeId fileFormatGet_propertyType(FileFormatProperty *property);

/**
 * fileFormatAdd_propertyInteger:
 * @format: a #FileFormat ;
 * @name: a null terminated string ;
 * @labelUTF8: a null terminated string ;
 * @value: the value to be stored.
 *
 * Use this method to create an integer property associated to the given #FileFormat.
 * This method is used only to create the property (and initialize it), use
 * fileFormatSet_propertyInteger() to change its value and fileFormatGet_propertyInteger()
 * to retrieve it. No checks are done If such a property already exists.
 */
void fileFormatAdd_propertyInteger(FileFormat *format, const gchar* name,
				   const gchar* labelUTF8, int value);
/**
 * fileFormatSet_propertyInteger:
 * @property: a pointer to the property to set the value to ;
 * @value: the value to be stored.
 *
 * Use this method to modify a value of a previously created property
 * (use fileFormatAdd_propertyInteger() to create one).
 *
 * Returns: TRUE if value is changed.
 */
gboolean fileFormatSet_propertyInteger(FileFormatProperty *property, int value);
/**
 * fileFormatGet_propertyInteger:
 * @property: a pointer to the property to get the value from.
 *
 * Use this method to retrieve a value of a previously created property
 * (use fileFormatAdd_propertyInteger() to create one).
 *
 * Returns: the value or -1 if something goes wrong.
 */
int fileFormatGet_propertyInteger(FileFormatProperty *property);

/**
 * fileFormatAdd_propertyBoolean:
 * @format: a #FileFormat ;
 * @name: a null terminated string ;
 * @labelUTF8: a null terminated string ;
 * @value: the value to be stored.
 *
 * Use this method to create a boolean property associated to the given #FileFormat.
 * This method is used only to create the property (and initialize it), use
 * fileFormatSet_propertyBoolean() to change its value and fileFormatGet_propertyBoolean()
 * to retrieve it. No checks are done if such a property already exists.
 */
void fileFormatAdd_propertyBoolean(FileFormat *format, const gchar* name,
				   const gchar* labelUTF8, gboolean value);
/**
 * fileFormatSet_propertyBoolean:
 * @property: a pointer to the property to set the value to ;
 * @value: the value to be stored.
 *
 * Use this method to modify a value of a previously created property
 * (use fileFormatAdd_propertyBoolean() to create one).
 *
 * Returns: TRUE if value is changed.
 */
gboolean fileFormatSet_propertyBoolean(FileFormatProperty *property, gboolean value);
/**
 * fileFormatGet_propertyBoolean:
 * @property: a pointer to the property to get the value from.
 *
 * Use this method to retrieve a value of a previously created property
 * (use fileFormatAdd_propertyBoolean() to create one).
 *
 * Returns: the value or FALSE if something goes wrong.
 */
gboolean fileFormatGet_propertyBoolean(FileFormatProperty *property);


G_END_DECLS


#endif
