/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtkdialog.h>

#include <math.h>

#include "gtk_orientationChooser.h"

#include <support.h>
#include <visu_tools.h>
#include <visu_object.h>
#include <visu_data.h>
#include <gtk_main.h>

#include <coreTools/toolMatrix.h>

/**
 * SECTION:orientationchooser
 * @short_description: Defines a dialog widget to choose the camera position.
 * @include: coreTools/toolMatrix.h
 * 
 * <para>This widget is a #GtkDialog window that can be used to choose
 * an orientation for the camera, using either the cartesian
 * coordinates, the box coordinates or the spherical coordinates.</para>
 */

enum {
  VALUES_CHANGED,
  LAST_SIGNAL
};

typedef enum
  {
    orientationChooser_ortho,
    orientationChooser_box,
    orientationChooser_angles
  } OrientationChooserBasis;

/* Object structures. */
struct _OrientationChooser
{
  GtkDialog dialog;

  /* The kind of chooser in box coordinates. */
  OrientationChooserKind kind;

  /* The spins for the ortho selection. */
  GtkWidget *spinsOrtho[3];
  gulong signalsOrtho[3];

  /* The spins for the box selection. */
  GtkWidget *hboxBox;
  GtkWidget *spinsBox[3];
  gulong signalsBox[3];

  /* The spins for the angle selection. */
  GtkWidget *spinsAngles[2];
  gulong signalsAngles[2];

  /* The check box for live update. */
  GtkWidget *checkLiveUpdate;

  /* Associated VisuData, if any. */
  VisuData *dataObj;
  /* Transformation matrix associated to the box. */
  float boxToOrtho[3][3];
  float orthoToBox[3][3];

  /* Memory gestion. */
  gboolean dispose_has_run;
};

struct _OrientationChooserClass
{
  GtkDialogClass parent_class;

  void (*orientationChooser) (OrientationChooser *orientation);
};

G_DEFINE_TYPE(OrientationChooser, orientationChooser, GTK_TYPE_DIALOG)

/* Local variables. */
static guint orientationChooser_signals[LAST_SIGNAL] = { 0 };

/* Local methods. */
static void orientationChooser_class_init(OrientationChooserClass *klass);
static void orientationChooser_init(OrientationChooser *orientationChooser);
static void orientationChooser_dispose(GObject *orientationChooser);
static void orientationChooser_finalize(GObject *obj);
static void orientationChanged(OrientationChooser *orientation,
			       OrientationChooserBasis changedBasis);
static void visuDataChanged(OrientationChooser *orientation, VisuData *data);

/* Local callbacks. */
static void onResponse(OrientationChooser *chooser, gint response, gpointer data);
static void onOrthoChanged(GtkSpinButton *spin, gpointer data);
static void onBoxChanged(GtkSpinButton *spin, gpointer data);
static void onAnglesChanged(GtkSpinButton *spin, gpointer data);
static void onDataReadyForRendering(GObject *obj, VisuData *dataObj, gpointer data);

static void orientationChooser_class_init(OrientationChooserClass *klass)
{
  DBG_fprintf(stderr, "Gtk OrientationChooser: creating the class of the widget.\n");

  DBG_fprintf(stderr, "                     - adding new signals ;\n");
  /**
   * OrientationChooser::values-changed:
   * @chooser: the #OrientationChooser that emits the signal.
   *
   * This signal is emitted when the values are changed and when the
   * live update checkbox is active.
   *
   * Since: 3.4
   */
  orientationChooser_signals[VALUES_CHANGED] =
    g_signal_new ("values-changed",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET(OrientationChooserClass, orientationChooser),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = orientationChooser_dispose;
  G_OBJECT_CLASS(klass)->finalize = orientationChooser_finalize;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void orientationChooser_dispose(GObject *orientationChooser)
{
  DBG_fprintf(stderr, "Gtk OrientationChooser: dispose object %p.\n",
	      (gpointer)orientationChooser);

  if (ORIENTATION_CHOOSER(orientationChooser)->dispose_has_run)
    return;

  ORIENTATION_CHOOSER(orientationChooser)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(orientationChooser_parent_class)->dispose(orientationChooser);
}
/* This method is called once only. */
static void orientationChooser_finalize(GObject *obj)
{
  OrientationChooser *orientationChooser;

  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk OrientationChooser: finalize object %p.\n",
	      (gpointer)obj);

  orientationChooser = ORIENTATION_CHOOSER(obj);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(orientationChooser_parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Gtk OrientationChooser: freeing ... OK.\n");
}


static void orientationChooser_init(OrientationChooser *orientationChooser)
{
  int i;

  DBG_fprintf(stderr, "Gtk OrientationChooser: initializing new object (%p).\n",
	      (gpointer)orientationChooser);

  orientationChooser->dispose_has_run = FALSE;

  orientationChooser->kind = orientationChooser_direction;

  for (i = 0; i < 3; i++)
    {
      orientationChooser->spinsOrtho[i] =
	gtk_spin_button_new_with_range(-1000, 1000, 1);
      gtk_spin_button_set_value
	(GTK_SPIN_BUTTON(orientationChooser->spinsOrtho[i]), 0);
      gtk_spin_button_set_digits
	(GTK_SPIN_BUTTON(orientationChooser->spinsOrtho[i]), 5);
      orientationChooser->signalsOrtho[i] = 
	g_signal_connect(G_OBJECT(orientationChooser->spinsOrtho[i]), "value-changed",
			 G_CALLBACK(onOrthoChanged), orientationChooser);
    }
  orientationChooser->hboxBox = gtk_hbox_new(FALSE, 0);
  for (i = 0; i < 3; i++)
    {
      orientationChooser->spinsBox[i] =
	gtk_spin_button_new_with_range(-1000, 1000, 1);
      gtk_spin_button_set_value
	(GTK_SPIN_BUTTON(orientationChooser->spinsBox[i]), 0);
      gtk_spin_button_set_digits
	(GTK_SPIN_BUTTON(orientationChooser->spinsBox[i]), 5);
      orientationChooser->signalsBox[i] = 
	g_signal_connect(G_OBJECT(orientationChooser->spinsBox[i]), "value-changed",
			 G_CALLBACK(onBoxChanged), orientationChooser);
    }
  for (i = 0; i < 2; i++)
    {
      orientationChooser->spinsAngles[i] =
	gtk_spin_button_new_with_range(-180, 180, 1);
      gtk_spin_button_set_value
	(GTK_SPIN_BUTTON(orientationChooser->spinsAngles[i]), 0);
      gtk_spin_button_set_digits
	(GTK_SPIN_BUTTON(orientationChooser->spinsAngles[i]), 5);
      orientationChooser->signalsAngles[i] = 
	g_signal_connect(G_OBJECT(orientationChooser->spinsAngles[i]), "value-changed",
			 G_CALLBACK(onAnglesChanged), orientationChooser);
    }
  orientationChooser->checkLiveUpdate =
    gtk_check_button_new_with_label(_("Update values on the fly."));

  /* Connect a signal when the dialog is closed. */
  g_signal_connect(G_OBJECT(orientationChooser), "response",
		   G_CALLBACK(onResponse), (gpointer)0);
}

GtkWidget* orientationChooserNew(OrientationChooserKind kind, gboolean liveUpdate,
				 VisuData *data, GtkWindow *parent)
{
  OrientationChooser *orientationChooser;
  GtkWidget *wd, *hbox, *hbox2, *vbox;
  int i;
  gchar *axesNames[3] = {"x:", "y:", "z:"};
  gchar *anglesNames[2] = {"theta:", "phi:"};

  orientationChooser = ORIENTATION_CHOOSER(g_object_new(orientationChooser_get_type(), NULL));

  DBG_fprintf(stderr, "Gtk OrientationChooser: creating new object: %p"
	      " with parent %p.\n", (gpointer)orientationChooser, (gpointer)parent);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(orientationChooser->checkLiveUpdate),
			       liveUpdate);

  gtk_dialog_add_buttons(GTK_DIALOG(orientationChooser),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL);
  gtk_dialog_set_default_response(GTK_DIALOG(orientationChooser),
				  GTK_RESPONSE_ACCEPT);
/*   gtk_window_set_type_hint(GTK_WINDOW(orientationChooser), */
/* 			   GDK_WINDOW_TYPE_HINT_UTILITY); */
  gtk_window_set_skip_pager_hint(GTK_WINDOW(orientationChooser), TRUE);
  if (!parent)
    parent = GTK_WINDOW(gtkMainClassGet_currentPanel());
  gtk_window_set_transient_for(GTK_WINDOW(orientationChooser), parent);

  /* Title. */
  wd = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_label_set_markup(GTK_LABEL(wd), _("<span size=\"larger\">Choose an orientation</span>"));
  gtk_widget_set_name(wd, "label_head_2");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(orientationChooser)->vbox), wd, FALSE, FALSE, 5);

  /* Ortho orientation. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(orientationChooser)->vbox), hbox, FALSE, FALSE, 3);
  wd = create_pixmap((GtkWidget*)0, "axes-ortho.png");
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  wd = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_label_set_markup(GTK_LABEL(wd), _("<b>On an orthonormal basis set</b>"));
  gtk_misc_set_alignment(GTK_MISC(wd), 0.1, 1.);
  gtk_box_pack_start(GTK_BOX(vbox), wd, TRUE, TRUE, 15);
  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
  for (i = 0; i < 3; i++)
    {
      wd = gtk_label_new(axesNames[i]);
      gtk_misc_set_alignment(GTK_MISC(wd), 1., .5);
      gtk_box_pack_start(GTK_BOX(hbox2), wd, TRUE, TRUE, 2);
      gtk_box_pack_start(GTK_BOX(hbox2), orientationChooser->spinsOrtho[i], FALSE, FALSE, 0);
    }

  /* Box orientation. */
  gtk_widget_set_sensitive(orientationChooser->hboxBox, data &&
			   visuDataGet_periodic(data));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(orientationChooser)->vbox),
		     orientationChooser->hboxBox, FALSE, FALSE, 3);
  wd = create_pixmap((GtkWidget*)0, "axes-box.png");
  gtk_box_pack_start(GTK_BOX(orientationChooser->hboxBox), wd, FALSE, FALSE, 0);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(orientationChooser->hboxBox), vbox, TRUE, TRUE, 0);
  wd = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_label_set_markup(GTK_LABEL(wd), _("<b>Following the box basis set</b>"));
  gtk_misc_set_alignment(GTK_MISC(wd), 0.1, 1.);
  gtk_box_pack_start(GTK_BOX(vbox), wd, TRUE, TRUE, 7);
  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
  for (i = 0; i < 3; i++)
    {
      wd = gtk_label_new(axesNames[i]);
      gtk_misc_set_alignment(GTK_MISC(wd), 1., .5);
      gtk_box_pack_start(GTK_BOX(hbox2), wd, TRUE, TRUE, 2);
      gtk_box_pack_start(GTK_BOX(hbox2), orientationChooser->spinsBox[i], FALSE, FALSE, 0);
    }

  /* Ortho orientation. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(orientationChooser)->vbox), hbox, FALSE, FALSE, 3);
  wd = create_pixmap((GtkWidget*)0, "axes-angles.png");
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
  wd = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_label_set_markup(GTK_LABEL(wd), _("<b>On a spherical basis set</b>"));
  gtk_misc_set_alignment(GTK_MISC(wd), 0.1, 1.);
  gtk_box_pack_start(GTK_BOX(vbox), wd, TRUE, TRUE, 15);
  hbox2 = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, 0);
  for (i = 0; i < 2; i++)
    {
      wd = gtk_label_new(anglesNames[i]);
      gtk_misc_set_alignment(GTK_MISC(wd), 1., .5);
      gtk_box_pack_start(GTK_BOX(hbox2), wd, TRUE, TRUE, 2);
      gtk_box_pack_start(GTK_BOX(hbox2), orientationChooser->spinsAngles[i], FALSE, FALSE, 0);
    }

  /* The live update checkbox. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(orientationChooser)->vbox),
		     hbox, FALSE, FALSE, 5);
  gtk_box_pack_start(GTK_BOX(hbox), orientationChooser->checkLiveUpdate,
		     FALSE, FALSE, 90);

  gtk_widget_show_all(GTK_WIDGET(orientationChooser));

  /* Set other internal values. */
  orientationChooser->kind = kind;
  visuDataChanged(orientationChooser, data);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReadyForRendering), (gpointer)orientationChooser);

  return GTK_WIDGET(orientationChooser);
}

static void onDataReadyForRendering(GObject *obj _U_, VisuData *dataObj,
				    gpointer data)
{
  /* We recompute the transformation matrix. */
  visuDataChanged(ORIENTATION_CHOOSER(data), dataObj);
}

static void visuDataChanged(OrientationChooser *orientation, VisuData *data)
{
  float tmpCoord[3], norm;
  float boxMatrix[3][3];
  int i, j;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  orientation->dataObj = data;

  /* Set or unset the box coordinate choice. */
  gtk_widget_set_sensitive(orientation->hboxBox, data &&
			   visuDataGet_periodic(data));

  if (!data)
    return;

  /* Change the transformation matrix. */
  for (j = 0; j < 3; j++)
    {
      tmpCoord[0] = (j == 0)?1.:0.;
      tmpCoord[1] = (j == 1)?1.:0.;
      tmpCoord[2] = (j == 2)?1.:0.;
      visuDataConvert_boxCoordinatestoXYZ(orientation->dataObj,
					  boxMatrix[j], tmpCoord);
    }
  DBG_fprintf(stderr, "Gtk OrientationChooser: get the box matrix.\n");
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n",
	      boxMatrix[0][0], boxMatrix[1][0], boxMatrix[2][0]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n",
	      boxMatrix[0][1], boxMatrix[1][1], boxMatrix[2][1]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n",
	      boxMatrix[0][2], boxMatrix[1][2], boxMatrix[2][2]);
  /* We create a matrix to transform the box coordinates to
     cartesian values keeping the orthogonality. */
  for (i = 0; i < 3; i++)
    {
      norm = 0.;
      for (j = 0; j < 3; j++)
	{
	  orientation->boxToOrtho[j][i] =
	    boxMatrix[(i + 1)%3][(j + 1)%3] *
	    boxMatrix[(i + 2)%3][(j + 2)%3] -
	    boxMatrix[(i + 1)%3][(j + 2)%3] *
	    boxMatrix[(i + 2)%3][(j + 1)%3];
	  norm += orientation->boxToOrtho[j][i] * orientation->boxToOrtho[j][i];
	}
      /* We normalise the tranformation matrix. */
      norm = sqrt(norm);
      for (j = 0; j < 3; j++)
	orientation->boxToOrtho[j][i] /= norm;
    }
  DBG_fprintf(stderr, "Gtk OrientationChooser: normal case, boxToOrtho matrix.\n");
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->boxToOrtho[0][0],
	      orientation->boxToOrtho[0][1], orientation->boxToOrtho[0][2]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->boxToOrtho[1][0],
	      orientation->boxToOrtho[1][1], orientation->boxToOrtho[1][2]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->boxToOrtho[2][0],
	      orientation->boxToOrtho[2][1], orientation->boxToOrtho[2][2]);
  /* We inverse using the trick that the matrix is down (upper side
     is zeros. */
  orientation->orthoToBox[0][0] = 1. / orientation->boxToOrtho[0][0];
  orientation->orthoToBox[1][0] = - orientation->boxToOrtho[1][0] / 
    orientation->boxToOrtho[1][1] / orientation->boxToOrtho[0][0];
  orientation->orthoToBox[2][0] = 
    (orientation->boxToOrtho[1][0] * orientation->boxToOrtho[2][1] -
     orientation->boxToOrtho[1][1] * orientation->boxToOrtho[2][0]) /
    orientation->boxToOrtho[0][0] /
    orientation->boxToOrtho[1][1] /
    orientation->boxToOrtho[2][2];
  orientation->orthoToBox[0][1] = 0.;
  orientation->orthoToBox[1][1] = 1. / orientation->boxToOrtho[1][1];
  orientation->orthoToBox[2][1] =  - orientation->boxToOrtho[2][1] / 
    orientation->boxToOrtho[1][1] / orientation->boxToOrtho[2][2];
  orientation->orthoToBox[0][2] = 0.;
  orientation->orthoToBox[1][2] = 0.;
  orientation->orthoToBox[2][2] = 1. / orientation->boxToOrtho[2][2];
  DBG_fprintf(stderr, "Gtk OrientationChooser: normal case, orthoToBox matrix.\n");
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->orthoToBox[0][0],
	      orientation->orthoToBox[0][1], orientation->orthoToBox[0][2]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->orthoToBox[1][0],
	      orientation->orthoToBox[1][1], orientation->orthoToBox[1][2]);
  DBG_fprintf(stderr, " | %8.4f %8.4f %8.4f\n", orientation->orthoToBox[2][0],
	      orientation->orthoToBox[2][1], orientation->orthoToBox[2][2]);
}

static void orientationChanged(OrientationChooser *orientation,
			       OrientationChooserBasis changedBasis)
{
  int i;
  float xyz[3], boxXYZ[3], angles[3];

  g_return_if_fail(changedBasis != orientationChooser_box || orientation->dataObj);

  /* Get the changed coordinates and update the others. */
  switch (changedBasis)
    {
    case orientationChooser_ortho:
      for (i = 0; i < 3; i++)
	xyz[i] = (float)gtk_spin_button_get_value
	  (GTK_SPIN_BUTTON(orientation->spinsOrtho[i]));
      if (orientation->dataObj)
	{
	  if (orientation->kind == orientationChooser_direction)
	    visuDataConvert_XYZtoBoxCoordinates(orientation->dataObj, boxXYZ, xyz);
	  else if (orientation->kind == orientationChooser_normal)
	    matrix_productVector(boxXYZ, orientation->orthoToBox, xyz);
	}
      matrix_cartesianToSpherical(angles, xyz);
      if (angles[2] > 180.) angles[2] -= 360.;
      break;
    case orientationChooser_box:
      for (i = 0; i < 3; i++)
	boxXYZ[i] = (float)gtk_spin_button_get_value
	  (GTK_SPIN_BUTTON(orientation->spinsBox[i]));
      /* The conversion to cartesian coordinates depends on the
	 attribute kind. */
      if (orientation->kind == orientationChooser_direction)
	visuDataConvert_boxCoordinatestoXYZ(orientation->dataObj, xyz, boxXYZ);
      else if (orientation->kind == orientationChooser_normal)
	matrix_productVector(xyz, orientation->boxToOrtho, boxXYZ);
      matrix_cartesianToSpherical(angles, xyz);
      if (angles[2] > 180.) angles[2] -= 360.;
      break;
    case orientationChooser_angles:
      for (i = 0; i < 3; i++)
	xyz[i] = (float)gtk_spin_button_get_value
	  (GTK_SPIN_BUTTON(orientation->spinsOrtho[i]));
      angles[0] = sqrt(xyz[0] * xyz[0] + xyz[1] * xyz[1] + xyz[2] * xyz[2]);
      if (angles[0] == 0.)
	angles[0] = 1.;
      for (i = 1; i < 3; i++)
	angles[i] = (float)gtk_spin_button_get_value
	  (GTK_SPIN_BUTTON(orientation->spinsAngles[i - 1]));
      matrix_sphericalToCartesian(xyz, angles);
      if (orientation->dataObj)
	{
	  if (orientation->kind == orientationChooser_direction)
	    visuDataConvert_XYZtoBoxCoordinates(orientation->dataObj, boxXYZ, xyz);
	  else if (orientation->kind == orientationChooser_normal)
	    matrix_productVector(boxXYZ, orientation->orthoToBox, xyz);
	}
      break;
    }
  DBG_fprintf(stderr, "Gtk OrientationChooser: set coordinates.\n");
  DBG_fprintf(stderr, " | ortho:  %8.4g %8.4g %8.4g\n", xyz[0], xyz[1], xyz[2]);
  DBG_fprintf(stderr, " | box:    %8.4g %8.4g %8.4g\n", boxXYZ[0], boxXYZ[1], boxXYZ[2]);
  DBG_fprintf(stderr, " | angles: %8.4g %8.4g\n", angles[1], angles[2]);
  /* Stop the internal spin signals, change the values and reset the signals. */
  for (i = 0; i < 3; i++)
    {
/*       DBG_fprintf(stderr, "Gtk OrientationChooser: change the ortho values\n"); */
      g_signal_handler_block(G_OBJECT(orientation->spinsOrtho[i]),
			     orientation->signalsOrtho[i]);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientation->spinsOrtho[i]), xyz[i]);
      g_signal_handler_unblock(G_OBJECT(orientation->spinsOrtho[i]),
			       orientation->signalsOrtho[i]);
    }
  if (orientation->dataObj)
    for (i = 0; i < 3; i++)
      {
/* 	DBG_fprintf(stderr, "Gtk OrientationChooser: change the box values\n"); */
	g_signal_handler_block(G_OBJECT(orientation->spinsBox[i]),
			       orientation->signalsBox[i]);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientation->spinsBox[i]),
				  boxXYZ[i]);
	g_signal_handler_unblock(G_OBJECT(orientation->spinsBox[i]),
				 orientation->signalsBox[i]);
      }
  for (i = 0; i < 2; i++)
    {
/*       DBG_fprintf(stderr, "Gtk OrientationChooser: change the angles values\n"); */
      g_signal_handler_block(G_OBJECT(orientation->spinsAngles[i]),
			     orientation->signalsAngles[i]);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(orientation->spinsAngles[i]),
				angles[i + 1]);
      g_signal_handler_unblock(G_OBJECT(orientation->spinsAngles[i]),
			       orientation->signalsAngles[i]);
    }

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(orientation->checkLiveUpdate)))
    {
      DBG_fprintf(stderr, "Gtk OrientationChooser: emit value changed (%d).\n",
		  changedBasis);
      g_signal_emit(G_OBJECT(orientation), orientationChooser_signals[VALUES_CHANGED],
		    0 , NULL);
    }
}
static void onOrthoChanged(GtkSpinButton *spin _U_, gpointer data)
{
  orientationChanged(ORIENTATION_CHOOSER(data), orientationChooser_ortho);
}
static void onBoxChanged(GtkSpinButton *spin _U_, gpointer data)
{
  orientationChanged(ORIENTATION_CHOOSER(data), orientationChooser_box);
}
static void onAnglesChanged(GtkSpinButton *spin _U_, gpointer data)
{
  orientationChanged(ORIENTATION_CHOOSER(data), orientationChooser_angles);
}
static void onResponse(OrientationChooser *chooser, gint response, gpointer data _U_)
{
  if (response == GTK_RESPONSE_ACCEPT &&
      !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(chooser->checkLiveUpdate)))
    {
      DBG_fprintf(stderr, "Gtk OrientationChooser: emit value changed on out.\n");
      g_signal_emit(G_OBJECT(chooser), orientationChooser_signals[VALUES_CHANGED],
		    0 , NULL);
    }
}


void orientationChooserSet_orthoValues(OrientationChooser *orientation,
				       float values[3])
{
  int i;
  gboolean update;
  GtkSpinButton *spin;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  DBG_fprintf(stderr, "Gtk OrientationChooser: change ortho.\n");
  /* Block the internal spin signals and raise manually the signal for one spin. */
  update = FALSE;
  for (i = 0; i < 3; i++)
    {
      spin = GTK_SPIN_BUTTON(orientation->spinsOrtho[i]);
      g_signal_handler_block(G_OBJECT(spin), orientation->signalsOrtho[i]);
      update = update || ((float)gtk_spin_button_get_value(spin) != values[i]);
      gtk_spin_button_set_value(spin, values[i]);
      g_signal_handler_unblock(G_OBJECT(spin), orientation->signalsOrtho[i]);
    }
  if (update)
    orientationChanged(orientation, orientationChooser_ortho);
}
void orientationChooserSet_boxValues(OrientationChooser *orientation,
				     float values[3])
{
  int i;
  gboolean update;
  GtkSpinButton *spin;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  DBG_fprintf(stderr, "Gtk OrientationChooser: change box.\n");
  /* Block the internal spin signals and raise manually the signal for one spin. */
  update = FALSE;
  for (i = 0; i < 3; i++)
    {
      spin = GTK_SPIN_BUTTON(orientation->spinsBox[i]);
      g_signal_handler_block(G_OBJECT(spin), orientation->signalsBox[i]);
      update = update || ((float)gtk_spin_button_get_value(spin) != values[i]);
      gtk_spin_button_set_value(spin, values[i]);
      g_signal_handler_unblock(G_OBJECT(spin), orientation->signalsBox[i]);
    }
  if (update)
    orientationChanged(orientation, orientationChooser_box);
}
void orientationChooserSet_anglesValues(OrientationChooser *orientation,
					float values[2])
{
  int i;
  gboolean update;
  GtkSpinButton *spin;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  DBG_fprintf(stderr, "Gtk OrientationChooser: change angles.\n");
  update = FALSE;
  for (i = 0; i < 2; i++)
    {
      spin = GTK_SPIN_BUTTON(orientation->spinsAngles[i]);
      g_signal_handler_block(G_OBJECT(spin), orientation->signalsAngles[i]);
      update = update || ((float)gtk_spin_button_get_value(spin) != values[i]);
      gtk_spin_button_set_value(spin, values[i]);
      g_signal_handler_unblock(G_OBJECT(spin), orientation->signalsAngles[i]);
    }
  if (update)
    orientationChanged(orientation, orientationChooser_box);
}

void orientationChooserGet_orthoValues(OrientationChooser *orientation,
				       float values[3])
{
  int i;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  for (i = 0; i < 3; i++)
    values[i] = (float)gtk_spin_button_get_value
      (GTK_SPIN_BUTTON(orientation->spinsOrtho[i]));
}
void orientationChooserGet_boxValues(OrientationChooser *orientation,
				     float values[3])
{
  int i;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  for (i = 0; i < 3; i++)
    values[i] = (float)gtk_spin_button_get_value
      (GTK_SPIN_BUTTON(orientation->spinsBox[i]));
}
void orientationChooserGet_anglesValues(OrientationChooser *orientation,
					float values[2])
{
  int i;

  g_return_if_fail(IS_ORIENTATION_CHOOSER(orientation));

  for (i = 0; i < 2; i++)
    values[i] = (float)gtk_spin_button_get_value
      (GTK_SPIN_BUTTON(orientation->spinsAngles[i]));
}
