/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "interactive.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include "view.h"
#include "objectList.h"
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_extension.h>
#include <renderingBackend/visu_actionInterface.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <visu_pickMesure.h>
#include <coreTools/toolConfigFile.h>

#define FLAG_PARAMETER_OBSERVE_METHOD "opengl_observe_method"
#define DESC_PARAMETER_OBSERVE_METHOD "Choose the observe method ; integer (0: constrained mode, 1: walker mode)"
int openGL_preferedObserveMethod;

struct observeData
{
  float old_theta;
  float old_phi;
  float old_omega;
  float old_xs;
  float old_ys;
  float old_gross;
  float old_d_red;
  float xi, yi;
};

/**
 * VisuPick:
 * @data: a pointer to the #VisuData whose @node belongs to ;
 * @nodes: a pointer to a #VisuNode that has been selected ;
 * @button: an integer identifying the button that clicked on the node ;
 * @shiftMod: a boolean to set if shift button was pushed during the click ;
 * @x: x coordinate of the click in window pixels ;
 * @y: y coordinate of the click in window pixels ;
 * @drag: TRUE if the mouse is doing a draging selection ;
 * @xOrig: the initial position of drag along x axis ;
 * @yOrig: idem for y axis ;
 * @xPrev: last cursor position along x axis ;
 * @yPrev: idem for y axis.
 *
 * This structure is used when the user uses the pick
 * interface. It stores some informations on the pick action.
 */
typedef struct _VisuPick
{
  VisuData *data;
  GList *nodes;
/*   VisuNode *node; */

  unsigned int button;
  int shiftMod;

  gboolean drag;
  int x, y;
  int xOrig, yOrig;
  int xPrev, yPrev;
} VisuPick;

struct _VisuInteractive
{
  /* Private stored information. */
  GenericRenderingWindow window;
  VisuData   *data;
  PickMesure *pick;  
  struct observeData observeValues;

  /* Call whenever a node is selected. */
  callBackOnNodeSelectedFunc onNodeSelection;
  gpointer dataNodeSelection;

  /* Call whenever a low level action is done. */
  callBackOnEventFunc onEvent;

  /* Call when the user stop the interactive session. */
  GDestroyNotify stop;
};


/* When an atom is moved (see move()), it is associated
   to a specific extension list to avoid to recreate
   the complete allElement list. */
#define  moveAtomName        "MovedANode"
#define  moveAtomNameI18n    _("Moved a node")
#define  moveAtomDescription _("Draw the node that is displaced.")
OpenGLExtension *moveAtomExtension;
int moveAtomExtension_list;

GList* pickObserveCallbackId;

/* Local methods. */
static gboolean exportParameters(GString *data, int *nbLinesWritten,
				 VisuData *dataObj);
static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj, GError **error);
static gboolean observe(SimplifiedEvents *ev, VisuInteractive *inter);
static gboolean pick(SimplifiedEvents *ev, VisuInteractive *inter);
static gboolean move(SimplifiedEvents *ev, VisuInteractive *inter);
static gboolean mark(SimplifiedEvents *ev, VisuInteractive *inter);
static gboolean pickAndObserve(SimplifiedEvents *ev, VisuInteractive *inter);
static int getSelectElement(VisuData *data, int x, int y);
static GList* getSelectElementsRegion(VisuData *data, int x1, int y1,
				      int x2, int y2);



void openGLInteractive_init()
{
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "Interactive : initialisation.\n");

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OBSERVE_METHOD,
					  DESC_PARAMETER_OBSERVE_METHOD,
					  1, readOpenGLObserveMethod);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);

  /* Set local variables to default. */
  openGL_preferedObserveMethod = OPENGL_OBSERVE_CONSTRAINED;
  moveAtomExtension = (OpenGLExtension*)0;
}

void openGLInteractiveSet_preferedObserveMethod(int method)
{
  g_return_if_fail(method == OPENGL_OBSERVE_CONSTRAINED ||
		   method == OPENGL_OBSERVE_WALKER);

  openGL_preferedObserveMethod = method;
}
int openGLInteractiveGet_preferedObserveMethod()
{
  return openGL_preferedObserveMethod;
}

VisuInteractive* openGLInteractiveInit_session(GenericRenderingWindow window,
					       callBackOnNodeSelectedFunc handler,
					       gpointer data,
					       GDestroyNotify stopPickObserve)
{
  VisuInteractive *inter;
  OpenGLView *view;

  g_return_val_if_fail(window, (VisuInteractive*)0);

  inter = g_malloc(sizeof(VisuInteractive));
  DBG_fprintf(stderr, "Interactive: start new interactive session %p.\n",
	      (gpointer)inter);

  inter->window = window;
  setInteractiveHandlers(window, inter);
  inter->data   = visuRenderingWindowGet_visuData(window);
  inter->pick   = g_object_get_data(G_OBJECT(inter->data), "pickMesure_data");
  g_return_val_if_fail(inter->pick, (VisuInteractive*)0);

  view = visuDataGet_openGLView(visuRenderingWindowGet_visuData(window));

  inter->observeValues.old_theta = view->camera->theta;
  inter->observeValues.old_phi   = view->camera->phi;
  inter->observeValues.old_omega = view->camera->omega;
  inter->observeValues.old_xs    = view->camera->xs;
  inter->observeValues.old_ys    = view->camera->ys;
  inter->observeValues.old_gross = view->camera->gross;
  inter->observeValues.old_d_red = view->camera->d_red;

  inter->onNodeSelection   = handler;
  inter->dataNodeSelection = data;

  inter->onEvent = observe;

  inter->stop = stopPickObserve;

  return inter;
}

void openGLInteractiveBegin_observe(VisuInteractive *inter)
{
  inter->onEvent = observe;
  setObserveEventListener(inter->window);
}
void openGLInteractiveBegin_pickAndObserve(VisuInteractive *inter)
{
  inter->onEvent = pickAndObserve;
  setObserveEventListener(inter->window);
}
void openGLInteractiveBegin_pick(VisuInteractive *inter)
{
  inter->onEvent = pick;
  setPickEventListener(inter->window);
}
void openGLInteractiveBegin_move(VisuInteractive *inter)
{
  /* If this is the first time we begin a move action,
     we initialise the extension list. */
  if (!moveAtomExtension)
    {
      moveAtomExtension_list = openGLObjectList_new(1);
      moveAtomExtension = OpenGLExtension_new(moveAtomName, moveAtomNameI18n,
					      moveAtomDescription,
					      moveAtomExtension_list, NULL);
      OpenGLExtensionSet_priority(moveAtomExtension, OPENGL_EXTENSION_PRIORITY_FIRST + 1);
      OpenGLExtensionRegister(moveAtomExtension);
    }

  inter->onEvent = move;
  setPickEventListener(inter->window);
}
void openGLInteractiveBegin_mark(VisuInteractive *inter)
{
  inter->onEvent = mark;
  setPickEventListener(inter->window);
}

void openGLInteractiveEnd_session(VisuInteractive *inter)
{
  DBG_fprintf(stderr, "Interactive: end interactive session %p.\n", (gpointer)inter);

  removeEventListener(inter->window);
  setInteractiveHandlers(inter->window, (VisuInteractive*)0);
  g_free(inter);
}

void openGLInteractiveHandle_event(SimplifiedEvents *ev, VisuInteractive *inter)
{
  g_return_if_fail(inter);

  if (inter->onEvent && inter->onEvent(ev, inter) && inter->stop)
    inter->stop(inter);
}

PickMesure* openGLInteractiveGet_pickMesure(VisuInteractive *inter)
{
  g_return_val_if_fail(inter, (PickMesure*)0);

  return inter->pick;
}

void openGLInteractiveSet_visuData(VisuInteractive *inter, VisuData *data)
{
  g_return_if_fail(inter && data);

  inter->data = data;
  inter->pick = g_object_get_data(G_OBJECT(inter->data), "pickMesure_data");
  g_return_if_fail(inter->pick);
}

/******************************************************************************/

static gboolean pickAndObserve(SimplifiedEvents *ev, VisuInteractive *inter)
{
  /* If button 1 or 2, use observe mode */
  if (ev->button != 3)
    return observe(ev, inter);
  /* If button 3, use pick mode with button 3 as button 1 */
  else
    {
      if (ev->shiftMod)
	{
	  ev->shiftMod = 0;
	  ev->button = 2;
	}
      else
	ev->button = 1;
      
      return pick(ev, inter);
    }
}

static void freeNodeInfo(gpointer data)
{
  DBG_fprintf(stderr, "Interactives: free list of picked nodes.\n");
  g_list_free(((VisuPick*)data)->nodes);
  g_free(data);
}

static gboolean observe(SimplifiedEvents *ev, VisuInteractive *inter)
{
  int reDrawNeeded;
  int sign_theta;
  int dx, dy;
  OpenGLView *view;
  float angles[3], ratio;
  gboolean zoom;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3 && ev->buttonType == BUTTON_TYPE_PRESS)
    return TRUE;
  /* If the realese event is triggered, we exit only if it's
     not a scroll event (button 4 and 5). */
  if (ev->buttonType == BUTTON_TYPE_RELEASE)
    if (ev->button != 4 && ev->button != 5 )
      {
	if (ev->button == 1)
	  visuDataEmit_observeMovement(inter->data, FALSE);
	return FALSE;
      }

  view = visuDataGet_openGLView(inter->data);
  g_return_val_if_fail(view, FALSE);

/*   fprintf(stderr, "%d %d, %d\n", ev->x, ev->y, ev->button); */
/*   fprintf(stderr, "%d %d, %d\n", ev->shiftMod, ev->controlMod, ev->motion); */
  reDrawNeeded = 0;
  /* Support de la roulette en zoom et perspective. */
  if (ev->specialKey == Key_Page_Up || ev->specialKey == Key_Page_Down ||
      ev->button == 4 || ev->button == 5)
    {
      if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross * 1.1);
      else if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red * 1.1);
    }
  else if (ev->button && !ev->motion)
    {
      inter->observeValues.xi = ev->x;
      inter->observeValues.yi = ev->y;
      if (ev->button == 1)
	visuDataEmit_observeMovement(inter->data, TRUE);
    }
  else if (ev->motion ||
	   ev->specialKey == Key_Arrow_Down  || ev->specialKey == Key_Arrow_Up ||
	   ev->specialKey == Key_Arrow_Right || ev->specialKey == Key_Arrow_Left)
    {
      if (ev->motion)
	{
	  dx =   ev->x - inter->observeValues.xi;
	  dy = -(ev->y - inter->observeValues.yi);
	}
      else
	{
	  dx = 0;
	  dy = 0;
	  if (ev->specialKey == Key_Arrow_Left)
	    dx = -10;
	  else if (ev->specialKey == Key_Arrow_Right)
	    dx = +10;
	  else if (ev->specialKey == Key_Arrow_Down)
	    dy = -10;
	  else if (ev->specialKey == Key_Arrow_Up)
	    dy = +10;
	}
      zoom = (ev->button == 2);
      if(!zoom && !ev->shiftMod && !ev->controlMod)
	{
	  if (openGL_preferedObserveMethod == OPENGL_OBSERVE_CONSTRAINED)
	    {
	      if(view->camera->theta > 0.0)
		sign_theta = 1;
	      else
		sign_theta = -1;
	      openGLViewRotate_box(view, dy * 180.0f / view->window->height,
				   -dx * 180.0f / view->window->width * sign_theta, angles);
	      reDrawNeeded = visuDataSet_angleOfView(inter->data, angles[0], angles[1],
						     0., MASK_THETA | MASK_PHI);
	    }
	  else if (openGL_preferedObserveMethod == OPENGL_OBSERVE_WALKER)
	    {
	      openGLViewRotate_camera(view, dy * 180.0f / view->window->height,
				      -dx * 180.0f / view->window->width, angles);
	      reDrawNeeded = visuDataSet_angleOfView(inter->data, angles[0], angles[1], angles[2],
						     MASK_THETA | MASK_PHI | MASK_OMEGA);
	    }
	}
      else if(!zoom && ev->shiftMod && !ev->controlMod)
	{
	  ratio = 1. / MIN(view->window->width, view->window->height) /
	    view->camera->gross * (view->camera->d_red - 1.f) / view->camera->d_red;
	  reDrawNeeded =
	    visuDataSet_positionOfView(inter->data,
				       view->camera->xs + (float)dx * ratio,
				       view->camera->ys + (float)dy * ratio,
				       MASK_XS | MASK_YS);
	}
      else if(!zoom && ev->controlMod && !ev->shiftMod)
	{
	  if (abs(dx) > abs(dy))
	    reDrawNeeded = visuDataSet_angleOfView(inter->data, 0., 0.,
						   view->camera->omega + dx *
						   180.0f / view->window->width,
						   MASK_OMEGA);
	  else
	    reDrawNeeded = visuDataSet_angleOfView(inter->data, 0., 0.,
						   view->camera->omega + dy *
						   180.0f / view->window->height,
						   MASK_OMEGA);
	}
      else if(zoom && !ev->shiftMod)
	reDrawNeeded = visuDataSet_zoomOfView(inter->data, view->camera->gross *
					      (1. + (float)dy *
					       3.0f / view->window->height));
      else if(zoom && ev->shiftMod)
	reDrawNeeded = visuDataSet_perspectiveOfView(inter->data, view->camera->d_red *
						     (1. - (float)dy *
						      5.0f / view->window->height));
      if (ev->motion)
	{
	  inter->observeValues.xi = ev->x;
	  inter->observeValues.yi = ev->y;
	}
    }
  else if (ev->letter == 'r')
    {
      reDrawNeeded = visuDataSet_angleOfView(inter->data,
					     inter->observeValues.old_theta,
					     inter->observeValues.old_phi,
					     inter->observeValues.old_omega,
					     MASK_THETA | MASK_PHI | MASK_OMEGA) ||
	reDrawNeeded;
      reDrawNeeded = visuDataSet_positionOfView(inter->data,
						inter->observeValues.old_xs,
						inter->observeValues.old_ys,
						MASK_XS | MASK_YS) ||
	reDrawNeeded;
      reDrawNeeded = visuDataSet_zoomOfView(inter->data,
					    inter->observeValues.old_gross) ||
	reDrawNeeded;
      reDrawNeeded = visuDataSet_perspectiveOfView(inter->data,
						   inter->observeValues.old_d_red) ||
	reDrawNeeded;
    }
  else if (ev->letter == 's')
    {
      inter->observeValues.old_theta = view->camera->theta;
      inter->observeValues.old_phi   = view->camera->phi;
      inter->observeValues.old_omega = view->camera->omega;
      inter->observeValues.old_xs    = view->camera->xs;
      inter->observeValues.old_ys    = view->camera->ys;
      inter->observeValues.old_gross = view->camera->gross;
      inter->observeValues.old_d_red = view->camera->d_red;
    }
  if (reDrawNeeded)
    g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		    GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
  return FALSE;
}
static void glDrawSelection(VisuPick *nodeInfo, int x, int y)
{
  int viewport[4];

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glGetIntegerv(GL_VIEWPORT, viewport);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, (float)viewport[2], 0., (float)viewport[3]);   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.f, 1.f, 1.f, 1.f);
  glLineWidth(2);
  glDrawBuffer(GL_FRONT);
  /* We erase the previous selecting rectangle. */
  glBegin(GL_LINE_LOOP);
  glVertex3f(nodeInfo->xOrig, (float)viewport[3] - nodeInfo->yOrig, 0.f);
  glVertex3f(nodeInfo->xPrev, (float)viewport[3] - nodeInfo->yOrig, 0.f);
  glVertex3f(nodeInfo->xPrev, (float)viewport[3] - nodeInfo->yPrev, 0.f);
  glVertex3f(nodeInfo->xOrig, (float)viewport[3] - nodeInfo->yPrev, 0.f);
  glEnd();
  glFlush();

  /* We draw the new selecting rectangle. */
  if (x > 0 && y > 0)
    {
      glBegin(GL_LINE_LOOP);
      glVertex3f(nodeInfo->xOrig, (float)viewport[3] - nodeInfo->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - nodeInfo->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - y, 0.f);
      glVertex3f(nodeInfo->xOrig, (float)viewport[3] - y, 0.f);
      glEnd();
      glFlush();
    }
  glDrawBuffer(GL_BACK);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopAttrib();
}

static gboolean pick(SimplifiedEvents *ev, VisuInteractive *inter)
{
  VisuPick *nodeInfo;
  int nodeId;

  g_return_val_if_fail(ev && inter, TRUE);
  /* We store the pickInfo into the VisuData on press,
     but we apply it only at release if there was no drag action. */
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }

  g_return_val_if_fail(inter->data, TRUE);

  if (ev->button == 1 && ev->motion)
    {
      /* We drag a selecting area. */
      /* A nodeInfo should have been stored, we retrieve it. */
      nodeInfo = g_object_get_data(G_OBJECT(inter->data), "opengl_nodeInfo");
      if (!nodeInfo)
	return FALSE;
      nodeInfo->drag = TRUE;
      DBG_fprintf(stderr, "Interactive: drag to %dx%d.\n", ev->x, ev->y);
      glDrawSelection(nodeInfo, ev->x, ev->y);
      nodeInfo->xPrev = ev->x;
      nodeInfo->yPrev = ev->y;
    }
  else if (ev->buttonType == BUTTON_TYPE_PRESS)
    {
      /* We create and store a nodeInfo data. */
      nodeInfo = g_object_get_data(G_OBJECT(inter->data), "opengl_nodeInfo");
      if (!nodeInfo)
	{
	  nodeInfo = g_malloc(sizeof(VisuPick));
	  g_object_set_data_full(G_OBJECT(inter->data), "opengl_nodeInfo",
				 (gpointer)nodeInfo, freeNodeInfo);
	  nodeInfo->nodes = (GList*)0;
	}
      if (nodeInfo->nodes)
	g_list_free(nodeInfo->nodes);
      nodeInfo->button   = ev->button;
      nodeInfo->x        = ev->x;
      nodeInfo->y        = ev->y;
      nodeInfo->shiftMod = ev->shiftMod;
      nodeInfo->data     = inter->data;
      nodeInfo->drag     = FALSE;
      nodeInfo->xOrig    = ev->x;
      nodeInfo->yOrig    = ev->y;
      nodeInfo->xPrev    = ev->x;
      nodeInfo->yPrev    = ev->y;
      nodeInfo->nodes    = (GList*)0;
    }
  else if (ev->buttonType == BUTTON_TYPE_RELEASE)
    {
      /* A nodeInfo should have been stored, we retrieve it. */
      nodeInfo = g_object_get_data(G_OBJECT(inter->data), "opengl_nodeInfo");
      if (!nodeInfo)
	return FALSE;

      /* If no drag action, we select the node and compute the pick mesure. */
      if (!nodeInfo->drag)
	{
	  g_return_val_if_fail(!nodeInfo->nodes, FALSE);

	  nodeId = getSelectElement(inter->data, nodeInfo->x, nodeInfo->y);

	  DBG_fprintf(stderr, "Interactive: set selection (single).\n");
	  if (nodeInfo->button == 1)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_SELECTED);
	  else if (nodeInfo->button == 2 && !nodeInfo->shiftMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_REFERENCE_1);
	  else if (nodeInfo->button == 2 && nodeInfo->shiftMod)
	    pickMesureSet_pickNode(inter->pick, nodeId, PICK_REFERENCE_2);
	  else
	    return FALSE;
	  DBG_fprintf(stderr, " | OK.\n");
	}
      /* If drag action, we compute all the nodes in the rectangle
	 and we erase it. */
      else
	{
	  /* We unset the pickMesure. */
/* 	  mesureData = pickMesureSet_selection(nodeInfo); */
	  /* We get the list of selected nodes. */
	  glDrawSelection(nodeInfo, -1, -1);
	  /* Get all nodes in the region. */
	  nodeInfo->nodes = getSelectElementsRegion(inter->data,
						    nodeInfo->xOrig, nodeInfo->yOrig,
						    nodeInfo->xPrev, nodeInfo->yPrev);
	  DBG_fprintf(stderr, "Interactive: set selection (drag).\n");
	  pickMesureSet_pickRegion(inter->pick, nodeInfo->nodes);
	  DBG_fprintf(stderr, " | OK.\n");
	  /* We reset the NodeInfo. */
	  g_list_free(nodeInfo->nodes);
	  nodeInfo->nodes = (GList*)0;
	  nodeInfo->drag = FALSE;
	}
      DBG_fprintf(stderr, "Interactive: call callback routine on pick.\n");
      if (inter->onNodeSelection)
	inter->onNodeSelection(inter->pick, inter->dataNodeSelection);
    }
  return FALSE;
}

static gboolean move(SimplifiedEvents *ev, VisuInteractive *inter)
{
  int dx, dy, i;
  float xAxis[3], yAxis[3], zAxis[3], ratio;
  OpenGLView *view;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }
  if (ev->button != 1 &&
      ev->specialKey != Key_Arrow_Left && ev->specialKey != Key_Arrow_Right &&
      ev->specialKey != Key_Arrow_Up && ev->specialKey != Key_Arrow_Down)
    return FALSE;

  g_return_val_if_fail(inter->data, TRUE);

  if (ev->motion == 1)
    {
      DBG_fprintf(stderr, "Interactive: drag action.\n");
      dx =   ev->x - inter->observeValues.xi;
      dy = -(ev->y - inter->observeValues.yi);
      DBG_fprintf(stderr, " | dx x dy : %d x %d\n", dx, dy);

      /* Update stored position for drag info. */
      inter->observeValues.xi = ev->x;
      inter->observeValues.yi = ev->y;

      /* Get the camera orientation. */
      view = visuDataGet_openGLView(inter->data);
      for (i = 0; i < 3; i++)
	{
	  xAxis[i] = 0.;
	  yAxis[i] = 0.;
	  zAxis[i] = 0.;
	}
      if (!ev->shiftMod && !ev->controlMod)
	OpenGLViewGet_screenAxes(view, xAxis, yAxis);
      else
	{
	  if (ev->shiftMod)
	    {
	      xAxis[0] = 1.;
	      yAxis[1] = 1.;
	    }
	  else if (ev->controlMod)
	    zAxis[2] = 1.;
	}
      ratio = OpenGLViewGet_fileUnitPerPixel(view);
      pickMesureSet_dragMove(inter->pick,
			     ratio * (dx * xAxis[0] + dy * yAxis[0] + dy * zAxis[0]),
			     ratio * (dx * xAxis[1] + dy * yAxis[1] + dy * zAxis[1]),
			     ratio * (dx * xAxis[2] + dy * yAxis[2] + dy * zAxis[2]));
    }
/*   else if (ev->specialKey != Key_None) */
/*     { */
/*       DBG_fprintf(stderr, "Interactive: keyboard drag action.\n"); */
      
/*       dx = dy = 0; */
/*       switch (ev->specialKey) */
/* 	{ */
/* 	case Key_Arrow_Left: */
/* 	  dx = -1; dy =  0; break; */
/* 	case Key_Arrow_Right: */
/* 	  dx = +1; dy =  0; break; */
/* 	case Key_Arrow_Up: */
/* 	  dx =  0; dy = +1; break; */
/* 	case Key_Arrow_Down: */
/* 	  dx =  0; dy = -1; break; */
/* 	default: */
/* 	  g_error("Key not handled for drag."); */
/* 	} */

/*       view = visuDataGet_openGLView(inter->data); */
/*       for (i = 0; i < 3; i++) */
/* 	{ */
/* 	  xAxis[i] = 0.; */
/* 	  yAxis[i] = 0.; */
/* 	  zAxis[i] = 0.; */
/* 	} */
/*       if (!ev->shiftMod && !ev->controlMod) */
/* 	OpenGLViewGet_screenAxes(view, xAxis, yAxis); */
/*       else */
/* 	{ */
/* 	  if (ev->shiftMod) */
/* 	    { */
/* 	      xAxis[0] = 1.; */
/* 	      yAxis[1] = 1.; */
/* 	    } */
/* 	  else if (ev->controlMod) */
/* 	    zAxis[2] = 1.; */
/* 	} */
/*       ratio = OpenGLViewGet_fileUnitPerPixel(view); */
/*       pickMesureSet_dragMove(inter->pick, */
/* 			     ratio * (dx * xAxis[0] + dy * yAxis[0] + dy * zAxis[0]), */
/* 			     ratio * (dx * xAxis[1] + dy * yAxis[1] + dy * zAxis[1]), */
/* 			     ratio * (dx * xAxis[2] + dy * yAxis[2] + dy * zAxis[2])); */
/*     } */
  else if (ev->button == 1 && ev->buttonType == BUTTON_TYPE_PRESS)
    {
      pickMesureSet_dragStart(inter->pick,
			      getSelectElement(inter->data, ev->x, ev->y));

      /* Store the position to find the drag values. */
      inter->observeValues.xi = ev->x;
      inter->observeValues.yi = ev->y;
    }
  else if (ev->button == 1 && ev->buttonType == BUTTON_TYPE_RELEASE)
    pickMesureSet_dragStop(inter->pick);

  DBG_fprintf(stderr, "Interactive: call callback routine on move.\n");
  if (inter->onNodeSelection)
    inter->onNodeSelection(inter->pick, inter->dataNodeSelection);
  return FALSE;
}
void openGLInteractiveStart_move(VisuInteractive *inter, GList *nodes)
{
  GList *tmpLst;
  OpenGLView *view;

  DBG_fprintf(stderr, "Interactive: start to drag a list of %d nodes.\n",
	      g_list_length(nodes));
  /* Hide all nodes. */
  tmpLst = nodes;
  while (tmpLst)
    {
      visuNodeSet_visibility((VisuNode*)tmpLst->data, FALSE);
      tmpLst = g_list_next(tmpLst);
    }
  if (nodes && !nodes->next)
    visuData_createNodes(inter->data, inter->data->fromIntToVisuElement[((VisuNode*)nodes->data)->posElement]);
  else
    visuData_createAllNodes(inter->data);

  /* Show again the selected nodes and add it in
     the specific moveAtom list. */
  view = visuDataGet_openGLView(inter->data);
  glNewList(moveAtomExtension_list, GL_COMPILE);
  glPushMatrix();
  glTranslated(- view->box->dxxs2,
	       - view->box->dyys2,
	       - view->box->dzzs2);
  tmpLst = nodes;
  while (tmpLst)
    {
      visuNodeSet_visibility((VisuNode*)tmpLst->data, TRUE);
      visuData_createNode(inter->data, (VisuNode*)tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  glPopMatrix();
  glEndList();
  moveAtomExtension->used = 1;

  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
}
void openGLInteractiveStop_move(VisuInteractive *inter, GList *nodes)
{
  DBG_fprintf(stderr, "Interactive: stop dragging a list of %d nodes.\n",
	      g_list_length(nodes));
  /* Show again the selected element in the allElement list. */
  if (nodes && !nodes->next)
    visuData_createNodes(inter->data, inter->data->fromIntToVisuElement[((VisuNode*)nodes->data)->posElement]);
  else
    visuData_createAllNodes(inter->data);

  /* Stop the move extension. */
  moveAtomExtension->used = 0;

  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
  DBG_fprintf(stderr, "Restore standard allElement after move.\n");
}
void openGLInteractiveMove(VisuInteractive *inter, GList *nodes, float drag[3])
{
  OpenGLView *view;
  VisuNode *node;

  DBG_fprintf(stderr, "Interactive: drag a list of %d nodes of %gx%gx%g.\n",
	      g_list_length(nodes), drag[0], drag[1], drag[2]);
  view = visuDataGet_openGLView(inter->data);
  glNewList(moveAtomExtension_list, GL_COMPILE);
  glPushMatrix();
  glTranslated(- view->box->dxxs2,
	       - view->box->dyys2,
	       - view->box->dzzs2);
  while (nodes)
    {
      node = (VisuNode*)nodes->data;
      node->xyz[0] += drag[0];
      node->xyz[1] += drag[1];
      node->xyz[2] += drag[2];
      visuData_createNode(inter->data, node);
      nodes = g_list_next(nodes);
    }
  glPopMatrix();
  glEndList();
  visuDataEmit_nodePositionChanged(inter->data);
	  
  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visuObjectRedraw,
		  GINT_TO_POINTER(TRUE), (GDestroyNotify)0);
}

static gboolean mark(SimplifiedEvents *ev, VisuInteractive *inter)
{
  GList *lst;
  int nodeId;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3 && ev->buttonType == BUTTON_TYPE_PRESS)
    return TRUE;
  if (ev->buttonType == BUTTON_TYPE_RELEASE)
    return FALSE;

  g_return_val_if_fail(inter->data, TRUE);

  nodeId = getSelectElement(inter->data, ev->x, ev->y);
  if (nodeId < 0)
    return FALSE;

  lst = (GList*)0;
  lst = g_list_append(lst, GINT_TO_POINTER(nodeId));
  pickMesureToggle_highlight(inter->pick, lst);
  g_list_free(lst);

  DBG_fprintf(stderr, "Interactive: call callback routine on mark.\n");
  if (inter->onNodeSelection)
    inter->onNodeSelection(inter->pick, inter->dataNodeSelection);

  return FALSE;
}

static GList* getSelectElementsRegion(VisuData *dataObj, int x1, int y1,
					 int x2, int y2)
{
   GLsizei bufsize;
   GLuint *select_buf;
   GLint viewport[4] = {0, 0, 0, 0};
   int hits, names, ptr, i;
   OpenGLView *view;
   GList *lst;
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), (GList*)0);

   DBG_fprintf(stderr, "Interactive: get elements in region %dx%d - %dx%d.\n",
	       x1, x2, y1, y2);

   if ((x1 == x2) || (y1 == y2))
     return (GList*)0;

   view = visuDataGet_openGLView(dataObj);

   bufsize = visuDataGet_nodeArray(dataObj)->nbOfAllStoredNodes * 4;
   select_buf = g_malloc(sizeof(GLuint) * bufsize);
   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
   
   lst = (GList*)0;
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;

   glNewList(10, GL_COMPILE);
   gluPickMatrix(0.5f * (x1 + x2) , (float)view->window->height - 0.5f * (y1 + y2),
		 (float)ABS(x2 - x1), (float)ABS(y2 - y1), viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 

   glCallList(visuDataGet_objectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "%d elements are on the z buffer %dx%d - %dx%d.\n", hits,
	       x1, y1, x2, y2);
   ptr = 0;

   /* return the buffer to normal */
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   for(i=0; i<hits; i++)
     {
       names = select_buf[ptr];
       if (names != 1)
	 {
	   g_warning("OpenGL picking is not working???\n");
	   return (GList*)0;
	 }
       ptr = ptr + 3;
       lst = g_list_prepend(lst, GINT_TO_POINTER((int)select_buf[ptr]));
       ptr = ptr + 1;
     }
   g_free(select_buf);

   return lst;
}

static int getSelectElement(VisuData *dataObj, int x, int y)
{
#define bufsize 512
   GLuint select_buf[bufsize];
   GLint viewport[4] = {0, 0, 0, 0};
#define wpck 2.0
#define hpck 2.0
   int hits, names, ptr, i;
   unsigned int z1;
   unsigned int z1_sauve = UINT_MAX;
   int number;
   int found;
   OpenGLView *view;
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0);

   view = visuDataGet_openGLView(dataObj);

   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
      
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;
   glNewList(10, GL_COMPILE);
      gluPickMatrix(1.0*x, 1.0*(view->window->height-y), wpck, hpck, viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 

   glCallList(visuDataGet_objectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "%d elements are on the z buffer.\n", hits);
   ptr = 0;

   /* return the buffer to normal */
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   found = 0;
   number = -1;
   for(i=0; i<hits; i++) {
      names = select_buf[ptr];
      if (names != 1)
	{
	  g_warning("OpenGL picking is not working???\n");
	  return -1;
	}
      ptr = ptr + 1;
      z1 = select_buf[ptr];
      DBG_fprintf(stderr, " | z position %f for %d\n", (float)z1/0x7fffffff,
		  (int)select_buf[ptr + 2]);
      ptr = ptr + 2;
      if (z1 < z1_sauve) {
         z1_sauve = z1;
	 number = (int)select_buf[ptr];
	 found = 1;
      }
      ptr = ptr + 1;
   }
   if (found && number >= 0)
     return number;
   else
     return -1;
}

static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj _U_, GError **error)
{
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_integer(lines[0], position, &val, 1, error))
    return FALSE;
  if (val != OPENGL_OBSERVE_CONSTRAINED && val != OPENGL_OBSERVE_WALKER)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, 500);
      return FALSE;
    }
  openGLInteractiveSet_preferedObserveMethod(val);

  return TRUE;
}
static gboolean exportParameters(GString *data, int *nbLinesWritten,
				 VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OBSERVE_METHOD);
  g_string_append_printf(data, "%s: %d\n\n", FLAG_PARAMETER_OBSERVE_METHOD,
	  openGL_preferedObserveMethod);
  *nbLinesWritten = 3;
  return TRUE;
}
