/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_pairs.h"
#include "pairsModeling/externalPairsExtensions.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include "visu_tools.h"
#include "visu_object.h"
#include "visu_nodes.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "visu_extension.h"
#include "visu_configFile.h"
#include "renderingBackend/visu_windowInterface.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/dataNode.h"

/**
 * SECTION:visu_pairs
 * @short_description: V_Sim can draw link between nodes. This part
 * defines a pair object and interface to draw pairs.
 *
 * <para>The visu_pairs.c defines only general methods to draw
 * pairs. It introduces a new object called #PairsData. This stores
 * some characteristics on links between two #VisuElement. The main
 * characteristic is that pairs are drawn only if the length between
 * two nodes is in a specific range. Use setPairsDistance() and
 * getPairsDistance() to tune this range.</para>
 *
 * <para>This file does not draw any pairs. But it gives some
 * interface to create rendering capabilities. To create a new pair
 * rendering module, called #PairsExtension, use
 * pairsExtension_new(). Basically, a #PairsExtension is characterized
 * by it drawing method. But it can have other methods that are called
 * in different cases. See initEndOpenGlPairsFunc() and
 * startEndPairsFunc() prototypes to have more informations.</para>
 */

#define PAIRS_ONOFF_DEFAULT 0

/* This structure is made to store pairs information between two
   elements. */
struct VisuPair_struct
{
  VisuElement *ele1;
  VisuElement *ele2;

  /* This is a GList of link (VisuPairData). */
  GList *pairs;

  /* A list of properties. */
  GHashTable *properties;
};

struct foreachPairsData_struct
{
  foreachPairsFunc func;
  gpointer userData;
};

#define BONDHISTOGRAM_ID   "bondDistribution_data"
#define BONDHISTOGRAM_STEP 0.1f
#define BONDHISTOGRAM_MAX  10.f

OpenGLExtension* extensionPairs;
int openGlListPairsId;

/* This hashtable as VisuElement* as keys and pointer to other hashtable as value.
   The main idea is to have an hashtable of 2 dimension, each two keys are
   VisuElement* and the final values are PairsData. Nevertheless, this hashtable
   must not be access directly but through the get and put methods. */
GHashTable *DminDmax;

GList *availablePairsExtensions;
PairsExtension *currentPairsExtension;
gboolean rebuildPairsNeeded;

/* Internal variables. */
#define BONDNUMBER_ID "bondNumber_data"
static DataNode *dataNode;

/* Parameters */
static Color *defaultPairColor;

/* Local methods. */
static void freeData(gpointer obj, gpointer data);
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data);
static int* getBond(VisuData *dataObj, VisuNode *node);
static void freePair(gpointer ele);
static void freePairData(VisuPairData *data);
static void freeHistoData(gpointer data);
static VisuPairData* newPairData(float minMax[2]);
static VisuPair* newPair(VisuElement *ele1, VisuElement *ele2);


/* Callbacks. */
static void onNewData(GObject* obj, VisuData *dataObj, gpointer data);
static void onDataReady(GObject* obj, VisuData *dataObj, gpointer data);
void rebuildPairs(VisuData *dataObj);
void facetteChanged(VisuData* obj, gpointer data);
void createPairsOnNodeRenderedChanged(VisuData *dataObj, gpointer data);
void createPairsOnElementRenderedChanged(VisuData *dataObj, VisuElement *element, gpointer data);
void createPairsOnNodePositionChanged(VisuData *dataObj, gpointer data);
static void createPairsOnNodePopulationChanged(VisuData *dataObj,
					       int *nodes, gpointer data);
static void createPairsOnResources(GObject *obj, VisuData *dataObj, gpointer bool);

/* This function details how to read what is needed by PairsData. */
static gboolean readPairsData(gchar **lines, int nbLines, int position,
			      VisuData *dataObj, GError **error);
static gboolean readPairLink(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static gboolean readPairsAreOn(gchar **lines, int nbLines, int position,
			       VisuData *dataObj, GError **error);
static gboolean readFavPairsMethod(gchar **lines, int nbLines, int position,
				   VisuData *dataObj, GError **error);
/* This function save the resources. */
static gboolean exportResourcesPairs(GString *data, int *nbLinesWritten,
				     VisuData *dataObj);
#define FLAG_RESOURCES_PAIRS      "pairs_are_on"
#define DESC_RESOURCES_PAIRS      "Ask the opengl engine to draw pairs between elements ; boolean 0 or 1"
#define RESOURCES_PAIRS_DEFAULT FALSE

#define FLAG_RESOURCES_PAIRS_DATA "pair_data"
#define DESC_RESOURCES_PAIRS_DATA "Draw pairs between [ele1] [ele2] [0. <= dmin] [0. <= dmax] [0. <= RGB <= 1.]x3"
#define FLAG_RESOURCES_PAIR_LINK "pair_link"
#define DESC1_RESOURCES_PAIR_LINK "Draw a link between [ele1] [ele2] [0. <= dmin] [0. <= dmax]"
#define DESC2_RESOURCES_PAIR_LINK "                    [0. <= RGB <= 1.]x3 [bool: drawn] [bool: printLength]"

#define FLAG_RESOURCES_FAV_PAIRS  "pairs_favoriteMethod"
#define DESC_RESOURCES_FAV_PAIRS  "Favorite method used to render files ; chain"



int initPairsModule()
{
  char *name = "Pairs";
  char *description = _("Draw pairs between elements with a criterion of distance.");
  int i, res;
  PairsExtension *extension;
  float rgbOfPairs[4] = {1.0, 0.6, 0.2, 1.};
  VisuConfigFileEntry *resourceEntry, *oldEntry;

  openGlListPairsId = openGLObjectList_new(1);
  extensionPairs = OpenGLExtension_new(name, _(name), description,
				       openGlListPairsId, rebuildPairs);
  OpenGLExtensionSet_priority(extensionPairs, OPENGL_EXTENSION_PRIORITY_LOW);
  OpenGLExtensionSet_sensitiveToRenderingMode(extensionPairs, TRUE);
  extensionPairs->used = PAIRS_ONOFF_DEFAULT;
  OpenGLExtensionRegister(extensionPairs);

  /* Create a VisuModule to registered the new resources as
     rgb and material. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_PAIRS,
					  DESC_RESOURCES_PAIRS,
					  1, readPairsAreOn);
  oldEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				     FLAG_RESOURCES_PAIRS_DATA,
				     DESC_RESOURCES_PAIRS_DATA,
				     1, readPairsData);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_PAIR_LINK,
					  DESC1_RESOURCES_PAIR_LINK,
					  2, readPairLink);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileSet_replace(resourceEntry, oldEntry);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_FAV_PAIRS,
					  DESC_RESOURCES_FAV_PAIRS,
					  1, readFavPairsMethod);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesPairs);


  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
		   G_CALLBACK(createPairsOnResources), (gpointer)0);

  DminDmax = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, freePair);
  
  defaultPairColor = colorNew_floatRGBA(rgbOfPairs);
  colorAdd_color(defaultPairColor);

  /* Initialise the pairs extensions. */
  availablePairsExtensions = (GList*)0;
  res = 1;
  for (i = 0; listInitPairsFunc[i]; i++)
    {
      extension = listInitPairsFunc[i]();
      if (!extension)
	res = 0;
      visuPairExtensionAdd(extension);
    }
  if (!res)
    g_warning("Some pairs extensions can't initialse.\n");

  rebuildPairsNeeded = TRUE;
  
  if (availablePairsExtensions)
    currentPairsExtension = (PairsExtension*)availablePairsExtensions->data;
  else
    currentPairsExtension = (PairsExtension*)0;

  /* Register a new NodeData. */
  dataNode = nodeDataNew(BONDNUMBER_ID, G_TYPE_INT);
  nodeDataSet_label(dataNode, _("Bonds"));

  return res;
}

static void freePair(gpointer ele)
{
  VisuPair *pair;
  GList *tmpLst;

  pair = (VisuPair*)ele;
  DBG_fprintf(stderr, "Visu Pairs: freeing pair between '%s' and '%s'.\n",
	      pair->ele1->name, pair->ele2->name);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    freePairData((VisuPairData*)tmpLst->data);
  g_list_free(pair->pairs);
  g_hash_table_destroy(pair->properties);
}
static void freePairData(VisuPairData *data)
{
  DBG_fprintf(stderr, "Visu Pairs: freeing link.\n");
  g_hash_table_destroy(data->properties);
}

static VisuPairData* newPairData(float minMax[2])
{
  VisuPairData *data;
  Color *color;

  /* We create one link. */
  data = g_malloc(sizeof(VisuPairData));
  data->minMax[PAIRS_MIN] = minMax[PAIRS_MIN];
  data->minMax[PAIRS_MAX] = minMax[PAIRS_MAX];
  data->drawn             = TRUE;
  data->printLength       = FALSE;
  data->properties        = g_hash_table_new_full(g_str_hash, g_str_equal,
						  NULL, g_free);
  g_return_val_if_fail(data->properties, (VisuPairData*)0);

  /* We create the color property . */
  color = g_malloc(sizeof(Color));
  colorCopy_color(color, defaultPairColor);
  g_hash_table_insert(data->properties, (gpointer)"color", (gpointer)color);

  return data;
}

/* Create a new PairsData structure with default values.
   The newly created structure can be freed by a call to free. */
static VisuPair* newPair(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;

  g_return_val_if_fail(ele1 && ele2, (VisuPair*)0);

  DBG_fprintf(stderr, "Visu Pairs: create a new pair between '%s' and '%s'.\n",
	      ele1->name, ele2->name);

  pair        = g_malloc(sizeof(VisuPair));
  pair->ele1  = ele1;
  pair->ele2  = ele2;
  pair->pairs = (GList*)0;
  pair->properties = g_hash_table_new_full(g_str_hash, g_str_equal,
					   NULL, g_free);
  g_return_val_if_fail(pair->properties, (VisuPair*)0);

/*   pair->pairs = g_list_prepend(pair->pairs, (gpointer)newPairData(minMax)); */

  return pair;
}

gboolean visuPairSet_status(gboolean onOff)
{
  GList *allObj;

  if (onOff == extensionPairs->used)
    return FALSE;

  extensionPairs->used = onOff;
  DBG_fprintf(stderr, "Visu Pairs : change pairs on/off status (rebuild"
	      " flag is %d).\n", rebuildPairsNeeded);

  /* Change the status of the bond count. */
  allObj = visuDataGet_allObjects();
  if (allObj)
    nodeDataSet_used(dataNode, VISU_DATA(allObj->data), (onOff)?1:0);
/*   visuPairBuild(); */
  return TRUE;
}
void visuPairSet_outOfDate()
{
  rebuildPairsNeeded = TRUE;
}
gboolean visuPairExtensionSet(PairsExtension *extension)
{
  DBG_fprintf(stderr, "Visu Pairs : set pairs drawing method to '%s'"
	      " (previous '%s').\n", extension->name, currentPairsExtension->name);

  if (extension == currentPairsExtension)
    return FALSE;
  
  currentPairsExtension = extension;

  rebuildPairsNeeded = TRUE;

  if (!extensionPairs->used)
    return FALSE;
  return TRUE;
}
gboolean visuPairSet_drawn(VisuPairData *data, gboolean drawn)
{
  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set drawn status %d (%d) for %p.\n",
	      (int)drawn, (int)data->drawn, (gpointer)data);
  if (data->drawn == drawn)
    return FALSE;
  data->drawn = drawn;

  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}
gboolean visuPairSet_printLength(VisuPairData *data, gboolean status)
{
  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set print length status %d (%d) for %p.\n",
	      (int)status, (int)data->printLength, (gpointer)data);
  if (data->printLength == status)
    return FALSE;

  data->printLength = status;
  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}

gboolean visuPairGet_drawn(VisuPairData *data)
{
  g_return_val_if_fail(data, FALSE);
  return data->drawn;
}
gboolean visuPairSet_color(VisuPairData *data, Color* destColor)
{
  Color *color;

  g_return_val_if_fail(data && destColor, FALSE);

  DBG_fprintf(stderr, "Visu Pairs: set color [%g;%g;%g] for %p.\n",
	      destColor->rgba[0], destColor->rgba[1], destColor->rgba[2],
	      (gpointer)data);

  color = (Color*)g_hash_table_lookup(data->properties, (gpointer)"color");
  if (!color)
    {
      color = g_malloc(sizeof(Color));
      g_hash_table_insert(data->properties, (gpointer)"color", (gpointer)color);
    }
  else
    if (colorEqual_color(color, destColor))
      return FALSE;

  /* Copy values of dest to current color. */
  colorCopy_color(color, destColor);

  rebuildPairsNeeded = TRUE;
  return extensionPairs->used;
}
gboolean visuPairSet_distance(VisuPairData *data, float val, int minOrMax)
{
  gboolean drawn;

  g_return_val_if_fail(data && (minOrMax == PAIRS_MIN ||
				minOrMax == PAIRS_MAX), FALSE);

  if (data->minMax[minOrMax] == val)
    return FALSE;

  data->minMax[minOrMax] = val;
  drawn = (data->minMax[PAIRS_MIN] < data->minMax[PAIRS_MAX] &&
	   data->minMax[PAIRS_MAX] > 0.f);
  rebuildPairsNeeded = (drawn || (drawn != data->drawn));
  data->drawn = drawn;

  return rebuildPairsNeeded && extensionPairs->used;
}
void visuPairSet_linkProperty(VisuPairData *data, const gchar* key, gpointer value)
{
  if (data && key)
    g_hash_table_insert(data->properties, (gpointer)key, (gpointer)value);

  rebuildPairsNeeded = TRUE;
}
void visuPairSet_property(VisuPair *pair, const gchar* key,
			  gpointer value, GDestroyNotify freeFunc)
{
  Property *prop;

  g_return_if_fail(pair && key && *key);

  prop           = g_malloc(sizeof(Property));
  prop->name     = key;
  prop->data     = value;
  prop->freeFunc = freeFunc;
  g_hash_table_insert(pair->properties, (gpointer)key, (gpointer)prop);
}
gpointer visuPairGet_property(VisuPair *pair, const gchar* key)
{
  Property *prop;

  g_return_val_if_fail(pair, (gpointer)0);

  prop = (Property*)g_hash_table_lookup(pair->properties, (gpointer)key);
  if (prop)
    return prop->data;
  else
    return (gpointer)0;
}

gboolean visuPairGet_status()
{
  return (gboolean)extensionPairs->used;
}
gboolean visuPairGet_printLength(VisuPairData *data)
{
  g_return_val_if_fail(data, FALSE);
  return data->printLength;
}
float visuPairGet_distance(VisuPairData *data, int minOrMax)
{
  g_return_val_if_fail(data, 0.);
  g_return_val_if_fail(minOrMax == PAIRS_MIN || minOrMax == PAIRS_MAX, 0.);

  return data->minMax[minOrMax];
}
gpointer visuPairGet_linkProperty(VisuPairData *data, const gchar* key)
{
  if (data)
    return g_hash_table_lookup(data->properties, (gpointer)key);
  else
    return (gpointer)0;
}

VisuPair* visuPairGet_pair(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;
  gchar *key;

  g_return_val_if_fail(ele1 && ele2, (VisuPair*)0);

  if (strcmp(ele1->name, ele2->name) < 0)
    key = g_strdup_printf("%s %s", ele1->name, ele2->name);
  else
    key = g_strdup_printf("%s %s", ele2->name, ele1->name);
  pair = (VisuPair*)g_hash_table_lookup(DminDmax, (gpointer)key);
  DBG_fprintf(stderr, "Visu Pairs: test key '%s' -> %p.\n", key, (gpointer)pair);
  if (!pair)
    {
      /* Ok, create one if none found. */
      pair = newPair(ele1, ele2);
      g_hash_table_insert(DminDmax, (gpointer)key, (gpointer)pair);
    }
  else
    g_free(key);

  return pair;
}
GList* visuPairGet_links(VisuElement *ele1, VisuElement *ele2)
{
  VisuPair *pair;
  float minMax[2] = {0.f, 0.f};

  pair = visuPairGet_pair(ele1, ele2);
  if (!pair->pairs)
    pair->pairs = g_list_append(pair->pairs, (gpointer)newPairData(minMax));
  return pair->pairs;
}
VisuPairData* visuPairGet_link(VisuElement *ele1, VisuElement *ele2, float minMax[2])
{
  VisuPair *pair;
  GList *tmpLst;
  VisuPairData *data;

  g_return_val_if_fail(minMax, (VisuPairData*)0);

  pair = visuPairGet_pair(ele1, ele2);
  g_return_val_if_fail(pair, (VisuPairData*)0);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      data = (VisuPairData*)tmpLst->data;
      DBG_fprintf(stderr, " | test %p (%g %g).\n", tmpLst->data,
		  data->minMax[0], data->minMax[1]);
      if (data->minMax[0] == minMax[0] && data->minMax[1] == minMax[1])
	return data;
    }
  data = newPairData(minMax);
  pair->pairs = g_list_append(pair->pairs, (gpointer)data);
  DBG_fprintf(stderr, " | new %p (%g %g).\n", (gpointer)data,
	      data->minMax[0], data->minMax[1]);
  rebuildPairsNeeded = TRUE;

  return data;
}
gboolean visuPairRemove_link(VisuElement *ele1, VisuElement *ele2, VisuPairData *data)
{
  VisuPair *pair;
  GList *tmpLst;

  g_return_val_if_fail(data, FALSE);

  pair = visuPairGet_pair(ele1, ele2);
  g_return_val_if_fail(pair, FALSE);
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      if ((VisuPairData*)tmpLst->data == data)
	{
	  freePairData(data);
	  if (tmpLst->prev)
	    tmpLst->prev->next = tmpLst->next;
	  if (tmpLst->next)
	    tmpLst->next->prev = tmpLst->prev;
	  if (pair->pairs == tmpLst)
	    pair->pairs = tmpLst->next;
	  g_list_free_1(tmpLst);
	  
	  rebuildPairsNeeded = TRUE;
	  return TRUE;
	}
    }
  return FALSE;
}
VisuPairData* visuPairGet_linkFromId(VisuElement *ele1, VisuElement *ele2, guint pos)
{
  VisuPair *pair;

  pair = visuPairGet_pair(ele1, ele2);
  g_return_val_if_fail(pair->pairs, (VisuPairData*)0);
  return (VisuPairData*)(g_list_nth(pair->pairs, pos)->data);
}


void createPairsOnNodeRenderedChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs : catch 'NodeRenderedChanged' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visuPairBuild(dataObj);
}
void createPairsOnElementRenderedChanged(VisuData *dataObj,
					 VisuElement *element _U_, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs : catch 'ElementRenderedChanged' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visuPairBuild(dataObj);
}
void createPairsOnNodePositionChanged(VisuData *dataObj, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs : catch 'NodePositionChanged' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visuPairBuild(dataObj);
}
static void createPairsOnNodePopulationChanged(VisuData *dataObj,
					       int *nodes _U_, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Pairs : catch 'NodePopulation[In|De]crease' signal,"
	      " recreating pairs...\n");
  rebuildPairsNeeded = TRUE;
  visuPairBuild(dataObj);
}
static void createPairsOnResources(GObject *obj _U_, VisuData *dataObj,
				   gpointer data _U_)
{
  if (!rebuildPairsNeeded)
    return;

  DBG_fprintf(stderr, "Visu Pairs : catch 'resourcesLoaded' signal,"
	      " recreating pairs...\n");
  visuPairBuild(dataObj);
}

static void freeHistoData(gpointer data)
{
  DBG_fprintf(stderr, "Visu Pairs: free '%s' data.\n", BONDHISTOGRAM_ID);
  g_free(((VisuPairDistribution*)data)->histo);
  g_free(data);
}

VisuPairDistribution* visuPairGet_distanceDistribution(VisuPair *pair,
						       VisuData *dataObj,
						       float step, float max)
{
  VisuPairDistribution *dd;
  int nRef, *iEle;
  VisuDataIter iter1, iter2;
  float d2;
  float xyz1[3], xyz2[3];

  g_return_val_if_fail(pair && IS_VISU_DATA_TYPE(dataObj), (VisuPairDistribution*)0);

  /* We create the storage structure. */
  dd = (VisuPairDistribution*)
    visuPairGet_property(pair, BONDHISTOGRAM_ID);
  if (!dd)
    {
      dd            = g_malloc(sizeof(VisuPairDistribution));
      dd->stepValue = (step > 0.f)?step:BONDHISTOGRAM_STEP;
      dd->nValues   = (int)((max > 0.f)?max:BONDHISTOGRAM_MAX / dd->stepValue) + 1;
      dd->histo     = g_malloc(sizeof(int) * dd->nValues);
      visuPairSet_property(pair, BONDHISTOGRAM_ID,
			   (gpointer)dd, freeHistoData);
    }
  else
    {
      dd->stepValue = (step > 0.f)?step:BONDHISTOGRAM_STEP;
      nRef = (int)(((max > 0.f)?max:BONDHISTOGRAM_MAX) / dd->stepValue) + 1;
      if (dd->nValues != nRef)
	{
	  dd->nValues = nRef;
	  dd->histo   = g_realloc(dd->histo, sizeof(int) * dd->nValues);
	}
    }
  DBG_fprintf(stderr, "Visu Pairs: compute distance distribution (%p %g %d).\n",
	      (gpointer)dd, dd->stepValue, dd->nValues);
  dd->nNodesEle1 = 0;
  dd->nNodesEle2 = 0;
  memset(dd->histo, 0, sizeof(int) * dd->nValues);

  /* We compute the distribution. */
  visuDataIter_new(dataObj, &iter1);
  iEle = (int*)g_hash_table_lookup(dataObj->fromVisuElementToInt, pair->ele1);
  g_return_val_if_fail(iEle, (VisuPairDistribution*)0);
  iter1.iElement = *iEle;
  iter1.element = pair->ele1;
  for(visuDataIter_restartNode(dataObj, &iter1); iter1.node;
      visuDataIter_nextNode(dataObj, &iter1))
    {
      if (!iter1.node->rendered)
	continue;
      dd->nNodesEle1 += 1;

      visuDataIter_new(dataObj, &iter2);
      iEle = (int*)g_hash_table_lookup(dataObj->fromVisuElementToInt, pair->ele2);
      g_return_val_if_fail(iEle, (VisuPairDistribution*)0);
      iter2.iElement = *iEle;
      iter2.element = pair->ele2;
      for(visuDataIter_restartNode(dataObj, &iter2); iter2.node;
	  visuDataIter_nextNode(dataObj, &iter2))
	{
	  if (!iter2.node->rendered)
	    continue;
	  /* Don't count the inter element pairs two times. */
	  if (iter1.element == iter2.element &&
	      iter2.node >= iter1.node)
	    break;

	  visuDataGet_nodePosition(dataObj, iter1.node, xyz1);
	  visuDataGet_nodePosition(dataObj, iter2.node, xyz2);
	  d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
	    (xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
	    (xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
	  /* We put the distance into the histogram. */
	  dd->histo[MIN((int)(sqrt(d2) / dd->stepValue),
			dd->nValues)] += 1;
	}
    }
  for(visuDataIter_restartNode(dataObj, &iter2); iter2.node;
      visuDataIter_nextNode(dataObj, &iter2))
    if (iter2.node->rendered)
      dd->nNodesEle2 += 1;

  return dd;
}

gboolean visuPairBuild(VisuData *dataObj)
{
  VisuDataIter iter1, iter2;
  float d2, d2min, d2max, d2min_buffered, d2max_buffered, l;
  VisuPair *pair;
  VisuPairData *data;
  float xyz1[3], xyz2[3], alpha;
  OpenGLView *view;
  char distStr[8];
  int *bond1, *bond2;
  GList *tmpLst;
  gboolean useBond;

  if (!dataObj)
    {
      glDeleteLists(openGlListPairsId, 1);
      rebuildPairsNeeded = TRUE;
      return TRUE;
    }

  if (!extensionPairs->used)
    {
      DBG_fprintf(stderr, "Visu Pairs : aborting creating pairs, extension is not used.\n");
      return FALSE;
    }
  if (!rebuildPairsNeeded)
    {
      DBG_fprintf(stderr, "Visu Pairs : aborting creating pairs, pairs are up to date.\n");
      return TRUE;
    }

  DBG_fprintf(stderr, "Visu Pairs : creating pairs between elements (%s).\n",
	      currentPairsExtension->name);

  view = visuDataGet_openGLView(dataObj);

  glDeleteLists(openGlListPairsId, 1);
  g_return_val_if_fail(currentPairsExtension, FALSE);

  rebuildPairsNeeded = FALSE;
  glNewList(openGlListPairsId, GL_COMPILE);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  if (currentPairsExtension->initOpenGl)
    {
      DBG_fprintf(stderr, "Visu Pairs : call to init OpenGl (%d).\n",
		  GPOINTER_TO_INT(currentPairsExtension->initOpenGl));
      currentPairsExtension->initOpenGl();
    }

  /* We get the counting array for bonds. */
  if (visuNodeGet_property(visuDataGet_nodeArray(dataObj),
			   (const gchar*)BONDNUMBER_ID))
    useBond = TRUE;
  else
    useBond = FALSE;

  visuDataIter_new(dataObj, &iter1);
  visuDataIter_new(dataObj, &iter2);
  for(visuDataIter_start(dataObj, &iter1); iter1.element;
      visuDataIter_nextElement(dataObj, &iter1))
    {
      if (!visuElementGet_rendered(iter1.element))
	continue;

      /* Initialise the bond count. */
      if (useBond)
	for(visuDataIter_restartNode(dataObj, &iter1); iter1.node;
	    visuDataIter_nextNode(dataObj, &iter1))
	  {
	    bond1 = getBond(dataObj, iter1.node);
	    *bond1 = 0;
	  }

      for(visuDataIter_start(dataObj, &iter2);
	  iter2.element && iter2.iElement <= iter1.iElement ;
	  visuDataIter_nextElement(dataObj, &iter2))
	{
	  if (!visuElementGet_rendered(iter2.element))
	    continue;
	  
	  DBG_fprintf(stderr, "Visu Pairs: draw pairs between '%s' and '%s'.\n",
		      iter1.element->name, iter2.element->name);
	  pair = visuPairGet_pair(iter1.element, iter2.element);
	  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
	    {
	      data = (VisuPairData*)tmpLst->data;
	      if (!data->drawn)
		continue;
	      d2min = data->minMax[PAIRS_MIN] * data->minMax[PAIRS_MIN];
	      d2max = data->minMax[PAIRS_MAX] * data->minMax[PAIRS_MAX];
	      if(d2min >= d2max || d2max <= 0.)
		continue;

	      l = data->minMax[PAIRS_MAX] - data->minMax[PAIRS_MIN];
	      d2min_buffered = (data->minMax[PAIRS_MIN] - 0.15 * l);
	      d2min_buffered *= d2min_buffered;
	      d2max_buffered = (data->minMax[PAIRS_MAX] + 0.15 * l);
	      d2max_buffered *= d2max_buffered;
	      if (currentPairsExtension->beginDrawingPairs)
		currentPairsExtension->beginDrawingPairs(iter1.element,
							 iter2.element, data);

	      for(visuDataIter_restartNode(dataObj, &iter1); iter1.node;
		  visuDataIter_nextNode(dataObj, &iter1))
		{
		  if (!iter1.node->rendered)
		    continue;

		  bond1 = (useBond)?getBond(dataObj, iter1.node):(int*)0;
		  for(visuDataIter_restartNode(dataObj, &iter2); iter2.node;
		      visuDataIter_nextNode(dataObj, &iter2))
		    {
		      if (!iter2.node->rendered)
			continue;
		      /* Don't draw the inter element pairs two times. */
		      if (iter1.element == iter2.element &&
			  iter2.node >= iter1.node)
			break;

		      visuDataGet_nodePosition(dataObj, iter1.node, xyz1);
		      visuDataGet_nodePosition(dataObj, iter2.node, xyz2);
		      d2 = (xyz1[0] - xyz2[0]) * (xyz1[0] - xyz2[0]) + 
			(xyz1[1] - xyz2[1]) * (xyz1[1] - xyz2[1]) + 
			(xyz1[2] - xyz2[2]) * (xyz1[2] - xyz2[2]);
		      if(d2 <= 0. || d2 < d2min_buffered || d2 > d2max_buffered)
			continue;

		      if (d2 < d2min)
			alpha = (d2 - d2min_buffered) /
			  (d2min - d2min_buffered);
		      else if (d2 > d2max)
			alpha = (d2max_buffered - d2) /
			  (d2max_buffered - d2max);
		      else
			{
			  alpha = 1.f;
			  /* Update bond count. */
			  if (bond1)
			    (*bond1) += 1;
			  bond2 = (useBond)?getBond(dataObj, iter2.node):(int*)0;
			  if (bond2)
			    (*bond2) += 1;
			}

		      currentPairsExtension->drawPairs
			(iter1.element, iter2.element, data, view,
			 xyz1[0], xyz1[1], xyz1[2],
			 xyz2[0], xyz2[1], xyz2[2],
			 d2, alpha);
		      if (data->printLength)
			{
			  glRasterPos3f((xyz1[0] + xyz2[0]) / 2.,
					(xyz1[1] + xyz2[1]) / 2.,
					(xyz1[2] + xyz2[2]) / 2.);
			  sprintf(distStr, "%7.3f", sqrt(d2));
			  openGLText_drawChars(distStr);
			}
		    }
		}
	      if (currentPairsExtension->endDrawingPairs)
		currentPairsExtension->endDrawingPairs
		  (iter1.element, iter2.element, data);
	    }
	}
    }

  if (currentPairsExtension->stopOpenGl)
    {
      DBG_fprintf(stderr, "Visu Pairs : call to stop OpenGl (%d).\n",
		  GPOINTER_TO_INT(currentPairsExtension->stopOpenGl));
      currentPairsExtension->stopOpenGl();
    }
      
  glPopMatrix();
  glEndList();

  /* Emit the valueChanged signal of dataNode. */
  nodeDataEmit_valueChanged(dataNode, dataObj);
  
  return TRUE;
}

void rebuildPairs(VisuData *dataObj)
{
  if (dataObj && extensionPairs->used)
    {
      /* Force to recreate pairs. */
      rebuildPairsNeeded = TRUE;
      visuPairBuild(dataObj);
    }
}

static int* getBond(VisuData *dataObj, VisuNode *node)
{
  int *bond;
  GValue val = {0, {{0}, {0}}};

  g_value_init(&val, G_TYPE_POINTER);
  visuNodeGet_propertyValue(visuDataGet_nodeArray(dataObj), node,
			    BONDNUMBER_ID, &val);
  bond = (int*)g_value_get_pointer(&val);

  if (!bond)
    {
      bond = (int*)newOrCopyData((gconstpointer)0, (gpointer)0);
      g_value_set_pointer(&val, (gpointer)bond);
      visuNodeSet_propertyValue(visuDataGet_nodeArray(dataObj), node,
				BONDNUMBER_ID, &val);
    }
  return bond;
}

static void freeData(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(int), obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data _U_)
{
  int* data;

#if GLIB_MINOR_VERSION > 9
  data = g_slice_alloc(sizeof(int));
#else
  data = g_malloc(sizeof(int));
#endif
  if (orig)
    *data = *(int*)orig;
  else
    *data = 0;
    
  return (gpointer)data;
}

static void onNewData(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "ElementRenderedChanged",
		   G_CALLBACK(createPairsOnElementRenderedChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodeRenderedChanged",
		   G_CALLBACK(createPairsOnNodeRenderedChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePositionChanged",
		   G_CALLBACK(createPairsOnNodePositionChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
                   G_CALLBACK(createPairsOnNodePopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationIncrease",
                   G_CALLBACK(createPairsOnNodePopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLFacetteChanged",
		   G_CALLBACK(facetteChanged), (gpointer)0);
}

static void onDataReady(GObject* obj _U_, VisuData *dataObj, gpointer data _U_)
{
  if (dataObj)
    {
      /* Create the Node property to count the number of bonds. */
      DBG_fprintf(stderr, "Visu Pairs: create the bond node property.\n");
      visuNodeNew_pointerProperty(visuDataGet_nodeArray(dataObj), BONDNUMBER_ID,
				  freeData, newOrCopyData, (gpointer)0);
      nodeDataSet_used(dataNode, dataObj, (extensionPairs->used)?1:0);
    }
    
  rebuildPairsNeeded = TRUE;

  if (extensionPairs->used)
    visuPairBuild(dataObj);
}

void facetteChanged(VisuData* obj, gpointer data _U_)
{
  if (extensionPairs->used && currentPairsExtension->sensitiveToFacette)
    visuPairBuild(obj);
}




/*****************************************/
/* Methods to organize pairs extensions. */
/*****************************************/

PairsExtension* visuPairExtensionNew(const char* name, const char* printName,
				     const char* description, gboolean sensitive,
				     initEndOpenGlPairsFunc init,
				     initEndOpenGlPairsFunc stop,
				     startEndPairsFunc start,
				     startEndPairsFunc end,
				     pairDefinitionFunc draw)
{
  PairsExtension *extension;

  extension = g_malloc(sizeof(PairsExtension));
  extension->name        = g_strdup(name);
  extension->printName   = g_strdup(printName);
  extension->description = g_strdup(description);
  
  extension->sensitiveToFacette = sensitive;

  extension->initOpenGl        = init;
  extension->stopOpenGl        = stop;
  extension->beginDrawingPairs = start;
  extension->endDrawingPairs   = end;
  extension->drawPairs         = draw;

  return extension;
}
void visuPairExtensionFree(PairsExtension* extension)
{
  if (!extension)
    return;
  if (extension->name)
    g_free(extension->name);
  if (extension->printName)
    g_free(extension->printName);
  if (extension->description)
    g_free(extension->description);
  g_free(extension);
}
void visuPairExtensionAdd(PairsExtension *extension)
{
  DBG_fprintf(stderr, "Visu Pairs: registering a new pairs extension ... ");
  g_return_if_fail(extension && extension->drawPairs && extension->name);

  availablePairsExtensions = g_list_append(availablePairsExtensions, (gpointer)extension);
  DBG_fprintf(stderr, "'%s' (%d).\n", extension->name, GPOINTER_TO_INT(extension));
}
PairsExtension* visuPairExtensionGet_current()
{
  return currentPairsExtension;
}
GList* visuPairExtensionGet_allMethods()
{
  return availablePairsExtensions;
}




static void foreachLevel2(gpointer key _U_, gpointer value, gpointer userData)
{
  VisuPair *pair;
  GList *tmpLst;
  struct foreachPairsData_struct *storage;

  pair = (VisuPair*)value;
  storage = (struct foreachPairsData_struct *)userData;
  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    storage->func(pair->ele1, pair->ele2,
		  (VisuPairData*)tmpLst->data, storage->userData);
}
void visuPairForeach(foreachPairsFunc whatToDo, gpointer userData)
{
  struct foreachPairsData_struct storage;

  storage.func = whatToDo;
  storage.userData = userData;
  g_hash_table_foreach(DminDmax, (GHFunc)foreachLevel2, (gpointer)(&storage));
}


/*****************************************/
/* Dealing with parameters and resources */
/*****************************************/

gboolean visuPairRead_linkFromTokens(gchar **tokens, int *index, VisuPairData **data,
				     int lineId, GError **error)
{
  VisuElement *ele[2];
  float minMax[2];
  gboolean res;

  g_return_val_if_fail(error && (*error == (GError*)0), FALSE);
  g_return_val_if_fail(data, FALSE);
  g_return_val_if_fail(tokens && index, FALSE);
  
  /* Read the two elements. */
  res = configFileRead_elementFromTokens(tokens, index, ele, 2, lineId, error);
  if (!res)
    return FALSE;
  /* Read the two distances. */
  res = configFileRead_floatFromTokens(tokens, index, minMax, 2, lineId, error);
  if (!res)
    return FALSE;
  
  /* Check the values. */
  res = FALSE;
  res = res || configFileClamp_float(&minMax[0], minMax[0], 0., -1.);
  res = res || configFileClamp_float(&minMax[1], minMax[1], 0., -1.);
  if (res)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d, given distance"
			     " are out of bounds (should be positive).\n"), lineId);
      return FALSE;
    }

  /* Set the values. */
  *data = visuPairGet_link(ele[0], ele[1], minMax);
  g_return_val_if_fail(*data, FALSE);

  return TRUE;
}

static gboolean readPairLink(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  gchar **tokens;
  int id;
  gboolean flags[2], res;
  VisuPairData *data;
  float rgb[4];
  Color *color;

  g_return_val_if_fail(nbLines == 2, FALSE);

  tokens = g_strsplit_set(lines[0], " \n", MAX_LINE_LENGTH);
  id = 0;
  if (!visuPairRead_linkFromTokens(tokens, &id, &data, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  /* Read second line with basic data. */
  tokens = g_strsplit_set(lines[1], " \n", MAX_LINE_LENGTH);
  id = 0;
  /* Read the color. */
  if (!configFileRead_floatFromTokens(tokens, &id, rgb, 3, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  /* Read the flags drawn and printLength. */
  if (!configFileRead_booleanFromTokens(tokens, &id, flags, 2, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  /* Check the values. */
  res = configFileClamp_float(&rgb[0], rgb[0], 0., 1.) ||
    configFileClamp_float(&rgb[1], rgb[1], 0., 1.) ||
    configFileClamp_float(&rgb[2], rgb[2], 0., 1.);
  if (res)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d, 3 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCES_PAIRS_DATA);
      return FALSE;
    }


  /* Set the values. */
  rgb[3] = 1.f;
  color = colorGet_byValues((int*)0, rgb[0], rgb[1], rgb[2], rgb[3]);
  if (!color)
    color = colorAdd_floatRGBA(rgb, (int*)0);
  visuPairSet_color(data, color);
  visuPairSet_drawn(data, flags[0]);
  visuPairSet_printLength(data, flags[1]);

  return TRUE;
}

/* This function details how to read the parameter FLAG_PARAMETER_PAIRS. */
static gboolean readPairsAreOn(gchar **lines, int nbLines, int position,
			       VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  visuPairSet_status(val);

  return TRUE;
}
static gboolean readFavPairsMethod(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  GList *pairsMethods;

  g_return_val_if_fail(nbLines == 1, FALSE);

  lines[0] = g_strstrip(lines[0]);

  if (!lines[0][0])
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d, 1 string value must appear"
			     " after the %s markup.\n"), position,
			   FLAG_RESOURCES_FAV_PAIRS);
      return FALSE;
    }
  else
    {
      pairsMethods = availablePairsExtensions;
      while (pairsMethods && strcmp(((PairsExtension*)pairsMethods->data)->name, lines[0]))
	pairsMethods = g_list_next(pairsMethods);
      if (!pairsMethods)
	{
	  *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			       _("Parse error at line %d, the method"
				 " '%s' is unknown.\n"), position, lines[0]);
	  return FALSE;
	}
      visuPairExtensionSet((PairsExtension*)pairsMethods->data);
    }

  return TRUE;
}
/* These functions save the resources. */
static void exportLevel1(gpointer key, gpointer value, gpointer user_data)
{
  VisuPair *pair;
  VisuPairData *data;
  struct foreachFuncExport_struct *str;
  GList *tmpLst;
  Color *color;
  
  pair = (VisuPair*)value;
  str = (struct foreachFuncExport_struct*)user_data;
  
  /* We export only if the two elements of the pair are used in the given dataObj. */
  if (str->dataObj)
    {
      /* We test the first element. */
      if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt,
			       (gpointer)pair->ele1))
	return;
      /* We test the second element. */
      if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt,
			       (gpointer)pair->ele2))
	return;
    }

  for (tmpLst = pair->pairs; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      data = (VisuPairData*)tmpLst->data;
      color = g_hash_table_lookup(data->properties, (gpointer)"color");
      g_return_if_fail(color);
      g_string_append_printf(str->data, "%s:\n", FLAG_RESOURCES_PAIR_LINK);
      g_string_append_printf(str->data, "    %s %4.3f %4.3f\n"
			     "    %4.3f %4.3f %4.3f  %d  %d\n", (gchar*)key,
			     data->minMax[PAIRS_MIN], data->minMax[PAIRS_MAX],
			     color->rgba[PAIRS_COLOR_R], color->rgba[PAIRS_COLOR_G],
			     color->rgba[PAIRS_COLOR_B], data->drawn,
			     data->printLength);

      *str->nbLinesWritten += 3;
    }  
}
static gboolean exportResourcesPairs(GString *data, int *nbLinesWritten,
				     VisuData *dataObj)
{
  struct foreachFuncExport_struct str;
  GList *pairsMeth;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_PAIRS);
  g_string_append_printf(data, "%s:\n    %i\n", FLAG_RESOURCES_PAIRS,
			 (int)extensionPairs->used);
  *nbLinesWritten += 3;

  if (currentPairsExtension)
    {
      g_string_append_printf(data, "# %s (", DESC_RESOURCES_FAV_PAIRS);
      pairsMeth = availablePairsExtensions;
      while (pairsMeth)
	{
	  g_string_append_printf(data, "'%s'", ((PairsExtension*)pairsMeth->data)->name);
	  if (pairsMeth->next)
	    g_string_append_printf(data, ", ");
	  pairsMeth = g_list_next(pairsMeth);
	}
      g_string_append_printf(data, ")\n");
      g_string_append_printf(data, "%s:\n    %s\n", FLAG_RESOURCES_FAV_PAIRS,
			     currentPairsExtension->name);
      *nbLinesWritten += 3;
    }

  g_string_append_printf(data, "# %s\n", DESC1_RESOURCES_PAIR_LINK);
  g_string_append_printf(data, "# %s\n", DESC2_RESOURCES_PAIR_LINK);
  *nbLinesWritten = 2;
  str.data = data;
  str.nbLinesWritten = nbLinesWritten;
  str.dataObj = dataObj;
  g_hash_table_foreach(DminDmax, (GHFunc)exportLevel1, (gpointer)&str);

  g_string_append_printf(data, "\n");
  *nbLinesWritten += 1;

  return TRUE;
}






/* OBSOLETE function, keep for backward compatibility. */
static gboolean readPairsData(gchar **lines, int nbLines, int position,
			      VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele[2];
  int res, token;
  float rgb[4];
  float minMax[2];
  VisuPairData *data;
  gchar **tokens;
  Color *color;

  /* Tokenize the line of values. */
  tokens = g_strsplit_set(g_strchug(lines[0]), " \n", MAX_LINE_LENGTH);
  token = 0;

  /* Get the two elements. */
  if (!configFileRead_elementFromTokens(tokens, &token, ele, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }

  /* Read 5 floats. */
  if (!configFileRead_floatFromTokens(tokens, &token, minMax, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  if (!configFileRead_floatFromTokens(tokens, &token, rgb, 3, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  res = configFileClamp_float(&rgb[0], rgb[0], 0., 1.) ||
    configFileClamp_float(&rgb[1], rgb[1], 0., 1.) ||
    configFileClamp_float(&rgb[2], rgb[2], 0., 1.) ||
    configFileClamp_float(&minMax[0], minMax[0], 0., -1.) ||
    configFileClamp_float(&minMax[1], minMax[1], 0., -1.);
  if (res)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d, 5 floating points"
			     " must appear after the %s markup.\n"),
			   position, FLAG_RESOURCES_PAIRS_DATA);
      return FALSE;
    }
  data = visuPairGet_link(ele[0], ele[1], minMax);
  g_return_val_if_fail(data, FALSE);

  rgb[3] = 1.f;
  color = colorGet_byValues((int*)0, rgb[0], rgb[1], rgb[2], rgb[3]);
  if (!color)
    color = colorAdd_floatRGBA(rgb, (int*)0);
  visuPairSet_color(data, color);
  
  return TRUE;
}
