/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_DATA_H
#define VISU_DATA_H

#include <glib.h>
#include <glib-object.h>

#include "visu_elements.h"
#include "visu_nodes.h"
#include "renderingBackend/visu_windowInterface_def.h"
#include "openGLFunctions/view.h"
#include "coreTools/toolFileFormat.h"

G_BEGIN_DECLS

/**
 * VISU_DATA_TYPE:
 *
 * return the type of #VisuData.
 */
#define VISU_DATA_TYPE	     (visu_data_get_type ())
/**
 * VISU_DATA:
 * @obj: a #GObject to cast.
 *
 * Cast the given @obj into #VisuData type.
 */
#define VISU_DATA(obj)	     (G_TYPE_CHECK_INSTANCE_CAST(obj, VISU_DATA_TYPE, VisuData))
/**
 * VISU_DATA_CLASS:
 * @klass: a #GClassObject to cast.
 *
 * Cast the given @klass into #VisuDataClass.
 */
#define VISU_DATA_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST(klass, VISU_DATA_TYPE, VisuDataClass))
/**
 * IS_VISU_DATA_TYPE:
 * @obj: a #GObject to test.
 *
 * Test if the given @ogj is of the type of #VisuData object.
 */
#define IS_VISU_DATA_TYPE(obj)    (G_TYPE_CHECK_INSTANCE_TYPE(obj, VISU_DATA_TYPE))
/**
 * IS_VISU_DATA_CLASS:
 * @klass: a #GClassObject to test.
 *
 * Test if the given @klass is of the type of #VisuDataClass class.
 */
#define IS_VISU_DATA_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE(klass, VISU_DATA_TYPE))
/**
 * VISU_DATA_GET_CLASS:
 * @obj: a #GObject to get the class of.
 *
 * It returns the class of the given @obj.
 */
#define VISU_DATA_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS(obj, VISU_DATA_TYPE, VisuDataClass))

/**
 * VisuData:
 *
 * A short way to identify #_VisuData structure.
 */
typedef struct _VisuData VisuData;
/**
 * VisuData_private:
 *
 * Private fields for #ViuData objects.
 */
typedef struct VisuData_private_struct VisuData_private;
/**
 * VisuDataClass:
 *
 * A short way to identify #_VisuDataClass structure.
 */
typedef struct _VisuDataClass VisuDataClass;

/**
 * setColorFunc:
 * @visuData: a pointer to the calling object ;
 * @ele: a #VisuElement ;
 * @node: a #VisuNode ;
 * @rgba: an 4 allocated float area to store the return values.
 *
 * This prototype is used to specify an optional method
 * to associate a color with external values to each node.
 */
typedef void (*setColorFunc)(VisuData *visuData, float rgba[4], VisuElement *ele, VisuNode* node);
/**
 * VisuDataScalingFunc:
 * @visuData: a pointer to the calling object ;
 * @ele: a #VisuElement ;
 * @node: a #VisuNode ;
 *
 */
typedef float (*VisuDataScalingFunc)(VisuData *visuData, VisuNode* node);

/**
 * VisuData:
 * @parent: an object to inherit from (NULL here).
 * @privateDt: a pointer to the private data.
 * @ntype: number of #VisuElement used in this object.
 * @fromVisuElementToInt: give the number of the #VisuElement when
 *                        its pointer is known.
 * @fromIntToVisuElement: give the pointer of a #ViosuElement when
 *                        its number is known.
 * @setColor: method called before drawing each node (can be NULL).
 *
 * This structure describes a #VisuData object.
 */
struct _VisuData
{
  GObject parent;

  /***********************/
  /* Elements attributes */
  /***********************/
  /* Number of VisuElements used by these data. */
  unsigned int ntype;
  /* This hashtable gives access to the good array  of VisuNodes
     when one has the VisuElement. */
  GHashTable *fromVisuElementToInt;
  /* This is a corresponding table from the nodes array indice
     to the associated VisuElement. */
  VisuElement **fromIntToVisuElement;

  /* Define a method to set the color of each node.
     If this method is NULL, the color of the element is used. */
  setColorFunc setColor;

  VisuData_private *privateDt;
};

/**
 * visu_data_get_type:
 *
 * This method returns the type of #VisuData, use VISU_DATA_TYPE instead.
 *
 * Returns: the type of #VisuData.
 */
GType visu_data_get_type(void);

VisuData* visuDataNew();
VisuData* visuDataNew_withSize(guint w, guint h);
/**
 * visuDataNew_withOpenGLView:
 * @view: an #OpenGLView object.
 *
 * This creates an empty #VisuData object with values for its veiw taken
 * from the given @view argument (the argument is indeed copied).
 *
 * Returns: a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visuDataNew_withOpenGLView(OpenGLView *view);






/**
 * visuDataGet_objectList:
 * @data: a valid #VisuObject ;
 *
 * All the drawned nodes are stored in an OpenGL list. The identifier of
 * this list can be access with the present method.
 *
 * Returns: an integer that identifies the OpenGL list of all drawn nodes.
 */
int visuDataGet_objectList(VisuData *data);

/**
 * EXT_VISU_DATA_ID:
 *
 * The id used to identify this extension, see
 * OpenGLExtensionRebuild_list() for instance.
 */
#define EXT_VISU_DATA_ID "AllElements"




enum visuData_boxVector
  {
    VISU_DATA_BOX_DXX,
    VISU_DATA_BOX_DYX,
    VISU_DATA_BOX_DYY,
    VISU_DATA_BOX_DZX,
    VISU_DATA_BOX_DZY,
    VISU_DATA_BOX_DZZ
  };

/**
 * visuDataSet_population:
 * @data: a #VisuData object ;
 * @nbOfTypes: number of #VisuElement ;
 * @nbOfNodesPerVisuElement: number of #VisuNode per VisuElement ;
 * @visuElementUsed: pointers to #VisuElement in the same order that
 *                   @nbOfNodesPerVisuElement.
 *
 * This method allocates the storing part of the given #VisuData structure and
 * store all the #VisuNodes.
 *
 * Returns: 1 if everything goes right.
 */
int visuDataSet_population(VisuData *data, unsigned int nbOfTypes,
			   unsigned int* nbOfNodesPerVisuElement,
			   VisuElement **visuElementUsed);
/**
 * visuDataFree_population:
 * @data: a VisuData to be freed.
 *
 * This method frees only the allocated memory that deals with
 * the nodes (i.e. everything except the data of the files,
 * the properties and the setColor method.
 */
void visuDataFree_population(VisuData *data);
/**
 * visuDataAdd_file:
 * @data: a #VisuData object ;
 * @file: a string that points to a file ;
 * @kind: an integer to qualify the file to add ;
 * @format: a file format.
 *
 * This method is used to add files
 * of type @kind to the @data. The @file
 * attribute is copied. The @format argument can be null.
 */
void visuDataAdd_file(VisuData *data, gchar* file, int kind, FileFormat *format);
/**
 * visuDataRemove_allFiles:
 * @data: a #VisuData object.
 *
 * This method is used to empty the list of
 * known file from the given @data.
 */
void visuDataRemove_allFiles(VisuData *data);
/**
 * visuDataGet_file:
 * @data: a #VisuData object.
 * @kind: an integer to qualify the required file ;
 * @format: a location for a file format (can be NULL).
 *
 * This prototype is used to retrieve stored
 * files identify by their @kind.
 *
 * Returns: the name of a file (it should not be deleted).
 */
gchar* visuDataGet_file(VisuData *data, int kind, FileFormat **format);
/**
 * visuDataSet_fileCommentary:
 * @data: a #VisuData object ;
 * @commentary: the message to be stored (null terminated) ;
 * @iSet: an integer.
 *
 * This method is used to store a description of the given @data. This
 * string is copied and @commentary can be freed. Before using this
 * method, the number of possible node sets must have been defined
 * using visuDataSet_nSet(), if not, only iSet == 0 is allowed.
 */
void visuDataSet_fileCommentary(VisuData *data, gchar* commentary, gint iSet);
/**
 * visuDataGet_fileCommentary:
 * @data: a #VisuData object ;
 * @iSet: an integer (>= 0).
 *
 * Get the commentary associated to the given @data, for the given
 * node set.
 *
 * Returns: a string description (possibly empty). This string is own by V_Sim
 *          and should not be freed.
 */
gchar* visuDataGet_fileCommentary(VisuData *data, gint iSet);
/**
 * visuDataSet_nSet:
 * @data: a #VisuData object ;
 * @nSet: an integer.
 *
 * Change the number of available sets of nodes for this
 * #VisuData. This has a side effect to delete all previously saved
 * file commentaries (see visuDataSet_fileCommentary()).
 */
void visuDataSet_nSet(VisuData *data, int nSet);
/**
 * visuDataGet_nSet:
 * @data: a #VisuData object.
 *
 * Retrieve the number of available sets of nodes for this #VisuData,
 * see visuDataSet_nSet().
 *
 * Returns: the number of set of nodes (1 is default).
 */
int visuDataGet_nSet(VisuData *data);
/**
 * visuDataSet_setId:
 * @data: a #VisuData object ;
 * @iSet: an integer.
 *
 * Change the current id of the set of data (ordered as in C,
 * beginning at 0).
 */
void visuDataSet_setId(VisuData *data, int iSet);
/**
 * visuDataGet_setId:
 * @data: a #VisuData object.
 *
 * Retrieve the id of the current set of data (ordered as in C,
 * beginning at 0).
 *
 * Returns: the id of the set of nodes currently loaded, -1 if none.
 */
int visuDataGet_setId(VisuData *data);

/**
 * visuData_compareElements:
 * @data1: a #VisuData object ;
 * @data2: an other #VisuData object.
 *
 * This method is used to compare the composition of the given two #VisuData objects.
 * The test is only done on #VisuElement lists.
 *
 * Returns: TRUE if the two objects contains exactly the same #VisuElemet objects (not
 *          one more or one less or one different), FALSE otherwise.
 */
gboolean visuData_compareElements(VisuData *data1, VisuData *data2);
/**
 * visuDataSet_changeElementFlag:
 * @data: a #VisuData object ;
 * @changeElement: a boolean.
 *
 * This method is mainly used by internal gears to set a flag. This flag control
 * if the @data object has the same #VisuElement objects than the previously rendered one.
 */
void visuDataSet_changeElementFlag(VisuData *data, gboolean changeElement);
/**
 * visuDataGet_changeElementFlag:
 * @data: a #VisuData object.
 *
 * V_Sim can use a flag set on @data object to know if @data has exactly the same
 * #VisuElement list than the previously rendered one.
 *
 * Returns: TRUE if the previously rendered #VisuData object has had the same
 *          #VisuElement list than the given one, FALSE otherwise.
 */
gboolean visuDataGet_changeElementFlag(VisuData *data);

float visuDataGet_allElementExtens(VisuData *dataObj);

/**
 * visuDataAdd_nodeFromElement:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @ele: the #VisuElement kind of the new #VisuNode ;
 * @xyz: its coordinates ;
 * @emitSignal: a boolean.
 *
 * This method adds a new #VisuNode to the specified #VisuData. If
 * @emitSignal is TRUE, then NodePopulationIncrease signal is
 * triggered.
 *
 * Returns: a pointer to the newly created node.
 */
VisuNode* visuDataAdd_nodeFromElement(VisuData *data, VisuElement *ele,
				      float xyz[3], gboolean emitSignal);
/**
 * visuDataAdd_nodeFromIndex:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @position: a integer corresponding to the position of
 *            a #VisuElement in the array **nodes in the structure;
 * @xyz: its coordinates ;
 * @emitSignal: a boolean.
 *
 * This method adds a new #VisuNode to the specified #VisuData. Position must be
 * chosen between 0 and (ntype - 1) and corresponds to the position of the array
 * of #VisuNodes of a #VisuElement. If @emitSignal is TRUE, then
 * NodePopulationIncrease signal is triggered.
 *
 * Returns: a pointer to the newly created node.
 */
VisuNode* visuDataAdd_nodeFromIndex(VisuData *data, unsigned int position,
				    float xyz[3], gboolean emitSignal);
/**
 * visuDataGet_nodeFromNumber:
 * @data: a %VisuData structure which stores the nodes.
 * @number: an integer.
 *
 * This methods retrieves the #VisuNode identified by the integer @number.
 * The number must be strictly positive. No eror is raised if no node corresponds
 * to the given number.
 *
 * Returns: the found #VisuNode or NULL if none corresponds to number.
 */
VisuNode* visuDataGet_nodeFromNumber(VisuData *data, unsigned int number);

/**
 * visuDataGet_nodeArray:
 * @data: a #VisuData object.
 *
 * This method retrieve the #VisuNodeArray associated to the given @data.
 *
 * Returns: the associated #VisuNodeArray.
 */
VisuNodeArray* visuDataGet_nodeArray(VisuData *data);


/**
 * visuDataSet_ColorFunc:
 * @data: a #VisuData object ;
 * @func: a method that colorize the nodes.
 *
 * This is a little trick to colorized the nodes. It should not be used since it
 * will probably be different in future release.
 */
void visuDataSet_ColorFunc(VisuData *data, setColorFunc func);

/**
 * visuDataGet_XYZtranslation:
 * @data: a #VisuData object.
 *
 * The nodes are rendered at thier coordinates plus a translation. This method
 * allows to retrieve that translation.
 *
 * Returns: a newly allocated array of 3 floats. It should be freed with a call
 *          to free() after use.
 */
float* visuDataGet_XYZtranslation(VisuData* data);
/**
 * visuDataSet_XYZtranslation:
 * @data: a #VisuData object ;
 * @xyz: an array of floating point values.
 *
 * This set the translations of the specified #VisuData whatever previous values.
 * The translation is done in the orthonormal referential, not the referential of
 * the box.
 *
 * Returns: if returns 1, visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
int visuDataSet_XYZtranslation(VisuData* data, float xyz[3]);
/**
 * visuDataConvert_XYZtoBoxCoordinates:
 * @data: a #VisuData object ;
 * @boxCoord: an array of floating point values to store the result ;
 * @xyz: an array of floating point values describing coordinates in cartesian.
 *
 * Use this method to transform cartesian coordinates to the box coordinates.
 */
void visuDataConvert_XYZtoBoxCoordinates(VisuData *data, float boxCoord[3], float xyz[3]);
/**
 * visuDataConvert_boxCoordinatestoXYZ:
 * @data: a #VisuData object ;
 * @xyz: an array of floating point values to store the result ;
 * @boxCoord: an array of floating point values that describes the box coordinates.
 *
 * Use this method to transform box coordinates into cartesian.
 */
void visuDataConvert_boxCoordinatestoXYZ(VisuData *data, float xyz[3], float boxCoord[3]);
/**
 * visuDataGet_boxMatrix:
 * @data: a #VisuData object ;
 * @matrix: an area to store the matrix.
 *
 * This method is used when the box matrix is required. This matrix can transform
 * a vector given in box coordinates into a cartesian vector. If a simple vector
 * multication is required, then the use of visuDataConvert_boxCoordinatestoXYZ()
 * should be prefered.
 */
void visuDataGet_boxMatrix(VisuData *data, float matrix[3][3]);
void visuDataGet_boxMatrixD0(VisuData *data, double matrix[3][3]);
/**
 * visuDataSet_boxGeometry:
 * @data: a #VisuData object ;
 * @geometry: a 6 floating point array ;
 * @periodic: a boolean ;
 * @refLength: a float value.
 *
 * This methods set the size of the box that defines the viewport. If the @periodic
 * argument is TRUE, the box is also the limit of a periodic
 * system. The @refLength argument is used to change the reference
 * length that gives the size for zoom 1. If @refLength is negative,
 * then the reference length is left unchanged, when @refLength is
 * null, then the length is computed from the box size and for
 * @refLength positive, the value is applied.
 */
void visuDataSet_boxGeometry(VisuData *data, double geometry[6],
			     OpenGLBoxBoundaries bc);
void visuDataApply_boxGeometry(VisuData *data, float refLength);
/**
 * visuDataGet_boxGeometry:
 * @data: a #VisuData object ;
 * @vector: an int corresponding to a vector of the box.
 *
 * Retrieve the value of a vector defining the bounding box. The vector
 * is chosen with an int, see the #visuData_boxVector enum for more
 * details.
 *
 * Returns: the value of the required vector (always a positive value for vector = 0, 2 or 5 !).
 */
float visuDataGet_boxGeometry(VisuData *data, int vector);
/**
 * visuDataGet_NodeBoxFromNumber:
 * @data: a #VisuData object.
 * @nodeId: the index of the node considered.
 * @nodeBox: the array to store the box of the node.
 *
 * This method retrieves the value of the box associated to a node (with respect to the unit cell).
 * 
 * Returns: TRUE if everything went well, FALSE otherwise. The box is stored in the nodeBox array.
 */
gboolean visuDataGet_nodeBoxFromNumber(VisuData *data, guint nodeId, int nodeBox[3]);
/**
 * visuDataGet_NodeBoxFromCoord:
 * @data: a #VisuData object.
 * @xcart: the coordinates of a node.
 * @nodeBox: the array to store the box of the node.
 *
 * This method retrieves the value of the box associated to the coordinates of the node (with respect to the unit cell).
 * 
 * Returns: TRUE if everything went well, FALSE otherwise. The box is stored in the nodeBox array.
 */
gboolean visuDataGet_nodeBoxFromCoord(VisuData *data, float xcart[3], int nodeBox[3]);
/**
 * visuDataGet_boxVertices:
 * @data: a #VisuData object.
 * @vertices: the position of the eight vertices of the bounding box.
 * @withExtension: a boolean.
 *
 * All nodes are rendered inside a bounding box, this method can be used to retrieve
 * it. This box is not the drawn box but the box containing all the
 * nodes, included possible extension. To get the box itself, use
 * visuDataGet_boxMatrix() instead. One can also get the vertices of
 * the box itself using FALSE as @withExtension argument.
 */
void visuDataGet_boxVertices(VisuData *data, float vertices[8][3],
			     gboolean withExtension);
VisuUnits visuDataGet_unit(VisuData *data);
gboolean visuDataSet_unit(VisuData *data, VisuUnits unit);
OpenGLBoxBoundaries visuDataGet_boundaryConditions(VisuData *data);
void visuDataSet_tightBox(VisuData *data);
/**
 * visuData_constrainedElementInTheBox:
 * @data: a #VisuData object ;
 * @element: a #VisuElement object.
 *
 * Check all the nodes of the specified @element and change their coordinates if they are out
 * of the bounding box. The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: TRUE if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visuData_constrainedElementInTheBox(VisuData *data, VisuElement *element);
/**
 * visuData_constrainedInTheBox:
 * @data: a #VisuData object.
 *
 * It does the same things that visuData_constrainedElementInTheBox() but for all
 * the #VisuElement of the given @data. I.e. it checks all the nodes and changes
 * their coordinates if they are out of the bounding box.
 * The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: TRUE if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visuData_constrainedInTheBox(VisuData *data);
/**
 * visuData_constrainedFree:
 * @data: a #VisuData object.
 *
 * Return all the nodes to their original position, except for the
 * global translation.
 *
 * Returns: TRUE if visuData_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visuData_constrainedFree(VisuData *data);
/**
 * visuDataGet_translationStatus:
 * @data: a #VisuData object.
 *
 * When a translation is applied (even with a [0,0,0] vector), the
 * nodes are shifted to be in the box. This routine returns the
 * translation status of all nodes. If one of them is translated, then
 * return value is TRUE.
 *
 * Returns: if one of the nodes is shifted.
 */
gboolean visuDataGet_translationStatus(VisuData *data);
/**
 * visuData_replicate:
 * @data: a #VisuData object ;
 * @extension: three floating point values ;
 * @rebuild: a location to store a flag.
 *
 * This routine will create (or remove) nodes to expand the initial box to
 * the required size. An extension of 0 means no extension, i.e. the initial
 * box. The extension is done symmetrically in each direction toward negative
 * and positive direction.
 *
 * If the @rebuild argument is TRUE, the nodes should be rebuilt (with
 * visuData_createAllNodes()).
 *
 * To remove added nodes, see visuData_restore().
 *
 * returns: TRUE if the redraw should be done.
 */
gboolean visuData_replicate(VisuData *data, float extension[3], gboolean *rebuild);
/**
 * visuData_restore:
 * @data: a #VisuData object.
 *
 * Remove all nodes that have been added by a visuData_replicate()
 * call.
 *
 * Returns: TRUE if some nodes has been indeed removed.
 */
gboolean visuData_restore(VisuData *data);
/**
 * visuDataGet_extension:
 * @dataObj: a #VisuData object ;
 * @extension: an allocated array to store the values.
 *
 * Using visuData_replicate(), it is possible to duplicate the primitive box
 * in each directions. Use this method to know the current extension. Returned
 * values are positive floating point values. An extension of 0. means that
 * only the primitive box exists, while a value of one means a duplication of
 * one box in each direction of the coordinate.
 */
void visuDataGet_extension(VisuData *dataObj, float extension[3]);
/**
 * visuDataGet_nodePosition:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @coord: an array of 3 floating point values to store the position.
 *
 * Position of nodes are subject to various translations and different transformations.
 * Their coordinates should not be access directly through node.[xyz]. This method
 * is used to retrieve the given node position.
 */
void visuDataGet_nodePosition(VisuData *data, VisuNode *node, float coord[3]);
void visuDataGet_reducedNodePosition(VisuData *data, VisuNode *node, float red[3]);
VisuNodeInfo* visuDataGet_distanceList(VisuData *data, guint nodeId, float *minVal);
gboolean visuDataSet_newBasis(VisuData *data, guint nO, guint nA, guint nB, guint nC);


/***************/
/* OpenGL Part */
/***************/
/**
 * visuData_createNodeInfos:
 * @data: a #VisuData object.
 *
 * This create is list with labels positions on nodes with some informations. These
 * informations are taken from the #DataNode stored in the property "dataNode". If
 * this property is 1, then the number of nodes are printed, if the value is 2,
 * the name of the element is used, instead the value is a pointer to a #DataNode
 * object and its method is used to get the label.
 */
void visuData_createNodeInfos(VisuData *data);
/**
 * visuData_createNodes:
 * @data: a #VisuData object ;
 * @ele: a #VisuElement object.
 *
 * This create the glObjectList registered at (%identifierAllNodes + the position
 * of the @ele in the @data object + 1) that contains all the nodes of the given #VisuElement.
 */
void visuData_createNodes(VisuData *data, VisuElement *ele);
/**
 * visuData_createNode:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object.
 *
 * This method call the createNode method of the current rendering method
 * for the given node. It also calls the list of the material associated
 * to the given node.
 */
void visuData_createNode(VisuData *data, VisuNode *node);
/**
 * visuData_createAllNodes:
 * @data: a #VisuData object.
 *
 * This create the glObjectList registered at %identifierAllNodes
 * that contains all the nodes. This glObjectList is made of all
 * nodes of all element that has a flag %rendered at true translated
 * to their own positions.
 *
 * This method doesn't ask for redraw and signal %OpenGLAskForReDraw
 * should be emitted manually.
 */
void visuData_createAllNodes(VisuData *data);
/**
 * visuData_createAllElements:
 * @data: a #VisuData object.
 *
 * This method will call the createOpenGLElementFunc() method of the current #RenderingMethod
 * on all the nodes of the given #VisuData.
 */
void visuData_createAllElements(VisuData *data);

/**
 * visuDataEmit_askForShowHideNodes:
 * @data: a valid #VisuData object ;
 * @redraw: a pointer to a location to store if redraw
 *          is needed after all listeners have modified the
 *          nodes of the given @data.
 *
 * This methods is used to emit the 'NodeAskForShowHide' signal. This
 * signal asks all modules that may hide nodes to recompute their
 * hiding scheme and put in @redraw if they have changed something. Since
 * all listeners write in @redraw, they should modify it with an union.
 * @redraw is initialised at FALSE before the signal is emitted.
 */
void visuDataEmit_askForShowHideNodes(VisuData *data, gboolean *redraw);
/**
 * visuDataEmit_nodePositionChanged:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodePositionChanged' signal. This signal asks
 * all modules that are dependent of the nodes' positions to recompute their
 * OpenGL lists.
 */
void visuDataEmit_nodePositionChanged(VisuData *data);
/**
 * visuDataEmit_elementRenderedChange:
 * @data: a valid #VisuData object ;
 * @element: a valid #VisuElement object.
 *
 * This method is used to emit the 'ElementRenderedChanged' signal. This signal asks
 * all modules that are dependent of the given @element visibility to recompute their
 * OpenGL lists.
 */
void visuDataEmit_elementRenderedChange(VisuData *data, VisuElement *element);
/**
 * visuDataEmit_nodeRenderedChange:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodeRenderedChanged' signal. This signal asks
 * all modules that are dependent of the nodes' visibility to recompute their
 * OpenGL lists. This signal is usualy emitted after a call to
 * visuDataEmit_askForShowHideNodes().
 */
void visuDataEmit_nodeRenderedChange(VisuData *data);
/**
 * visuDataEmit_observeMovement:
 * @data: a valid #VisuData object ;
 * @start: TRUE to signify a starting movement, FALSE for an ending one.
 *
 * This method is used to emit the 'OpenGLObserveMovement' signal. This signal
 * signify that an observe movement has been initiated or has just finished.
 */
void visuDataEmit_observeMovement(VisuData *data, gboolean start);

/**
 * visuDataAdd_timeout:
 * @data: a valid #VisuData object ;
 * @time: the period of call in milliseconds ;
 * @func: the callback function to be called ;
 * @user_data: a pointer to some user defined informations.
 *
 * This method is used to add the @func method to be called regularly at the period
 * @time. This methos calls in fact g_timeout_add() with the given arguments. But
 * the source id is stored internaly and the timeout function is removed automatically
 * when the object @data is destroyed. It is convienient to add a method working
 * on the #VisuData object that is called periodically during the life of te object.
 *
 * Returns: the source id if the calling method need to work with it. To remove
 *          the callback, don't use g_source_remove() but visuDataRemove_timeout()
 *          to inform the #VisuData object that this source has been removed and
 *          not to remove it when the object will be destroyed.
 */
guint visuDataAdd_timeout(VisuData *data, guint time, GSourceFunc func, gpointer user_data);
/**
 * visuDataRemove_timeout:
 * @data: a valid #VisuData object ;
 * @timeoutId: a source id.
 *
 * This method is used to remove a timeout that has been associated to the given
 * @data (see visuDataAdd_timeout()).
 *
 * Returns: TRUE if the source has been found and removed.
 */
gboolean visuDataRemove_timeout(VisuData *data, guint timeoutId);

/**
 * visuDataGet_renderingWindow:
 * @data: a #VisuData object.
 *
 * The #VisuData objects can be rendered into a window. They are attached
 * to one using visuRenderingWindowSet_visuData() method. The actual method is
 * then used to retrieve the window the argument @data is attached to.
 * 
 * Returns: the window the @data argument is attached to, or NULL
 *          if the object has not yet been attached or if the rendering
 *          is done off-screen.
 */
GenericRenderingWindow visuDataGet_renderingWindow(VisuData *data);
/**
 * visuDataSet_renderingWindow:
 * @data: a #VisuData object ;
 * @window: a pointer to a rendering window (can be NULL).
 *
 * This method is used to tell the @data object which
 * window is used to render it. Normally, one has not to use
 * this method, using visuRenderingWindowSet_visuData() calls it
 * already.
 */
void visuDataSet_renderingWindow(VisuData *data, GenericRenderingWindow window);

/**
 * visuDataGet_openGLView:
 * @data: a #VisuData object.
 *
 * Once the object @data has been initialised, an #OpenGLView object
 * is automattically attached and this method can be used to retrieve
 * it.
 *
 * Returns: the #OpenGLView attached to the given @data.
 */
OpenGLView* visuDataGet_openGLView(VisuData *data);
/**
 * visuDataSet_angleOfView:
 * @data: a #VisuData object ;
 * @valueTheta: a floatinf point value in degrees ;
 * @valuePhi: a floating point value in degrees ;
 * @valueOmega: a floating point value in degrees ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera orientation for the given @data.
 * If necessary, this method will emit the 'OpenGLThetaPhiOmega' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_angleOfView(VisuData *data, float valueTheta,
			    float valuePhi, float valueOmega, int mask);
/**
 * visuDataSet_positionOfView:
 * @data: a #VisuData object ;
 * @valueX: a floatinf point value in the bounding box scale
 *          (1 is the size of the bounding box) ;
 * @valueY: a floating point value in bounding box scale ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera position for the given @data.
 * If necessary, this method will emit the 'OpenGLXsYs' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_positionOfView(VisuData *data, float valueX, float valueY, int mask);
/**
 * visuDataSet_zoomOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value.
 *
 * This method is used to change the camera zoom for the given @data.
 * If necessary, this method will emit the 'OpenGLGross' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_zoomOfView(VisuData *data, float value);
/**
 * visuDataSet_perspectiveOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value (> 1.1).
 *
 * This method is used to change the camera perspective for the given @data.
 * If necessary, this method will emit the 'OpenGLPersp' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_perspectiveOfView(VisuData *data, float value);
/**
 * visuDataSet_sizeOfView:
 * @data: a valid #VisuData object ;
 * @width: the new horizontal size ;
 * @height: the new vertical size.
 *
 * It changes the size of the OpenGl area and reccompute the OpenGL viewport.
 * Warning : it doesn't change the size of the window.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visuDataSet_sizeOfView(VisuData *data, guint width, guint height);
/**
 * visuDataEmit_facettesChanged:
 * @data: a valid #VisuData object.
 *
 * Emit the 'OpenGLFacetteChanged' signal.
 */
void visuDataEmit_facettesChanged(VisuData *data);

void visuDataSet_nodeScalingFunc(VisuData *data, VisuDataScalingFunc scaling);
float visuDataGet_nodeScalingFactor(VisuData *data, VisuNode *node);



/**
 * visuDataGet_allObjects:
 *
 * This methods is used to retrieve all #VisuObject currently allocated
 * in V_Sim. It is usefull to apply some changes on all objects (resources
 * for example).
 *
 * Returns: a list of V_Sim own #VisuData objects.
 */
GList* visuDataGet_allObjects();

/**
 * visuDataRemove_nodes:
 * @data: a #VisuData object ;
 * @nodeNumbers: an allocated int array with number identifying nodes.
 *
 * Delete some nodes using this method. The nodes are identified by their
 * number and have not to be of the same element. This routine raises
 * the "NodePopulationDecrease" signal. The @nodeNumbers argument must
 * be terminated with a negative value.
 */
void visuDataRemove_nodes(VisuData *data, int *nodeNumbers);

/**
 * VisuDataIter_struct:
 * @data: a pointer the iterator is associated to ;
 * @nAllStoredNodes: the total number of stored nodes for the
 *                   associated #VisuData ;
 * @nElements: the number of #VisuElement for the associated #VisuData ;
 * @nStoredNodes: the number of stored nodes for each element ;
 * @iElement: the index corresponding to @element (or -1 if no set);
 * @node: a pointer on a current node ;
 * @element: a pointer on a current element.
 *
 * This structure is an iterator over the nodes of a #VisuData object.
 * Create it with visuDataIter_new(). Then the numbers are allocated and
 * correspond to the value of the #VisuData object. Use visuDataIter_start()
 * to initialise the iterator for a run over the nodes, visuDataIter_next()
 * to associate @node and @element to the next node, or NULL if there is no
 * more node to run over.
 */
typedef struct VisuDataIter_struct
{
  VisuData *data;

  guint idMax;
  guint nAllStoredNodes;
  guint nElements;
  guint *nStoredNodes;

  guint iElement;
  VisuNode *node;
  VisuElement *element;
} VisuDataIter;
/**
 * visuDataIter_new:
 * @data: a #VisuData object ;
 * @iter: an alocated iterator.
 *
 * Set values to a #VisuDataIter object to iterate over nodes.
 * Its contain is initialised with the data size (number of elements,
 * number of nodes per element...).
 */
void visuDataIter_new(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_start:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run over the nodes.
 */
void visuDataIter_start(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_startVisible:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run over the 
 * visible nodes (see visuDataIter_nextVisible).
 */
void visuDataIter_startVisible(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_startNumber:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run
 * following the node oder.
 */
void visuDataIter_startNumber(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_restartNode:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * The element internal pointer must be associated. Then, it returns the
 * node pointer to the first node for this element.
 */
void visuDataIter_restartNode(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_next:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node and element internal pointers to the next node, or NULL if
 * none remains.
 */
void visuDataIter_next(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_nextNode:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node internal pointer to the next node, or NULL if
 * none remains. Contrary to visuDataIter_next() it does not go to the
 * next element if one exists.
 */
void visuDataIter_nextNode(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_nextNodeNumber:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node internal pointer to the next node, increasing the id of
 * the current node. The element internal pointer is also updated
 * accordingly. If no more nodes exist after the given one, node and
 * element internal pointers are set to NULL.
 */
void visuDataIter_nextNodeNumber(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_nextElement:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify element internal pointer to the next element and set node
 * to the first one, or NULL if none remains.
 */
void visuDataIter_nextElement(VisuData *data, VisuDataIter *iter);
/**
 * visuDataIter_nextVisible:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Go to the next rendered node (changing element if required).
 */
void visuDataIter_nextVisible(VisuData *data, VisuDataIter *iter);

G_END_DECLS

#endif
