/*   EXTRAITS DE LA LICENCE
     Copyright CEA, contributeurs : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2005)
  
     Adresse ml :
     BILLARD, non joignable par ml ;
     CALISTE, damien P caliste AT cea P fr.

     Ce logiciel est un programme informatique servant  visualiser des
     structures atomiques dans un rendu pseudo-3D. 

     Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
     respectant les principes de diffusion des logiciels libres. Vous pouvez
     utiliser, modifier et/ou redistribuer ce programme sous les conditions
     de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
     sur le site "http://www.cecill.info".

     Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
     pris connaissance de la licence CeCILL, et que vous en avez accept les
     termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
     Copyright CEA, contributors : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2005)

     E-mail address:
     BILLARD, not reachable any more ;
     CALISTE, damien P caliste AT cea P fr.

     This software is a computer program whose purpose is to visualize atomic
     configurations in 3D.

     This software is governed by the CeCILL  license under French law and
     abiding by the rules of distribution of free software.  You can  use, 
     modify and/ or redistribute the software under the terms of the CeCILL
     license as circulated by CEA, CNRS and INRIA at the following URL
     "http://www.cecill.info". 

     The fact that you are presently reading this means that you have had
     knowledge of the CeCILL license and that you accept its terms. You can
     find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <glib.h>

#include <config.h>
#include <ab6_invars.h>
#include <visu_tools.h>
#include <visu_basic.h>
#include <gtk_interactive.h>
#include <openGLFunctions/interactive.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolElements.h>
#include <renderingMethods/renderingAtomic.h>
#include <renderingMethods/renderingSpin.h>

#ifdef HAVE_ABINIT_SYMMETRY
#include "ab_symmetry.h"
#endif

/* For compatibility with older version. */
#ifndef AB6_INVARS_STR
#define AB6_INVARS_STR(A) #A
#endif

#ifndef ab6_error_string_from_id
#define ab6_error_string_from_id(A) #A
#endif

#define ABINIT_DESCRIPTION _("<span size=\"smaller\">"			\
			     "This plug-in introduces support for\n"	\
			     "crystallographic structures in\n"		\
			     "<b>ABINIT</b> input files.</span>")
#define ABINIT_AUTHORS     "Caliste Damien"

/* Local methods */
static RenderingFormatLoad* abStructuralInit();
static RenderingFormatLoad* abSpinInit();
static gboolean loadAbinitIn(VisuData *data, const gchar* filename,
			     FileFormat *format, int nSet, GError **error);
static gboolean loadAbinitSpin(VisuData *data, const gchar* filename,
			       FileFormat *format, int nSet, GError **error);
static gpointer parseAbinitThread(gpointer data);
static GError* loadAbinit(VisuData *data, Ab6Invars *dt, int nSet);
/* Local variables */
static gchar *iconPath;

/* Required methods for a loadable module. */
gboolean abinitInit()
{
  RenderingFormatLoad* meth, *spin;

  DBG_fprintf(stderr, "Abinit: loading plug-in 'abinit'...\n");

  DBG_fprintf(stderr, "Abinit: declare a new rendering load method.\n");
  meth = abStructuralInit();
  renderingAtomicAdd_loadMethod(meth);

  DBG_fprintf(stderr, "Abinit: declare a new spin load method.\n");
  spin = abSpinInit();
  rspin_addLoadMethod(spin);

  iconPath = g_build_filename(V_SIM_PIXMAPS_DIR, "abinit.png", NULL);

  return TRUE;
}

gboolean abinitInitGtk()
{
  DBG_fprintf(stderr, "Abinit: declare a new interactive tab.\n");
#ifdef HAVE_ABINIT_SYMMETRY
  gtkInteractiveAdd_action(buildTab, startSelect, stopSelect);
#endif

  return TRUE;
}

const char* abinitGet_description()
{
  return ABINIT_DESCRIPTION;
}

const char* abinitGet_authors()
{
  return ABINIT_AUTHORS;
}

const char* abinitGet_icon()
{
  return iconPath;
}

static RenderingFormatLoad* abStructuralInit()
{
  char *type[] = {"*.in", (char*)0};
  char *descr = _("ABINIT input file format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "ABINIT input file format";
  meth->fmt = fileFormatNew(descr, type);
  if (!meth->fmt)
    {
      g_error("Can't initialize the ABINIT loading method, aborting...\n");
    }
  meth->priority = 90;
  meth->load = loadAbinitIn;

  return meth;
}

static Ab6Invars *dt;

static gboolean loadAbinitIn(VisuData *data, const gchar* filename,
			     FileFormat *format _U_, int nSet, GError **error)
{
#ifdef G_THREADS_ENABLED
  GThread *ld_thread;
#endif

  g_return_val_if_fail(error && !*error, FALSE);

#ifdef G_THREADS_ENABLED
  DBG_fprintf(stderr, "AB structure: run ABINIT parsing into a thread.\n");
  ld_thread = g_thread_create(parseAbinitThread, (gpointer)filename,
			      TRUE, error);
  if (ld_thread)
    *error = (GError*)g_thread_join(ld_thread);
  else
    g_warning("Can't run thread for ABINIT parsing.");
#else
  DBG_fprintf(stderr, "AB structure: run ABINIT parsing directly.\n");
  *error = (GError*)parseAbinitThread((gpointer)filename);
#endif

  if (*error && (*error)->code == RENDERING_ERROR_FILE)
    return FALSE;
  else if (*error)
    return TRUE;
  else
    *error = loadAbinit(data, dt, nSet);
  ab6_invars_free(dt);

  return TRUE;
}

static GError* loadAbinit(VisuData *data, Ab6Invars *dt, int nSet)
{
  int ndtset, i, res;
  int ntypat, natom, nzero, *index;
  int *typat;
  double *znucl, rprimd[3][3], *coord, *spinat;
  VisuElement **types, **ntypes;
  float rcov, cart[3], red[3];
  double  box[6];
  char *ptChar, *name;
  unsigned int *nattyp;
  Ab6Error error;

  /* Ok, try to find the required keywords. */
  error = ab6_invars_get_ndtset(dt, &ndtset);
  if (error != AB6_NO_ERROR)
    return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
		       "Abinit loader report error %d while getting n datasets.",
		       (int)error);
  DBG_fprintf(stderr, "AB structure: found %d dtsets.\n", ndtset);
  /* Store the number of datasets. */
  visuDataSet_nSet(data, ndtset);

  g_return_val_if_fail(nSet >= 0 && nSet < ndtset,
		       g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
				   "Can't load dataset %d.", nSet));

  error = ab6_invars_get_integer(dt, AB6_INVARS_NTYPAT, nSet + 1, &ntypat);
  if (error != AB6_NO_ERROR)
    return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
		       "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
		       ab6_error_string_from_id(error),
		       AB6_INVARS_STR(AB6_INVARS_NTYPAT));
  error = ab6_invars_get_integer(dt, AB6_INVARS_NATOM,  nSet + 1, &natom);
  if (error != AB6_NO_ERROR)
    return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
		       "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
		       ab6_error_string_from_id(error),
		       AB6_INVARS_STR(AB6_INVARS_NATOM));
  DBG_fprintf(stderr, "AB structure: with %d atoms and %d types.\n",
	      natom, ntypat);
  
  typat = g_malloc(sizeof(int) * natom);
  error = ab6_invars_get_integer_array(dt, typat, natom,
				       AB6_INVARS_TYPAT, nSet + 1);
  if (error != AB6_NO_ERROR)
    {
      g_free(typat);
      return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			 "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
			 ab6_error_string_from_id(error),
			 AB6_INVARS_STR(AB6_INVARS_TYPAT));
    }

  znucl = g_malloc(sizeof(double) * ntypat);
  types = g_malloc(sizeof(VisuElement*) * ntypat);
  nattyp = g_malloc(sizeof(unsigned int) * ntypat);
  error = ab6_invars_get_real_array(dt, znucl, ntypat,
				    AB6_INVARS_ZNUCL, nSet + 1);
  if (error != AB6_NO_ERROR)
    {
      g_free(typat); g_free(znucl); g_free(types); g_free(nattyp);
      return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			 "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
			 ab6_error_string_from_id(error),
			 AB6_INVARS_STR(AB6_INVARS_ZNUCL));
    }
  DBG_fprintf(stderr, "AB structure: read znucl OK.\n");
  for (i = 0; i < ntypat; i++)
    {
      /* Try to find a name instead of a z number. */
      toolElementsGet_element(&ptChar, &rcov, znucl[i]);
      name = g_strdup(ptChar);
      /* adding name to the hashtable */
      types[i] = visuElementGet_fromName(name);
      if (!types[i])
	{
	  types[i] = visuElementNew_withName(name);
	  g_return_val_if_fail(types[i], (gpointer)0);
	  res = visuElementAdd(types[i]);
	  g_return_val_if_fail(!res, (gpointer)0);
	  renderingAtomicSet_radius(types[i], rcov);
	}
      g_free(name);
    }
  g_free(znucl);
  DBG_fprintf(stderr, "AB structure: all new elements created.\n");
  memset(nattyp, 0, sizeof(unsigned int) * ntypat);
  ntypes = g_malloc(sizeof(VisuElement*) * natom);
  for (i = 0; i < natom; i++)
    {
      nattyp[typat[i] - 1] += 1;
      ntypes[i] = types[typat[i] - 1];
    }
  /* Reduce the arrays when nattyp is 0. */
  nzero = 0;
  index = g_malloc(sizeof(int) * ntypat);
  for (i = 0; i < ntypat; i++)
    {
      if (i > nzero)
	{
	  nattyp[nzero] = nattyp[i];
	  types[nzero]  = types[i];
	}
      index[i] = nzero;
      if (nattyp[i] > 0)
	nzero += 1;
    }
  DBG_fprintf(stderr, "AB structure: removing null types.\n");
  for (i = 0; i < natom; i++)
    {
      DBG_fprintf(stderr, "AB structure: atom %d (%d)", i, typat[i] - 1);
      DBG_fprintf(stderr, " -> %d.\n", index[typat[i] - 1]);
      ntypes[i] = types[index[typat[i] - 1]];
    }
  g_free(typat);
  g_free(index);
  ntypat = nzero;

  DBG_fprintf(stderr, "AB structure: there are %d types in this file.\n", ntypat);
  if (DEBUG)
    for (i = 0; i < ntypat; i++)
      fprintf(stderr, " | %d atom(s) for type %d.\n", nattyp[i], i);
  if (DEBUG)
    for (i = 0; i < natom; i++)
      fprintf(stderr, " | atom %d of type %p.\n", i, (gpointer)ntypes[i]);
  res = visuDataSet_population(data, ntypat, nattyp, types);
  if (!res)
    {
      g_error("Can't store the nodes in the VisuData object.");
    }
  g_free(nattyp);
  g_free(types);

  error = ab6_invars_get_real_array(dt, (double*)rprimd, 9,
				    AB6_INVARS_RPRIMD_ORIG, nSet + 1);
  if (error != AB6_NO_ERROR)
    {
      g_free(ntypes);
      return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			 "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
			 ab6_error_string_from_id(error),
			 AB6_INVARS_STR(AB6_INVARS_RPRIMD_ORIG));
    }
  DBG_fprintf(stderr, " | box definition : ( %f %f %f )\n",
	      rprimd[0][0], rprimd[0][1], rprimd[0][2]);
  DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
	      rprimd[1][0], rprimd[1][1], rprimd[1][2]);
  DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
	      rprimd[2][0], rprimd[2][1], rprimd[2][2]);
  g_return_val_if_fail(matrix_reducePrimitiveVectors(box, rprimd), (gpointer)0);
  visuDataSet_boxGeometry(data, box, BOX_PERIODIC);
  
  coord = g_malloc(sizeof(double) * 3 * natom);
  error = ab6_invars_get_real_array(dt, coord, 3 * natom,
				    AB6_INVARS_XRED_ORIG, nSet + 1);
  if (error != AB6_NO_ERROR)
    {
      g_free(ntypes);
      g_free(coord);
      return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			 "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
			 ab6_error_string_from_id(error),
			 AB6_INVARS_STR(AB6_INVARS_XRED_ORIG));
    }
  for (i = 0; i < natom; i++)
    {
      red[0] = (float)*(coord + 3 * i + 0);
      red[1] = (float)*(coord + 3 * i + 1);
      red[2] = (float)*(coord + 3 * i + 2);
      DBG_fprintf(stderr, " |                  ( %f %f %f )\n",
		  red[0], red[1], red[2]);
      /* Transform the reduced coordinates into cartesians. */
      visuDataConvert_boxCoordinatestoXYZ(data, cart, red);
      DBG_fprintf(stderr, " |               -> ( %f %f %f )\n",
		  cart[0], cart[1], cart[2]);
      visuDataAdd_nodeFromElement(data, ntypes[i], cart, FALSE);
    }
  /* We reset the box size after we set all the coordinates to get the
     OpenGL box right. */
  visuDataApply_boxGeometry(data, 0.f);
  g_free(ntypes);
  g_free(coord);

  /* We set the units. */
  visuDataSet_unit(data, unit_bohr);

  /* We store the spinat array as a property to be used later by the spin
     loading method. */
  spinat = g_malloc(sizeof(double) * 3 * natom);
  g_object_set_data_full(G_OBJECT(data), "ABINIT_spinat", (gpointer)spinat, g_free);
  error = ab6_invars_get_real_array(dt, spinat, 3 * natom,
				    AB6_INVARS_SPINAT, nSet + 1);
  if (error != AB6_NO_ERROR)
    {
      g_free(spinat);
      return g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			 "Abinit loader report error:\n %s\nwhile getting attribute '%s'.",
			 ab6_error_string_from_id(error),
			 AB6_INVARS_STR(AB6_INVARS_SPINAT));
    }

  return (GError*)0;
}

static gpointer parseAbinitThread(gpointer data)
{
  /* Read the file and store its contain as a string. */
  DBG_fprintf(stderr, "AB(1) structure: read and store input file.\n");
  dt = (Ab6Invars*)0;
  dt = ab6_invars_new_from_file((gchar*)data);

  return (gpointer)0;
}

void FC_FUNC(wrtout, WRTOUT)(int *unit,char message[500], char mode_paral[4])
{
  gchar *buf, *ptError, *ptInvars0, *ptInstrng, *ptSize, *ptLen, *ptMinus;
  GError *error;

  buf = g_strndup(message, 500);
  g_strstrip(buf);
  DBG_fprintf(stderr, "AB(1) structure: (%d %c%c%c%c) %s\n",
	      *unit, mode_paral[0], mode_paral[1], mode_paral[2], mode_paral[3], buf);
  /* We analyse buf. If, it contains an error, we test if it is
     about natom in inarvs0. If so, the file is not a valid ABINIT
     file. On the contrary, we get the message and raise an error. */
  ptError = strstr(buf, "ERROR");
  ptInvars0 = strstr(buf, "Input natom must be defined");
  ptInstrng = strstr(buf, "The occurence of a tab");
  ptSize    = strstr(buf, "The size of your input file");
  ptLen     = strstr(buf, "The number of lines already read from input file=");
  ptMinus   = strstr(buf, "the occurence of a minus sign followed");
  if (ptError && ptInvars0)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file (no 'natom' keyword found).");
  else if (ptError && ptInstrng)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file (tab characters found in the file).");
  else if (ptError && ptSize)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file (input file too long).");
  else if (ptError && ptLen)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file (too many lines).");
  else if (ptError && ptMinus)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			"Not an ABINIT file (minus space error).");
  else if (ptError)
    error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FORMAT, buf);
  else
    error = (GError*)0;

  g_free(buf);

  if (error)
    {
      DBG_fprintf(stderr, "AB(1) structure: an error occured while parsing.\n");
      if (dt)
	ab6_invars_free(dt);
      dt = (Ab6Invars*)0;
#ifdef G_THREADS_ENABLED
      g_thread_exit((gpointer)error);
#else
      g_error("ABINIT plug-in requires threads.");
#endif
    }
}
void FC_FUNC(leave_new, LEAVE_NEW)(char mode_paral[4])
{
  GError *error;

  DBG_fprintf(stderr, "AB(1) structure: leave_new(%c%c%c%c)\n",
	      mode_paral[0], mode_paral[1], mode_paral[2], mode_paral[3]);
  if (dt)
    ab6_invars_free(dt);
  dt = (Ab6Invars*)0;
  error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
		      "Not an ABINIT file");
#ifdef G_THREADS_ENABLED
  g_thread_exit((gpointer)error);
#else
  g_error("ABINIT plug-in requires threads.");
#endif
}
void FC_FUNC(timab, TIMAB)()
{
}

static RenderingFormatLoad* abSpinInit()
{
  char *type[] = {"*.in", (char*)0};
  char *descr = _("ABINIT input file format");
  RenderingFormatLoad *meth;
  
  meth = g_malloc(sizeof(RenderingFormatLoad));
  meth->name = "ABINIT input file format";
  meth->fmt = fileFormatNew(descr, type);
  if (!meth->fmt)
    {
      g_error("Can't initialize the XSF loading method, aborting...\n");
    }
  meth->priority = 90;
  meth->load = loadAbinitSpin;

  return meth;
}

static void freeSpin(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * 3, obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopySpin(gconstpointer obj, gpointer data _U_)
{
  float *spinData;

#if GLIB_MINOR_VERSION > 9
  spinData = g_slice_alloc(sizeof(float) * 3);
#else
  spinData = g_malloc(sizeof(float) * 3);
#endif
  if (obj)
    memcpy(spinData, obj, sizeof(float) * 3);
  else
    memset(spinData, 0, sizeof(float) * 3);
    
  return (gpointer)spinData;
}

static gboolean loadAbinitSpin(VisuData *data, const gchar* filename,
			       FileFormat *format _U_, int nSet _U_, GError **error)
{
  double *spinat;
  float spins[3];
  float *svgMaxSpinModulus, *svgSpinValues;
  float sph[3], vals[3];
  VisuNodeProperty *spin;
  VisuDataIter iter;
  GValue spinValue = {0, {{0}, {0}}};

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);
  g_return_val_if_fail(data && filename, FALSE);

  spinat = (double*)g_object_get_data(G_OBJECT(data), "ABINIT_spinat");
  if (!spinat)
    return FALSE;

  /* We check that spin and position are the same. */
  /* TODO... */

  /* Create a storage for max values of spin modulus for each element. */
  svgMaxSpinModulus = g_malloc(sizeof(float) * data->ntype);
  memset(svgMaxSpinModulus, 0, sizeof(float) * data->ntype);
  g_object_set_data_full(G_OBJECT(data), SPINMAXMODULUS_ID,
			 (gpointer)svgMaxSpinModulus, g_free);
  spin = visuNodeNew_pointerProperty(visuDataGet_nodeArray(data), SPINVALUES_ID,
				     freeSpin, newOrCopySpin, (gpointer)0);

  g_value_init(&spinValue, G_TYPE_POINTER);
  visuDataIter_new(data, &iter);
  for(visuDataIter_startNumber(data, &iter); iter.node;
      visuDataIter_nextNodeNumber(data, &iter))
    {
      spins[0] = (float)(spinat + iter.node->number * 3)[0];
      spins[1] = (float)(spinat + iter.node->number * 3)[1];
      spins[2] = (float)(spinat + iter.node->number * 3)[2];
      matrix_cartesianToSpherical(sph, spins);
      vals[SPIN_MODULUS] = sph[0];
      vals[SPIN_THETA]   = sph[1];
      vals[SPIN_PHI]     = sph[2];
      svgSpinValues = newOrCopySpin(vals, (gpointer)0);
      g_value_set_pointer(&spinValue, svgSpinValues);
      visuNodePropertySet_value(spin, iter.node, &spinValue);
      svgMaxSpinModulus[iter.iElement] = MAX(vals[SPIN_MODULUS],
					     svgMaxSpinModulus[iter.iElement]);
    }

  /* We kill the temporary spinat property. */
  g_free(g_object_steal_data(G_OBJECT(data), "ABINIT_spinat"));

  /* Everything is OK. */
  *error = (GError*)0;
  return TRUE;
}
