/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "support.h"
#include "interface.h"
#include "gtk_main.h"
#include "visu_object.h"
#include "visu_pickMesure.h"

#include "gtk_move.h"
#include "gtk_pick.h"
#include "gtk_interactive.h"
#include "gtk_renderingWindowWidget.h"
#include "renderingBackend/visu_windowInterface.h"
#include "openGLFunctions/interactive.h"
#include "extraGtkFunctions/gtk_numericalEntryWidget.h"

/* Callbacks. */
static void onNewData(GObject *obj, VisuData *dataObj, gpointer bool);
static void onDataReady(GObject *obj, VisuData *dataObj, gpointer bool);
static void refreshMoveAxesValues(VisuData *obj, OpenGLView *view, gpointer data);
static void onMovePositionChanged(NumericalEntry *entry,
				  double oldValue, gpointer data);
static void onMoveToOriginalClicked(GtkButton *button, gpointer data);
static void onRemoveNodeClicked(GtkButton *button, gpointer user_data);
static void onAddNodeClicked(GtkButton *button, gpointer user_data);
static void onMoveMethod(GtkToggleButton *toggle, gpointer data);
static void onPageEnter(GtkNotebook *notebook, GtkNotebookPage *page,
			guint page_num, gpointer user_data);
static void onCheckDuplicate(GtkToggleButton *button, gpointer data);
static void onDuplicate(GtkButton *button, gpointer data);
static void updateInfos(VisuInteractive *inter,
			PickMesure *pickMesure, gpointer data);
static void onMoveToggled(GtkToggleButton *toggle, gpointer data);
static void onSpinBasis(GtkSpinButton *spin, gpointer data);
static gboolean removeHighlight(gpointer data);
static void onChooseBasis(GtkToggleButton *toggle, gpointer data);
static void onBasisSelected(VisuInteractive *inter, PickMesure *pick, gpointer data);
static void onApplyBasis(GtkButton *button, gpointer data);
static void onPopulationChanged(VisuData *dataObj, gint *nodes, gpointer data);

#define GTK_MOVE_INFO				\
  _("left-button\t\t: drag node(s) in the screen plane\n"	\
    "shift-left-button\t: drag node(s) along x or y axis\n"	\
    "control-left-button\t: drag node(s) along z axis\n"	\
    "right-button\t\t: switch to observe")

/* Pick informations hook */
static int movedNode = -1;
static float moveNodeOrigin[3];
#define GTK_MOVE_NO_NODE         _("<i>none</i>")
#define GTK_MOVE_NO_REGION       _("<i>unset</i>")
static gulong onMove_id;
static gulong onSelection_id;
static gulong onSpin_id[4];
static VisuInteractiveId currentMode = interactive_move;
static guint currentAxe;

/* Widgets */
static GtkWidget *notebookAction;
static GtkWidget *observeWindow;
static GtkWidget *entryMoveXYZ[3];
static GtkWidget *entryAddXYZ[3];
static GtkWidget *comboElements;
static GtkListStore *listElements;
static GtkWidget *labelMoveId, *labelNMoves;
static GtkWidget *radioMovePick, *radioMoveRegion;
enum
  {
    COLUMN_NAME,             /* The label shown */
    COLUMN_POINTER_TO_DATA,  /* Pointer to the VisuElement. */
    N_COLUMNS
  };
static GtkWidget *checkDuplicate, *comboDuplicate, *buttonDupplicate;
static GtkWidget *labelOriginX, *labelOriginY, *labelOriginZ;
static GtkWidget *labelScreenHorizontal, *labelScreenVertical;
static GtkWidget *spinABC[4], *toggleABC[4], *applyBasis;
static gchar *lblSpinABC[4] = {"O:", "A:", "B:", "C:"};
static gint prevBasis[4] = {0, 0, 0, 0};
static guint timeoutBasis[4];


/********************/
/* Public routines. */
/********************/
GtkWidget* gtkMoveBuild_interface(GtkMain *main, gchar **label,
				  gchar **help, GtkWidget **radio)
{
  GtkWidget *wd, *hbox, *vbox;
  GtkCellRenderer *renderer;
  RenderingWindow *window;
  VisuData *data;
  GtkTreeIter iter;
  VisuDataIter dataIter;
  int i;

  *label = g_strdup("Pick");
  *help  = g_strdup(GTK_MOVE_INFO);
  *radio = lookup_widget(main->interactiveDialog, "radioMove");
  g_signal_connect(G_OBJECT(*radio), "toggled",
		   G_CALLBACK(onMoveToggled), (gpointer)main->interactiveDialog);

  window = visuRenderingWindowGet_current();
  g_return_val_if_fail(window, (GtkWidget*)0);
  data = renderingWindowGet_visuData(window);
  g_return_val_if_fail(data, (GtkWidget*)0);

  observeWindow = main->interactiveDialog;
  
  vbox = lookup_widget(main->interactiveDialog, "vbox21");

  notebookAction = lookup_widget(main->interactiveDialog, "notebookAction");
  g_signal_connect(G_OBJECT(notebookAction), "switch-page",
		   G_CALLBACK(onPageEnter), (gpointer)0);

  labelMoveId = lookup_widget(main->interactiveDialog, "labelMoveId");
  gtk_label_set_markup(GTK_LABEL(labelMoveId), GTK_MOVE_NO_NODE);

  labelNMoves = lookup_widget(main->interactiveDialog, "labelNMoves");
  gtk_label_set_markup(GTK_LABEL(labelNMoves), GTK_MOVE_NO_REGION);

  radioMovePick = lookup_widget(main->interactiveDialog, "radioMovePick");
  gtk_widget_set_name(radioMovePick, "message_radio");
  radioMoveRegion = lookup_widget(main->interactiveDialog, "radioMoveRegion");
  gtk_widget_set_name(radioMoveRegion, "message_radio");

  labelOriginX = lookup_widget(main->interactiveDialog, "labelOriginalX");
  labelOriginY = lookup_widget(main->interactiveDialog, "labelOriginalY");
  labelOriginZ = lookup_widget(main->interactiveDialog, "labelOriginalZ");

  labelScreenHorizontal = lookup_widget(main->interactiveDialog, "labelHorizontalAxe");
  labelScreenVertical   = lookup_widget(main->interactiveDialog, "labelVerticalAxe");
  g_signal_connect(G_OBJECT(data), "OpenGLThetaPhiOmega",
		   G_CALLBACK(refreshMoveAxesValues), (gpointer)0);

  wd = lookup_widget(main->interactiveDialog, "tableMovePick");
  entryMoveXYZ[0] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[0]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[0], 1, 2, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  gtk_widget_show(entryMoveXYZ[0]);
  entryMoveXYZ[1] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[1]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[1], 2, 3, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  gtk_widget_show(entryMoveXYZ[1]);
  entryMoveXYZ[2] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryMoveXYZ[2]), 6);
  gtk_table_attach(GTK_TABLE(wd), entryMoveXYZ[2], 3, 4, 2, 3,
		   GTK_SHRINK, GTK_SHRINK, 0, 0);
  gtk_widget_show(entryMoveXYZ[2]);

  /* The add line. */
  wd = lookup_widget(main->interactiveDialog, "hboxAddNode");
  entryAddXYZ[0] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[0]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[0], FALSE, FALSE, 0);
  entryAddXYZ[1] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[1]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[1], FALSE, FALSE, 0);
  entryAddXYZ[2] = numericalEntry_new(0.);
  gtk_entry_set_width_chars(GTK_ENTRY(entryAddXYZ[2]), 6);
  gtk_box_pack_start(GTK_BOX(wd), entryAddXYZ[2], FALSE, FALSE, 0);
  /* We create the structure that store the VisuElements */
  listElements = gtk_list_store_new(N_COLUMNS, G_TYPE_STRING, G_TYPE_POINTER);
  comboElements = gtk_combo_box_new_with_model(GTK_TREE_MODEL(listElements));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(comboElements), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(comboElements),
				renderer, "text", COLUMN_NAME);
  gtk_box_pack_start(GTK_BOX(wd), comboElements, FALSE, FALSE, 0);
  gtk_box_reorder_child(GTK_BOX(wd), comboElements, 0);
  gtk_widget_show_all(wd);
  /* We populate the combobox. */
  visuDataIter_new(data, &dataIter);
  for(visuDataIter_start(data, &dataIter); dataIter.element;
      visuDataIter_nextElement(data, &dataIter))
    {
      gtk_list_store_append(listElements, &iter);
      gtk_list_store_set(listElements, &iter,
			 COLUMN_NAME, dataIter.element->name,
			 COLUMN_POINTER_TO_DATA, dataIter.element,
			 -1);
    }
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboElements), 0);

  /* The duplicate line. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
  wd = gtk_label_new(_("<b>Duplicate nodes:</b>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 0);
  wd = gtk_label_new(_("<span size=\"smaller\">(the nodes listed in the pick tab)</span>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  gtk_widget_show_all(hbox);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  wd = gtk_check_button_new_with_mnemonic(_("du_plicate nodes as they are"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  checkDuplicate = wd;
  wd = gtk_label_new(_(" or as new: "));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  wd = gtk_combo_box_new_with_model(GTK_TREE_MODEL(listElements));
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(wd), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(wd),
				renderer, "text", COLUMN_NAME);
  gtk_combo_box_set_active(GTK_COMBO_BOX(wd), 0);
  gtk_widget_set_sensitive(wd, FALSE);
  comboDuplicate = wd;
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  wd = gtk_button_new_with_mnemonic(_("_duplicate"));
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 5);
  buttonDupplicate = wd;
  gtk_widget_show_all(hbox);

  /* The Basis line. */
  wd = gtk_label_new(_("<b>Change the basis set:</b>"));
  gtk_widget_show(wd);
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(vbox), wd, FALSE, FALSE, 5);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  for (i = 0; i < 4; i++)
    {
      wd = gtk_label_new(_(lblSpinABC[i]));
      gtk_misc_set_alignment(GTK_MISC(wd), 1., 0.5);
      gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 5);
      spinABC[i] = gtk_spin_button_new_with_range(0, dataIter.idMax, 1);
      onSpin_id[i] = g_signal_connect(G_OBJECT(spinABC[i]), "value-changed",
				      G_CALLBACK(onSpinBasis), GINT_TO_POINTER(i));
      gtk_box_pack_start(GTK_BOX(hbox), spinABC[i], FALSE, FALSE, 0);
      toggleABC[i] = gtk_toggle_button_new();
      gtk_box_pack_start(GTK_BOX(hbox), toggleABC[i], FALSE, FALSE, 0);
      gtk_container_add(GTK_CONTAINER(toggleABC[i]),
			gtk_image_new_from_stock(GTK_STOCK_FIND,
						 GTK_ICON_SIZE_MENU));
      g_signal_connect(G_OBJECT(toggleABC[i]), "toggled",
		       G_CALLBACK(onChooseBasis), GINT_TO_POINTER(i));
    }
  applyBasis = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), applyBasis, FALSE, FALSE, 5);
  gtk_container_add(GTK_CONTAINER(applyBasis),
		    gtk_image_new_from_stock(GTK_STOCK_APPLY, GTK_ICON_SIZE_MENU));
  gtk_widget_set_sensitive(applyBasis, FALSE);
  g_signal_connect(G_OBJECT(applyBasis), "clicked",
		   G_CALLBACK(onApplyBasis), (gpointer)0);
  gtk_widget_show_all(hbox);


  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onNewData), (gpointer)0);
  onNewData((GObject*)0, data, (gpointer)0);

  g_signal_connect(G_OBJECT(entryMoveXYZ[0]), "value-changed",
		   G_CALLBACK(onMovePositionChanged), GINT_TO_POINTER(0));
  g_signal_connect(G_OBJECT(entryMoveXYZ[1]), "value-changed",
		   G_CALLBACK(onMovePositionChanged), GINT_TO_POINTER(1));
  g_signal_connect(G_OBJECT(entryMoveXYZ[2]), "value-changed",
		   G_CALLBACK(onMovePositionChanged), GINT_TO_POINTER(2));
  wd = lookup_widget(main->interactiveDialog, "buttonMoveOriginal");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onMoveToOriginalClicked), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "buttonRemoveNode");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onRemoveNodeClicked), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "buttonAddNode");
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onAddNodeClicked), (gpointer)0);
  wd = lookup_widget(main->interactiveDialog, "radioMovePick");
  g_signal_connect(G_OBJECT(wd), "toggled",
		   G_CALLBACK(onMoveMethod), GINT_TO_POINTER(0));
  wd = lookup_widget(main->interactiveDialog, "radioMoveRegion");
  g_signal_connect(G_OBJECT(wd), "toggled",
		   G_CALLBACK(onMoveMethod), GINT_TO_POINTER(1));

  g_signal_connect(G_OBJECT(checkDuplicate), "toggled",
		   G_CALLBACK(onCheckDuplicate), comboDuplicate);
  g_signal_connect(G_OBJECT(buttonDupplicate), "clicked",
		   G_CALLBACK(onDuplicate), (gpointer)0);

  return (GtkWidget*)0;
}


static void setLabelsOrigin(GList *nodes)
{
  gchar numero[256];
  VisuNode *node;

  if (nodes && !nodes->next)
    {
      node = (VisuNode*)nodes->data;
      sprintf(numero, _("node %d"), node->number + 1);
      gtk_label_set_markup(GTK_LABEL(labelMoveId), numero);
      gtk_label_set_markup(GTK_LABEL(labelNMoves), GTK_MOVE_NO_REGION);
      /* Set the origin position values. */
      moveNodeOrigin[0] = node->xyz[0];
      moveNodeOrigin[1] = node->xyz[1];
      moveNodeOrigin[2] = node->xyz[2];
      sprintf(numero, "%5.2f", moveNodeOrigin[0]);
      gtk_label_set_text(GTK_LABEL(labelOriginX), numero);
      sprintf(numero, "%5.2f", moveNodeOrigin[1]);
      gtk_label_set_text(GTK_LABEL(labelOriginY), numero);
      sprintf(numero, "%5.2f", moveNodeOrigin[2]);
      gtk_label_set_text(GTK_LABEL(labelOriginZ), numero);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), node->xyz[0]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), node->xyz[1]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), node->xyz[2]);
    }
  else if (nodes)
    {
      movedNode = -1;
      gtk_label_set_markup(GTK_LABEL(labelMoveId), GTK_MOVE_NO_NODE);
      sprintf(numero, _("%d node(s)"), g_list_length(nodes));
      gtk_label_set_markup(GTK_LABEL(labelNMoves), numero);
      numero[0] = '\0';
      gtk_label_set_text(GTK_LABEL(labelOriginX), numero);
      gtk_label_set_text(GTK_LABEL(labelOriginY), numero);
      gtk_label_set_text(GTK_LABEL(labelOriginZ), numero);
    }
  else
    {
      movedNode = -1;
      gtk_label_set_markup(GTK_LABEL(labelMoveId), GTK_MOVE_NO_NODE);
      gtk_label_set_markup(GTK_LABEL(labelNMoves), GTK_MOVE_NO_REGION);
      numero[0] = '\0';
      gtk_label_set_text(GTK_LABEL(labelOriginX), numero);
      gtk_label_set_text(GTK_LABEL(labelOriginY), numero);
      gtk_label_set_text(GTK_LABEL(labelOriginZ), numero);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
    }
}
static void updateLabelsPosition(GList *nodes, float drag[3])
{
  VisuNode *node;
  double val;

  /* Modify the entry labels. */
  if (nodes && !nodes->next)
    {
      node = (VisuNode*)nodes->data;
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), node->xyz[0]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), node->xyz[1]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), node->xyz[2]);
    }
  else if (nodes)
    {
      val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]));
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), val + drag[0]);
      val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]));
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), val + drag[1]);
      val = numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]));
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), val + drag[2]);
    }
  else
    {
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
    }
}

void gtkMoveStart(RenderingWindow *window)
{
  VisuInteractive *inter;

  inter = renderingWindowGet_interactive(window);

  DBG_fprintf(stderr, "Gtk Move: set the two signals.\n");
  visuInteractiveSet_type(inter, currentMode);
  onMove_id = g_signal_connect(G_OBJECT(inter), "move", G_CALLBACK(updateInfos),
			       (gpointer)renderingWindowGet_visuData(window));
  if (currentMode != interactive_move)
    g_signal_handler_block(G_OBJECT(inter), onMove_id);
  onSelection_id = g_signal_connect(G_OBJECT(inter), "selection",
				    G_CALLBACK(onBasisSelected),
				    (gpointer)renderingWindowGet_visuData(window));
  if (currentMode != interactive_pick)
    g_signal_handler_block(G_OBJECT(inter), onSelection_id);

  if (currentMode == interactive_pick)
    renderingWindowPush_message(window, _("Pick a node with the mouse"));
}
void gtkMoveStop(RenderingWindow *window)
{
  DBG_fprintf(stderr, "Gtk Move: unset the two signals.\n");

  g_signal_handler_disconnect(G_OBJECT(renderingWindowGet_interactive(window)),
			      onMove_id);
  g_signal_handler_disconnect(G_OBJECT(renderingWindowGet_interactive(window)),
			      onSelection_id);
  visuInteractiveSet_type(renderingWindowGet_interactive(window),
			  interactive_none);

  if (currentMode == interactive_pick)
    renderingWindowPop_message(window);
}
static void updateInfos(VisuInteractive *inter,
			PickMesure *pickMesure, gpointer data)
{
  PickMesureType type;
  GList *nodes, *lst, *tmpLst;
  float *drag;
  VisuNode *node;
  VisuData *dataObj;

  DBG_fprintf(stderr, "Gtk Move: callback on move action.\n");

  if (!pickMesureGet_newsAvailable(pickMesure, &type))
    return /* FALSE */;

  nodes = (GList*)0;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      node = pickMesureGet_selectedNode(pickMesure);
      if (!node)
	return /* FALSE */;
      nodes = g_list_append(nodes, (gpointer)node);
      movedNode = node->number;
    }
  else
    {
      dataObj = VISU_DATA(data);
      g_return_if_fail(dataObj);

      movedNode = -1;
      tmpLst = lst = gtkPickGet_nodeSelection();
      while(tmpLst)
	{
	  node = visuDataGet_nodeFromNumber(dataObj, GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    nodes = g_list_prepend(nodes, (gpointer)node);
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);

      if (!nodes)
	return /* FALSE */;
    }
  switch (type)
    {
    case PICK_DRAG_START:
      visuInteractiveStart_move(inter, nodes);
      setLabelsOrigin(nodes);
      break;
    case PICK_DRAG_STOP:
      visuInteractiveStop_move(inter, nodes);
      break;
    case PICK_DRAG_MOVE:
      drag = pickMesureGet_drag(pickMesure);
      visuInteractiveMove(inter, nodes, drag);
      updateLabelsPosition(nodes, drag);
      break;
    default:
      g_error("Not a drag action!");
    }
  g_list_free(nodes);

  return /* FALSE */;
}

/****************/
/* Private part */
/****************/

static void onPageEnter(GtkNotebook *notebook _U_, GtkNotebookPage *page _U_,
			guint page_num, gpointer user_data _U_)
{
  gchar numero[256];
  GList *lst;

  if (page_num != 1)
    return;

  lst = gtkPickGet_nodeSelection();
  sprintf(numero, _("%d node(s)"), g_list_length(lst));
  gtk_label_set_markup(GTK_LABEL(labelNMoves), numero);
  g_list_free(lst);
}

static void onNewData(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLThetaPhiOmega",
		   G_CALLBACK(refreshMoveAxesValues), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		   G_CALLBACK(onPopulationChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "NodePopulationIncrease",
		   G_CALLBACK(onPopulationChanged), (gpointer)0);
}

static void onDataReady(GObject *obj _U_, VisuData *dataObj, gpointer data _U_)
{
  GtkTreeIter iter;
  VisuDataIter dataIter;
  int i;

  if (dataObj)
    {
      DBG_fprintf(stderr, "Gtk Move: caught 'dataReadyForRendering' signal,"
		  " setting the element combo.\n");
      visuDataIter_new(dataObj, &dataIter);
      /* We update the element combo if necessary. */
      if (!visuDataGet_changeElementFlag(dataObj))
	{
	  gtk_list_store_clear(listElements);
	  for(visuDataIter_start(dataObj, &dataIter); dataIter.element;
	      visuDataIter_nextElement(dataObj, &dataIter))
	    {
	      gtk_list_store_append(listElements, &iter);
	      gtk_list_store_set(listElements, &iter,
				 COLUMN_NAME, dataIter.element->name,
				 COLUMN_POINTER_TO_DATA, dataIter.element,
				 -1);
	    }
	  gtk_combo_box_set_active(GTK_COMBO_BOX(comboElements), 0);
	}
      for (i = 0; i < 4; i++)
	gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[i]), 0,
				  dataIter.idMax);
    }
  else
    {
      gtk_list_store_clear(listElements);
    }
}

static void refreshMoveAxesValues(VisuData *obj _U_, OpenGLView *view, gpointer data)
{
  gint id;
  float x[3], y[3];
  char tmpChr[20];

  id = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebookAction));
  DBG_fprintf(stderr, "Gtk Observe/pick: refresh screen basis set.\n");
  DBG_fprintf(stderr, " | %d %d\n", id + 1, action_move);
  if (id + 1 == action_move || GPOINTER_TO_INT(data))
    {
      OpenGLViewGet_screenAxes(view, x, y);
      sprintf(tmpChr, "(%4.2f;%4.2f;%4.2f)", x[0], x[1], x[2]);
      gtk_label_set_text(GTK_LABEL(labelScreenHorizontal), tmpChr);
      sprintf(tmpChr, "(%4.2f;%4.2f;%4.2f)", y[0], y[1], y[2]);
      gtk_label_set_text(GTK_LABEL(labelScreenVertical), tmpChr);
    }
}
static void onMoveMethod(GtkToggleButton *toggle, gpointer data _U_)
{
  if (!gtk_toggle_button_get_active(toggle))
    return;

  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
  numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
}
static void onMovePositionChanged(NumericalEntry *entry, double oldValue,
				  gpointer data)
{
  float *valOfNode;
  VisuData *dataObj;
  VisuNode *node;
  GList *lst, *tmpLst;

  g_return_if_fail(GPOINTER_TO_INT(data) >= 0 && GPOINTER_TO_INT(data) < 3);

  /* We are not typing. */
  if (!gtk_widget_is_focus(GTK_WIDGET(entry)))
    return;

  dataObj = renderingWindowGet_visuData
    (RENDERING_WINDOW(visuRenderingWindowGet_current()));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visuDataGet_nodeFromNumber(dataObj, movedNode);
      g_return_if_fail(node);

      valOfNode = node->xyz + GPOINTER_TO_INT(data);
  
      if (*valOfNode == (float)numericalEntryGet_value(entry))
	return;

      *valOfNode = (float)numericalEntryGet_value(entry);
      visuData_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);
      visuDataEmit_nodePositionChanged(dataObj);
    }
  else
    {
      tmpLst = lst = gtkPickGet_nodeSelection();
      while(tmpLst)
	{
	  node = visuDataGet_nodeFromNumber(dataObj, GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    node->xyz[GPOINTER_TO_INT(data)] +=
	      numericalEntryGet_value(entry) - oldValue;
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);

      visuData_createAllNodes(dataObj);

    }

  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onMoveToOriginalClicked(GtkButton *button _U_, gpointer data _U_)
{
  VisuData *dataObj;
  VisuNode *node;
  float trans[3];
  GList *lst, *tmpLst;

  dataObj = renderingWindowGet_visuData
    (RENDERING_WINDOW(visuRenderingWindowGet_current()));

  /* We are not typing. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visuDataGet_nodeFromNumber(dataObj, movedNode);
      g_return_if_fail(node);

      node->xyz[0] = moveNodeOrigin[0];
      node->xyz[1] = moveNodeOrigin[1];
      node->xyz[2] = moveNodeOrigin[2];
      visuData_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);

      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), moveNodeOrigin[0]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), moveNodeOrigin[1]);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), moveNodeOrigin[2]);
    }
  else
    {
      trans[0] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]));
      trans[1] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]));
      trans[2] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]));

      tmpLst = lst = gtkPickGet_nodeSelection();
      while(tmpLst)
	{
	  node = visuDataGet_nodeFromNumber(dataObj, GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    {
	      node->xyz[0] -= trans[0];
	      node->xyz[1] -= trans[1];
	      node->xyz[2] -= trans[2];
	    }
	  tmpLst = g_list_next(tmpLst);
	}
      g_list_free(lst);

      visuData_createAllNodes(dataObj);

      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[0]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[1]), 0.);
      numericalEntrySet_value(NUMERICAL_ENTRY(entryMoveXYZ[2]), 0.);
    }
  visuDataEmit_nodePositionChanged(dataObj);

  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onRemoveNodeClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  int *nodes;
  VisuData *dataObj;
  VisuNode *node;
  int iEle, i, n;
  GList *lst, *tmpLst;

  DBG_fprintf(stderr, "Gtk Observe/pick: remove the selected node %d.\n", movedNode);
  
  dataObj = renderingWindowGet_visuData
    (RENDERING_WINDOW(visuRenderingWindowGet_current()));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radioMovePick)))
    {
      if (movedNode < 0)
	return;

      node = visuDataGet_nodeFromNumber(dataObj, movedNode);
      if (!node)
	return;
      iEle = node->posElement;

      nodes = g_malloc(sizeof(int) * 2);
      nodes[0] = movedNode;
      nodes[1] = -1;
      visuDataRemove_nodes(dataObj, nodes);
      g_free(nodes);

      /* Copy the coordinates to the add entries. */
      for (i = 0; i < 3; i++)
	numericalEntrySet_value(NUMERICAL_ENTRY(entryAddXYZ[i]),
				numericalEntryGet_value(NUMERICAL_ENTRY(entryMoveXYZ[i])));
      gtk_combo_box_set_active(GTK_COMBO_BOX(comboElements), iEle);

      visuData_createNodes(dataObj, dataObj->fromIntToVisuElement[iEle]);
    }
  else
    {
      DBG_fprintf(stderr, "Gtk Move: get list of selected nodes.\n");
      tmpLst = lst = gtkPickGet_nodeSelection();
      n = g_list_length(lst);
      if (n > 0)
	{
	  nodes = g_malloc(sizeof(int) * (n + 1));
	  i = 0;
	  while(tmpLst)
	    {
	      nodes[i] = GPOINTER_TO_INT(tmpLst->data);
	      DBG_fprintf(stderr, " | will remove %3d %d\n", i, nodes[i]);
	      i += 1;
	      tmpLst = g_list_next(tmpLst);
	    }
	  nodes[i] = -1;
	  visuDataRemove_nodes(dataObj, nodes);
	  g_free(nodes);
	}
      g_list_free(lst);

      visuData_createAllNodes(dataObj);
    }
  /* We remove the node as selected one. */
  setLabelsOrigin((GList*)0);

  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onAddNodeClicked(GtkButton *button _U_, gpointer user_data _U_)
{
  RenderingWindow *window;
  VisuData *dataObj;
  GtkTreeIter iter;
  gboolean valid;
  VisuElement *element;
  float xyz[3];

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  g_return_if_fail(window);

  dataObj = renderingWindowGet_visuData(window);

  valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboElements), &iter);
  g_return_if_fail(valid);

  gtk_tree_model_get(GTK_TREE_MODEL(listElements), &iter,
		     COLUMN_POINTER_TO_DATA, &element,
		     -1);
  xyz[0] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[0]));
  xyz[1] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[1]));
  xyz[2] = (float)numericalEntryGet_value(NUMERICAL_ENTRY(entryAddXYZ[2]));
  visuDataAdd_nodeFromElement(dataObj, element, xyz, TRUE);

  /* We update all drawings. */
  visuDataEmit_nodePositionChanged(dataObj);
  visuData_createNodes(dataObj, element);
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onCheckDuplicate(GtkToggleButton *button, gpointer data)
{
  gtk_widget_set_sensitive(GTK_WIDGET(data), !gtk_toggle_button_get_active(button));
}
static void onDuplicate(GtkButton *button _U_, gpointer data _U_)
{
  RenderingWindow *window;
  VisuData *dataObj;
  GList *tmpLst, *lst;
  GtkTreeIter iter;
  gboolean valid, custom;
  VisuElement *element;
  float coord[3], *trans;
  VisuNode *node;

  DBG_fprintf(stderr, "Gtk Move: duplicate selected nodes.\n");

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  g_return_if_fail(window);

  dataObj = renderingWindowGet_visuData(window);

  custom = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDuplicate));
  if (custom)
    {
      valid = gtk_combo_box_get_active_iter(GTK_COMBO_BOX(comboDuplicate), &iter);
      g_return_if_fail(valid);

      gtk_tree_model_get(GTK_TREE_MODEL(listElements), &iter,
			 COLUMN_POINTER_TO_DATA, &element,
			 -1);
      DBG_fprintf(stderr, "Gtk Move: change element to '%s'.\n", element->name);
    }

  lst = gtkPickGet_nodeSelection();
  if (!lst)
    return;

  trans = visuDataGet_XYZtranslation(dataObj);
  for (tmpLst = lst; tmpLst; tmpLst = g_list_next(tmpLst))
    {
      node = visuDataGet_nodeFromNumber
	(dataObj, (guint)GPOINTER_TO_INT(tmpLst->data));
      visuDataGet_nodePosition(dataObj, node, coord);
      coord[0] -= trans[0];
      coord[1] -= trans[1];
      coord[2] -= trans[2];
      if (custom)
	visuDataAdd_nodeFromElement(dataObj, element, coord, !tmpLst->next);
      else
	visuDataAdd_nodeFromIndex(dataObj, node->posElement, coord, !tmpLst->next);
    }
  g_list_free(lst);
  g_free(trans);

  /* We update all drawings. */
  if (custom)
    visuData_createNodes(dataObj, element);
  else
    visuData_createAllNodes(dataObj);
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onMoveToggled(GtkToggleButton *toggle, gpointer data)
{
  GtkWidget *wd;
  gboolean value;

  value = gtk_toggle_button_get_active(toggle);
  wd = lookup_widget(GTK_WIDGET(data), "alignment41");
  gtk_widget_set_sensitive(wd, value);
}
static void onSpinBasis(GtkSpinButton *spin, gpointer data)
{
  int i;
  RenderingWindow *window;
  PickMesure *pick;
  gboolean valid;
  VisuData *dataObj;
  
  i = GPOINTER_TO_INT(data);

  window = RENDERING_WINDOW(currentRenderingWindow);
  pick = visuInteractiveGet_pickMesure(renderingWindowGet_interactive(window));
  dataObj = renderingWindowGet_visuData(window);

  /* Remove the previous one. */
  if (prevBasis[i] > 0)
    pickMesureSet_highlight(pick, (guint)(prevBasis[i] - 1), FALSE);
  if (timeoutBasis[i] > 0)
    g_source_remove(timeoutBasis[i]);

  prevBasis[i] = (gint)gtk_spin_button_get_value(spin);
  if (prevBasis[i] > 0 && visuDataGet_nodeFromNumber(dataObj, prevBasis[i]))
    {
      /* Add a new highlight. */
#if GLIB_MINOR_VERSION > 13
      timeoutBasis[i] = g_timeout_add_seconds(1, removeHighlight,
					      GINT_TO_POINTER(prevBasis[i]));
#else
      timeoutBasis[i] = g_timeout_add(1000, removeHighlight,
				      GINT_TO_POINTER(prevBasis[i]));
#endif
      pickMesureSet_highlight(pick, (guint)(prevBasis[i] - 1), TRUE);
    }

  valid = TRUE;
  for(i = 0; i < 4; i++)
    valid = valid && (gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[i])) > 0.);
  gtk_widget_set_sensitive(applyBasis, valid);
  
  pickMesureRebuild_classicalList(dataObj);
  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
}
static gboolean removeHighlight(gpointer data)
{
  int i;
  RenderingWindow *window;
  PickMesure *pick;

  i = GPOINTER_TO_INT(data);
  g_return_val_if_fail(i > 0, FALSE);

  window = RENDERING_WINDOW(currentRenderingWindow);
  pick = visuInteractiveGet_pickMesure(renderingWindowGet_interactive(window));

  /* Remove the previous one. */
  pickMesureSet_highlight(pick, (guint)(i - 1), FALSE);
  pickMesureRebuild_classicalList(renderingWindowGet_visuData(window));
  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));

  return FALSE;
}
static void onChooseBasis(GtkToggleButton *toggle, gpointer data)
{
  guint i;
  VisuInteractive *inter;
  PickMesure *pick;
  gboolean valid;

  inter = renderingWindowGet_interactive(RENDERING_WINDOW(currentRenderingWindow));

  DBG_fprintf(stderr, "Gtk Move: one toggle chooser has been toggled.\n");
  valid = gtk_toggle_button_get_active(toggle);
  if (valid)
    {
      currentAxe = GPOINTER_TO_INT(data);
      for(i = 0; i < 4; i++)
	if (i != currentAxe)
	  gtk_widget_set_sensitive(toggleABC[i], FALSE);
      DBG_fprintf(stderr, "Gtk Move: change the mode to pick.\n");
      currentMode = interactive_pick;
      pick = visuInteractiveGet_pickMesure(inter);
      pickMesureSet_pickNode(pick, -1, PICK_REFERENCE_1);
      pickMesureSet_pickNode(pick, -1, PICK_REFERENCE_2);
      g_signal_handler_block(G_OBJECT(inter), onMove_id);
      g_signal_handler_unblock(G_OBJECT(inter), onSelection_id);

      renderingWindowPush_message(RENDERING_WINDOW(currentRenderingWindow),
				  _("Pick a node with the mouse"));
    }
  else
    {
      for(i = 0; i < 4; i++)
	gtk_widget_set_sensitive(toggleABC[i], TRUE);
      DBG_fprintf(stderr, "Gtk Move: change the mode to move.\n");
      currentMode = interactive_move;
      g_signal_handler_unblock(G_OBJECT(inter), onMove_id);
      g_signal_handler_block(G_OBJECT(inter), onSelection_id);

      renderingWindowPop_message(RENDERING_WINDOW(currentRenderingWindow));
    }
  visuInteractiveSet_type(inter, currentMode);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "alignment43"), !valid);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "alignment44"), !valid);
  gtk_widget_set_sensitive(lookup_widget(observeWindow, "alignment27"), !valid);
}
static void onBasisSelected(VisuInteractive *inter _U_,
			    PickMesure *pick, gpointer data _U_)
{
  VisuNode *node;

  node = pickMesureGet_selectedNode(pick);
  if (node)
    {
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinABC[currentAxe]),
				node->number + 1);
      gtk_toggle_button_set_active
	(GTK_TOGGLE_BUTTON(toggleABC[currentAxe]), FALSE);
    }
}
static void onApplyBasis(GtkButton *button _U_, gpointer data _U_)
{
  VisuData *dataObj;

  dataObj = renderingWindowGet_visuData(RENDERING_WINDOW(currentRenderingWindow));
  visuDataSet_newBasis(dataObj,
		       (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[0])) - 1,
		       (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[1])) - 1,
		       (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[2])) - 1,
		       (guint)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinABC[3])) - 1);
}
static void onPopulationChanged(VisuData *dataObj, gint *nodes _U_, gpointer data _U_)
{
  VisuDataIter iter;

  visuDataIter_new(dataObj, &iter);
  g_signal_handler_block(G_OBJECT(spinABC[0]), onSpin_id[0]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[0]), 0, iter.idMax);
  g_signal_handler_unblock(G_OBJECT(spinABC[0]), onSpin_id[0]);

  g_signal_handler_block(G_OBJECT(spinABC[1]), onSpin_id[1]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[1]), 0, iter.idMax);
  g_signal_handler_unblock(G_OBJECT(spinABC[1]), onSpin_id[1]);

  g_signal_handler_block(G_OBJECT(spinABC[2]), onSpin_id[2]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[2]), 0, iter.idMax);
  g_signal_handler_unblock(G_OBJECT(spinABC[2]), onSpin_id[2]);

  g_signal_handler_block(G_OBJECT(spinABC[3]), onSpin_id[3]);
  gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinABC[3]), 0, iter.idMax);
  g_signal_handler_unblock(G_OBJECT(spinABC[3]), onSpin_id[3]);
}
