/* valadelegate.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valadelegate.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valatypeparameter.h>
#include <vala/valaformalparameter.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>

struct _ValaDelegatePrivate {
	gboolean _instance;
	GeeList* type_parameters;
	GeeList* parameters;
	char* cname;
	ValaDataType* _return_type;
};
#define VALA_DELEGATE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DELEGATE, ValaDelegatePrivate))
enum  {
	VALA_DELEGATE_DUMMY_PROPERTY,
	VALA_DELEGATE_RETURN_TYPE,
	VALA_DELEGATE_INSTANCE
};
static void vala_delegate_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_delegate_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_delegate_real_get_cname (ValaTypesymbol* base, gboolean const_type);
static void vala_delegate_process_ccode_attribute (ValaDelegate* self, ValaAttribute* a);
static gboolean vala_delegate_real_is_reference_type (ValaTypesymbol* base);
static char* vala_delegate_real_get_type_id (ValaTypesymbol* base);
static char* vala_delegate_real_get_marshaller_type_name (ValaTypesymbol* base);
static char* vala_delegate_real_get_get_value_function (ValaTypesymbol* base);
static char* vala_delegate_real_get_set_value_function (ValaTypesymbol* base);
static void vala_delegate_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_delegate_parent_class = NULL;
static void vala_delegate_dispose (GObject * obj);


/**
 * Creates a new delegate.
 *
 * @param name        delegate type name
 * @param return_type return type
 * @param source      reference to source code
 * @return            newly created delegate
 */
ValaDelegate* vala_delegate_new (const char* name, ValaDataType* return_type, ValaSourceReference* source_reference) {
	ValaDelegate * self;
	g_return_val_if_fail (return_type == NULL || VALA_IS_DATA_TYPE (return_type), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_DELEGATE, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_delegate_set_return_type (self, return_type);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_delegate_add_type_parameter (ValaDelegate* self, ValaTypeParameter* p) {
	g_return_if_fail (VALA_IS_DELEGATE (self));
	g_return_if_fail (p == NULL || VALA_IS_TYPEPARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_TYPESYMBOL (self);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (p)), VALA_SYMBOL (p));
}


/**
 * Appends paramater to this callback function.
 *
 * @param param a formal parameter
 */
void vala_delegate_add_parameter (ValaDelegate* self, ValaFormalParameter* param) {
	g_return_if_fail (VALA_IS_DELEGATE (self));
	g_return_if_fail (param == NULL || VALA_IS_FORMAL_PARAMETER (param));
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (self)), vala_symbol_get_name (VALA_SYMBOL (param)), VALA_SYMBOL (param));
}


/**
 * Return copy of parameter list.
 *
 * @return parameter list
 */
GeeCollection* vala_delegate_get_parameters (ValaDelegate* self) {
	g_return_val_if_fail (VALA_IS_DELEGATE (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, GEE_COLLECTION (self->priv->parameters)));
}


/**
 * Checks whether the arguments and return type of the specified method
 * matches this callback.
 *
 * @param m a method
 * @return  true if the specified method is compatible to this callback
 */
gboolean vala_delegate_matches_method (ValaDelegate* self, ValaMethod* m) {
	GeeCollection* method_params;
	GeeIterator* method_params_it;
	gboolean first;
	gboolean _tmp3;
	g_return_val_if_fail (VALA_IS_DELEGATE (self), FALSE);
	g_return_val_if_fail (m == NULL || VALA_IS_METHOD (m), FALSE);
	if (!vala_data_type_stricter (vala_method_get_return_type (m), vala_delegate_get_return_type (self))) {
		return FALSE;
	}
	method_params = vala_method_get_parameters (m);
	method_params_it = gee_iterable_iterator (GEE_ITERABLE (method_params));
	first = TRUE;
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				ValaFormalParameter* method_param;
				/* use first callback parameter as instance parameter if
				 * an instance method is being compared to a static
				 * callback
				 */
				if (first && vala_method_get_instance (m) && !vala_delegate_get_instance (self)) {
					first = FALSE;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					continue;
				}
				/* method is allowed to accept less arguments */
				if (!gee_iterator_next (method_params_it)) {
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				method_param = gee_iterator_get (method_params_it);
				if (!vala_data_type_stricter (vala_formal_parameter_get_type_reference (param), vala_formal_parameter_get_type_reference (method_param))) {
					gboolean _tmp1;
					return (_tmp1 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp1);
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				(method_param == NULL ? NULL : (method_param = (g_object_unref (method_param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	/* method may not expect more arguments */
	if (gee_iterator_next (method_params_it)) {
		gboolean _tmp2;
		return (_tmp2 = FALSE, (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp2);
	}
	return (_tmp3 = TRUE, (method_params == NULL ? NULL : (method_params = (g_object_unref (method_params), NULL))), (method_params_it == NULL ? NULL : (method_params_it = (g_object_unref (method_params_it), NULL))), _tmp3);
}


static void vala_delegate_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_delegate (visitor, self);
}


static void vala_delegate_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_delegate_get_return_type (self)), visitor);
	{
		GeeList* param_collection;
		GeeIterator* param_it;
		param_collection = self->priv->parameters;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (param), visitor);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
}


static char* vala_delegate_real_get_cname (ValaTypesymbol* base, gboolean const_type) {
	ValaDelegate * self;
	const char* _tmp2;
	self = VALA_DELEGATE (base);
	if (self->priv->cname == NULL) {
		char* _tmp1;
		char* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cname = (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (VALA_SYMBOL (self)))), vala_symbol_get_name (VALA_SYMBOL (self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
		(_tmp0 = (g_free (_tmp0), NULL));
	}
	_tmp2 = NULL;
	return (_tmp2 = self->priv->cname, (_tmp2 == NULL ? NULL : g_strdup (_tmp2)));
}


/**
 * Sets the name of this callback as it is used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_delegate_set_cname (ValaDelegate* self, const char* cname) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (VALA_IS_DELEGATE (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->cname = (_tmp1 = (_tmp0 = cname, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (self->priv->cname = (g_free (self->priv->cname), NULL)), _tmp1);
}


static void vala_delegate_process_ccode_attribute (ValaDelegate* self, ValaAttribute* a) {
	g_return_if_fail (VALA_IS_DELEGATE (self));
	g_return_if_fail (a == NULL || VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* _tmp0;
		_tmp0 = NULL;
		vala_delegate_set_cname (self, (_tmp0 = vala_attribute_get_string (a, "cname")));
		(_tmp0 = (g_free (_tmp0), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_delegate_process_attributes (ValaDelegate* self) {
	g_return_if_fail (VALA_IS_DELEGATE (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = VALA_CODE_NODE (self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = a_it->data, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_delegate_process_ccode_attribute (self, a);
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_delegate_real_is_reference_type (ValaTypesymbol* base) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	return FALSE;
}


static char* vala_delegate_real_get_type_id (ValaTypesymbol* base) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	return g_strdup ("G_TYPE_POINTER");
}


static char* vala_delegate_real_get_marshaller_type_name (ValaTypesymbol* base) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	return g_strdup ("POINTER");
}


static char* vala_delegate_real_get_get_value_function (ValaTypesymbol* base) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	return g_strdup ("g_value_get_pointer");
}


static char* vala_delegate_real_get_set_value_function (ValaTypesymbol* base) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	return g_strdup ("g_value_set_pointer");
}


static void vala_delegate_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaDelegate * self;
	self = VALA_DELEGATE (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_delegate_get_return_type (self) == old_type) {
		vala_delegate_set_return_type (self, new_type);
	}
}


ValaDataType* vala_delegate_get_return_type (ValaDelegate* self) {
	g_return_val_if_fail (VALA_IS_DELEGATE (self), NULL);
	return self->priv->_return_type;
}


void vala_delegate_set_return_type (ValaDelegate* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_DELEGATE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_return_type), VALA_CODE_NODE (self));
}


gboolean vala_delegate_get_instance (ValaDelegate* self) {
	g_return_val_if_fail (VALA_IS_DELEGATE (self), FALSE);
	return self->priv->_instance;
}


void vala_delegate_set_instance (ValaDelegate* self, gboolean value) {
	g_return_if_fail (VALA_IS_DELEGATE (self));
	self->priv->_instance = value;
}


static void vala_delegate_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaDelegate * self;
	self = VALA_DELEGATE (object);
	switch (property_id) {
		case VALA_DELEGATE_RETURN_TYPE:
		g_value_set_object (value, vala_delegate_get_return_type (self));
		break;
		case VALA_DELEGATE_INSTANCE:
		g_value_set_boolean (value, vala_delegate_get_instance (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_delegate_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaDelegate * self;
	self = VALA_DELEGATE (object);
	switch (property_id) {
		case VALA_DELEGATE_RETURN_TYPE:
		vala_delegate_set_return_type (self, g_value_get_object (value));
		break;
		case VALA_DELEGATE_INSTANCE:
		vala_delegate_set_instance (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_delegate_class_init (ValaDelegateClass * klass) {
	vala_delegate_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaDelegatePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_delegate_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_delegate_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_delegate_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_delegate_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_delegate_real_accept_children;
	VALA_TYPESYMBOL_CLASS (klass)->get_cname = vala_delegate_real_get_cname;
	VALA_TYPESYMBOL_CLASS (klass)->is_reference_type = vala_delegate_real_is_reference_type;
	VALA_TYPESYMBOL_CLASS (klass)->get_type_id = vala_delegate_real_get_type_id;
	VALA_TYPESYMBOL_CLASS (klass)->get_marshaller_type_name = vala_delegate_real_get_marshaller_type_name;
	VALA_TYPESYMBOL_CLASS (klass)->get_get_value_function = vala_delegate_real_get_get_value_function;
	VALA_TYPESYMBOL_CLASS (klass)->get_set_value_function = vala_delegate_real_get_set_value_function;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_delegate_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DELEGATE_RETURN_TYPE, g_param_spec_object ("return-type", "return-type", "return-type", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DELEGATE_INSTANCE, g_param_spec_boolean ("instance", "instance", "instance", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_delegate_init (ValaDelegate * self) {
	self->priv = VALA_DELEGATE_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPEPARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	self->priv->parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_FORMAL_PARAMETER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_delegate_dispose (GObject * obj) {
	ValaDelegate * self;
	self = VALA_DELEGATE (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->_return_type == NULL ? NULL : (self->priv->_return_type = (g_object_unref (self->priv->_return_type), NULL)));
	G_OBJECT_CLASS (vala_delegate_parent_class)->dispose (obj);
}


GType vala_delegate_get_type (void) {
	static GType vala_delegate_type_id = 0;
	if (G_UNLIKELY (vala_delegate_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDelegateClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_delegate_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDelegate), 0, (GInstanceInitFunc) vala_delegate_init };
		vala_delegate_type_id = g_type_register_static (VALA_TYPE_TYPESYMBOL, "ValaDelegate", &g_define_type_info, 0);
	}
	return vala_delegate_type_id;
}




