/* valainterfacewriter.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include <vala/valainterfacewriter.h>
#include <stdio.h>
#include <gee/collection.h>
#include <vala/valasymbol.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefile.h>
#include <vala/valatypesymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valadatatype.h>
#include <vala/valaarray.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>

struct _ValaInterfaceWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	gboolean bol;
	char* current_cheader_filename;
};
#define VALA_INTERFACE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE_WRITER, ValaInterfaceWriterPrivate))
enum  {
	VALA_INTERFACE_WRITER_DUMMY_PROPERTY
};
static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_interface_writer_write_error_domains (ValaInterfaceWriter* self, GeeCollection* error_domains);
static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GeeCollection* params);
static void vala_interface_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb);
static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_interface_writer_write_indent (ValaInterfaceWriter* self);
static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_return_type (ValaInterfaceWriter* self, ValaDataType* type);
static void vala_interface_writer_write_type (ValaInterfaceWriter* self, ValaDataType* type);
static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_newline (ValaInterfaceWriter* self);
static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self);
static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self);
static gboolean vala_interface_writer_check_accessibility (ValaInterfaceWriter* self, ValaSymbol* sym);
static void vala_interface_writer_write_accessibility (ValaInterfaceWriter* self, ValaSymbol* sym);
static gpointer vala_interface_writer_parent_class = NULL;
static void vala_interface_writer_dispose (GObject * obj);


/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_interface_writer_write_file (ValaInterfaceWriter* self, ValaCodeContext* context, const char* filename) {
	ValaCodeContext* _tmp1;
	ValaCodeContext* _tmp0;
	FILE* _tmp2;
	char* _tmp4;
	char* _tmp3;
	FILE* _tmp5;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->context = (_tmp1 = (_tmp0 = context, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp1);
	_tmp2 = NULL;
	self->priv->stream = (_tmp2 = fopen (filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), _tmp2);
	_tmp4 = NULL;
	_tmp3 = NULL;
	vala_interface_writer_write_string (self, (_tmp4 = g_strdup_printf ("/* %s generated by %s, do not modify. */", (_tmp3 = g_path_get_basename (filename)), g_get_prgname ())));
	(_tmp4 = (g_free (_tmp4), NULL));
	(_tmp3 = (g_free (_tmp3), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_newline (self);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	_tmp5 = NULL;
	self->priv->stream = (_tmp5 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), _tmp5);
}


static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaInterfaceWriter * self;
	char* _tmp2;
	char* _tmp1;
	char* _tmp0;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	if (vala_namespace_get_pkg (ns)) {
		return;
	}
	if (vala_symbol_get_name (VALA_SYMBOL (ns)) == NULL) {
		vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
		return;
	}
	vala_interface_writer_write_indent (self);
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_interface_writer_write_string (self, (_tmp2 = g_strdup_printf ("[CCode (cprefix = \"%s\", lower_case_cprefix = \"%s\")]", (_tmp0 = vala_symbol_get_cprefix (VALA_SYMBOL (ns))), (_tmp1 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (ns))))));
	(_tmp2 = (g_free (_tmp2), NULL));
	(_tmp1 = (g_free (_tmp1), NULL));
	(_tmp0 = (g_free (_tmp0), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "namespace ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (ns)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp25;
	char* _tmp24;
	gboolean _tmp26;
	char* _tmp29;
	GeeCollection* type_params;
	GeeCollection* base_types;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (cl))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (cl));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					cheaders = (_tmp1 = (_tmp0 = cheader, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (cheaders = (g_free (cheaders), NULL)), _tmp1);
					first = FALSE;
				} else {
					char* _tmp2;
					_tmp2 = NULL;
					cheaders = (_tmp2 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), _tmp2);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	vala_interface_writer_write_string (self, "[CCode (");
	if (vala_typesymbol_is_reference_counting (VALA_TYPESYMBOL (cl))) {
		char* _tmp5;
		char* _tmp4;
		char* _tmp3;
		gboolean _tmp6;
		char* _tmp11;
		char* _tmp10;
		char* _tmp9;
		gboolean _tmp12;
		_tmp5 = NULL;
		_tmp4 = NULL;
		_tmp3 = NULL;
		if ((_tmp6 = vala_class_get_base_class (cl) == NULL || (_tmp3 = vala_typesymbol_get_ref_function (VALA_TYPESYMBOL (vala_class_get_base_class (cl)))) == NULL || g_utf8_collate ((_tmp4 = vala_typesymbol_get_ref_function (VALA_TYPESYMBOL (vala_class_get_base_class (cl)))), (_tmp5 = vala_typesymbol_get_ref_function (VALA_TYPESYMBOL (cl)))) != 0, (_tmp5 = (g_free (_tmp5), NULL)), (_tmp4 = (g_free (_tmp4), NULL)), (_tmp3 = (g_free (_tmp3), NULL)), _tmp6)) {
			char* _tmp8;
			char* _tmp7;
			_tmp8 = NULL;
			_tmp7 = NULL;
			vala_interface_writer_write_string (self, (_tmp8 = g_strdup_printf ("ref_function = \"%s\", ", (_tmp7 = vala_typesymbol_get_ref_function (VALA_TYPESYMBOL (cl))))));
			(_tmp8 = (g_free (_tmp8), NULL));
			(_tmp7 = (g_free (_tmp7), NULL));
		}
		_tmp11 = NULL;
		_tmp10 = NULL;
		_tmp9 = NULL;
		if ((_tmp12 = vala_class_get_base_class (cl) == NULL || (_tmp9 = vala_typesymbol_get_unref_function (VALA_TYPESYMBOL (vala_class_get_base_class (cl)))) == NULL || g_utf8_collate ((_tmp10 = vala_typesymbol_get_unref_function (VALA_TYPESYMBOL (vala_class_get_base_class (cl)))), (_tmp11 = vala_typesymbol_get_unref_function (VALA_TYPESYMBOL (cl)))) != 0, (_tmp11 = (g_free (_tmp11), NULL)), (_tmp10 = (g_free (_tmp10), NULL)), (_tmp9 = (g_free (_tmp9), NULL)), _tmp12)) {
			char* _tmp14;
			char* _tmp13;
			_tmp14 = NULL;
			_tmp13 = NULL;
			vala_interface_writer_write_string (self, (_tmp14 = g_strdup_printf ("unref_function = \"%s\", ", (_tmp13 = vala_typesymbol_get_unref_function (VALA_TYPESYMBOL (cl))))));
			(_tmp14 = (g_free (_tmp14), NULL));
			(_tmp13 = (g_free (_tmp13), NULL));
		}
	} else {
		char* _tmp15;
		gboolean _tmp16;
		char* _tmp20;
		char* _tmp19;
		gboolean _tmp21;
		_tmp15 = NULL;
		if ((_tmp16 = (_tmp15 = vala_typesymbol_get_dup_function (VALA_TYPESYMBOL (cl))) != NULL, (_tmp15 = (g_free (_tmp15), NULL)), _tmp16)) {
			char* _tmp18;
			char* _tmp17;
			_tmp18 = NULL;
			_tmp17 = NULL;
			vala_interface_writer_write_string (self, (_tmp18 = g_strdup_printf ("copy_function = \"%s\", ", (_tmp17 = vala_typesymbol_get_dup_function (VALA_TYPESYMBOL (cl))))));
			(_tmp18 = (g_free (_tmp18), NULL));
			(_tmp17 = (g_free (_tmp17), NULL));
		}
		_tmp20 = NULL;
		_tmp19 = NULL;
		if ((_tmp21 = g_utf8_collate ((_tmp19 = vala_typesymbol_get_free_function (VALA_TYPESYMBOL (cl))), (_tmp20 = vala_class_get_default_free_function (cl))) != 0, (_tmp20 = (g_free (_tmp20), NULL)), (_tmp19 = (g_free (_tmp19), NULL)), _tmp21)) {
			char* _tmp23;
			char* _tmp22;
			_tmp23 = NULL;
			_tmp22 = NULL;
			vala_interface_writer_write_string (self, (_tmp23 = g_strdup_printf ("free_function = \"%s\", ", (_tmp22 = vala_typesymbol_get_free_function (VALA_TYPESYMBOL (cl))))));
			(_tmp23 = (g_free (_tmp23), NULL));
			(_tmp22 = (g_free (_tmp22), NULL));
		}
	}
	_tmp25 = NULL;
	_tmp24 = NULL;
	if ((_tmp26 = g_utf8_collate ((_tmp24 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (cl), FALSE)), (_tmp25 = vala_class_get_default_cname (cl))) != 0, (_tmp25 = (g_free (_tmp25), NULL)), (_tmp24 = (g_free (_tmp24), NULL)), _tmp26)) {
		char* _tmp28;
		char* _tmp27;
		_tmp28 = NULL;
		_tmp27 = NULL;
		vala_interface_writer_write_string (self, (_tmp28 = g_strdup_printf ("cname = \"%s\", ", (_tmp27 = vala_typesymbol_get_cname (VALA_TYPESYMBOL (cl), FALSE)))));
		(_tmp28 = (g_free (_tmp28), NULL));
		(_tmp27 = (g_free (_tmp27), NULL));
	}
	_tmp29 = NULL;
	vala_interface_writer_write_string (self, (_tmp29 = g_strdup_printf ("cheader_filename = \"%s\")]", cheaders)));
	(_tmp29 = (g_free (_tmp29), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (cl));
	if (vala_class_get_is_static (cl)) {
		vala_interface_writer_write_string (self, "static ");
	} else {
		if (vala_class_get_is_abstract (cl)) {
			vala_interface_writer_write_string (self, "abstract ");
		}
	}
	vala_interface_writer_write_string (self, "class ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (cl)));
	type_params = vala_class_get_type_parameters (cl);
	if (gee_collection_get_size (type_params) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeCollection* type_param_collection;
			GeeIterator* type_param_it;
			type_param_collection = type_params;
			type_param_it = gee_iterable_iterator (GEE_ITERABLE (type_param_collection));
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = gee_iterator_get (type_param_it);
				{
					if (first) {
						first = FALSE;
					} else {
						vala_interface_writer_write_string (self, ",");
					}
					vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (type_param)));
					(type_param == NULL ? NULL : (type_param = (g_object_unref (type_param), NULL)));
				}
			}
			(type_param_it == NULL ? NULL : (type_param_it = (g_object_unref (type_param_it), NULL)));
		}
		vala_interface_writer_write_string (self, ">");
	}
	base_types = vala_class_get_base_types (cl);
	if (gee_collection_get_size (base_types) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = base_types;
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaDataType* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					vala_interface_writer_write_type (self, base_type);
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
	}
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
	(type_params == NULL ? NULL : (type_params = (g_object_unref (type_params), NULL)));
	(base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL)));
}


static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp3;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (st == NULL || VALA_IS_STRUCT (st));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (st)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (st))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (st))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (st));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					cheaders = (_tmp1 = (_tmp0 = cheader, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (cheaders = (g_free (cheaders), NULL)), _tmp1);
					first = FALSE;
				} else {
					char* _tmp2;
					_tmp2 = NULL;
					cheaders = (_tmp2 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), _tmp2);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	_tmp3 = NULL;
	vala_interface_writer_write_string (self, (_tmp3 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(_tmp3 = (g_free (_tmp3), NULL));
	vala_interface_writer_write_newline (self);
	if (vala_struct_is_simple_type (st)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[SimpleType]");
		vala_interface_writer_write_newline (self);
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (st));
	vala_interface_writer_write_string (self, "struct ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (st)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
}


static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp3;
	GeeCollection* type_params;
	GeeCollection* prerequisites;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (iface == NULL || VALA_IS_INTERFACE (iface));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (iface))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (iface))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (iface));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					cheaders = (_tmp1 = (_tmp0 = cheader, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (cheaders = (g_free (cheaders), NULL)), _tmp1);
					first = FALSE;
				} else {
					char* _tmp2;
					_tmp2 = NULL;
					cheaders = (_tmp2 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), _tmp2);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	_tmp3 = NULL;
	vala_interface_writer_write_string (self, (_tmp3 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(_tmp3 = (g_free (_tmp3), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (iface));
	vala_interface_writer_write_string (self, "interface ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (iface)));
	type_params = vala_interface_get_type_parameters (iface);
	if (gee_collection_get_size (type_params) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeCollection* type_param_collection;
			GeeIterator* type_param_it;
			type_param_collection = type_params;
			type_param_it = gee_iterable_iterator (GEE_ITERABLE (type_param_collection));
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = gee_iterator_get (type_param_it);
				{
					if (first) {
						first = FALSE;
					} else {
						vala_interface_writer_write_string (self, ",");
					}
					vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (type_param)));
					(type_param == NULL ? NULL : (type_param = (g_object_unref (type_param), NULL)));
				}
			}
			(type_param_it == NULL ? NULL : (type_param_it = (g_object_unref (type_param_it), NULL)));
		}
		vala_interface_writer_write_string (self, ">");
	}
	prerequisites = vala_interface_get_prerequisites (iface);
	if (gee_collection_get_size (prerequisites) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeCollection* prerequisite_collection;
			GeeIterator* prerequisite_it;
			prerequisite_collection = prerequisites;
			prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
			while (gee_iterator_next (prerequisite_it)) {
				ValaDataType* prerequisite;
				prerequisite = gee_iterator_get (prerequisite_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					vala_interface_writer_write_type (self, prerequisite);
					(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
				}
			}
			(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
		}
	}
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
	(type_params == NULL ? NULL : (type_params = (g_object_unref (type_params), NULL)));
	(prerequisites == NULL ? NULL : (prerequisites = (g_object_unref (prerequisites), NULL)));
}


static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* _tmp4;
	char* _tmp3;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (en == NULL || VALA_IS_ENUM (en));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (en)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (en))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (en))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (en));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					cheaders = (_tmp1 = (_tmp0 = cheader, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (cheaders = (g_free (cheaders), NULL)), _tmp1);
					first = FALSE;
				} else {
					char* _tmp2;
					_tmp2 = NULL;
					cheaders = (_tmp2 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), _tmp2);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	vala_interface_writer_write_string (self, (_tmp4 = g_strdup_printf ("[CCode (cprefix = \"%s\", cheader_filename = \"%s\")]", (_tmp3 = vala_enum_get_cprefix (en)), cheaders)));
	(_tmp4 = (g_free (_tmp4), NULL));
	(_tmp3 = (g_free (_tmp3), NULL));
	if (vala_enum_get_is_flags (en)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[Flags]");
	}
	if (vala_enum_get_error_domain (en)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[ErrorDomain]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (en));
	vala_interface_writer_write_string (self, "enum ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (en)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
}


static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev) {
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (ev == NULL || VALA_IS_ENUM_VALUE (ev));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (ev)));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (c == NULL || VALA_IS_CONSTANT (c));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (c)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (c))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (c))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (c));
	vala_interface_writer_write_string (self, "const ");
	vala_interface_writer_write_type (self, vala_constant_get_type_reference (c));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (c)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaInterfaceWriter * self;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (f == NULL || VALA_IS_FIELD (f));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (f)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (f))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (f))) {
		return;
	}
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = g_utf8_collate ((_tmp0 = vala_field_get_cname (f)), (_tmp1 = vala_field_get_default_cname (f))) != 0, (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2)) {
		char* _tmp4;
		char* _tmp3;
		vala_interface_writer_write_indent (self);
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_interface_writer_write_string (self, (_tmp4 = g_strdup_printf ("[CCode (cname = \"%s\")]", (_tmp3 = vala_field_get_cname (f)))));
		(_tmp4 = (g_free (_tmp4), NULL));
		(_tmp3 = (g_free (_tmp3), NULL));
	}
	if (vala_field_get_no_array_length (f) && VALA_IS_ARRAY (vala_data_type_get_data_type (vala_field_get_type_reference (f)))) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[NoArrayLength]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (f));
	if (vala_data_type_get_data_type (vala_field_get_type_reference (f)) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_field_get_type_reference (f))) && !vala_data_type_get_takes_ownership (vala_field_get_type_reference (f))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	vala_interface_writer_write_type (self, vala_field_get_type_reference (f));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (f)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_write_error_domains (ValaInterfaceWriter* self, GeeCollection* error_domains) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (error_domains == NULL || GEE_IS_COLLECTION (error_domains));
	if (gee_collection_get_size (error_domains) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, " throws ");
		first = TRUE;
		{
			GeeCollection* type_collection;
			GeeIterator* type_it;
			type_collection = error_domains;
			type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
			while (gee_iterator_next (type_it)) {
				ValaDataType* type;
				type = gee_iterator_get (type_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					vala_interface_writer_write_type (self, type);
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				}
			}
			(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
		}
	}
}


static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GeeCollection* params) {
	gboolean first;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (params == NULL || GEE_IS_COLLECTION (params));
	vala_interface_writer_write_string (self, "(");
	first = TRUE;
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (!first) {
					vala_interface_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				if (vala_formal_parameter_get_ellipsis (param)) {
					vala_interface_writer_write_string (self, "...");
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					continue;
				}
				if (vala_data_type_get_is_ref (vala_formal_parameter_get_type_reference (param)) || vala_data_type_get_is_out (vala_formal_parameter_get_type_reference (param))) {
					if (vala_data_type_get_is_ref (vala_formal_parameter_get_type_reference (param))) {
						vala_interface_writer_write_string (self, "ref ");
					} else {
						if (vala_data_type_get_is_out (vala_formal_parameter_get_type_reference (param))) {
							vala_interface_writer_write_string (self, "out ");
						}
					}
					if (vala_data_type_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (vala_formal_parameter_get_type_reference (param))) && !vala_data_type_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
						vala_interface_writer_write_string (self, "weak ");
					}
				}
				vala_interface_writer_write_type (self, vala_formal_parameter_get_type_reference (param));
				if (vala_data_type_get_transfers_ownership (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "#");
				}
				vala_interface_writer_write_string (self, " ");
				vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (param)));
				if (vala_formal_parameter_get_default_expression (param) != NULL) {
					char* _tmp0;
					vala_interface_writer_write_string (self, " = ");
					_tmp0 = NULL;
					vala_interface_writer_write_string (self, (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_default_expression (param)))));
					(_tmp0 = (g_free (_tmp0), NULL));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	vala_interface_writer_write_string (self, ")");
}


static void vala_interface_writer_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* cb) {
	ValaInterfaceWriter * self;
	GeeCollection* _tmp0;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (cb == NULL || VALA_IS_DELEGATE (cb));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (cb)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cb))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (cb))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (cb));
	vala_interface_writer_write_string (self, "static delegate ");
	vala_interface_writer_write_return_type (self, vala_delegate_get_return_type (cb));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (cb)));
	vala_interface_writer_write_string (self, " ");
	_tmp0 = NULL;
	vala_interface_writer_write_params (self, (_tmp0 = vala_delegate_get_parameters (cb)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaInterfaceWriter * self;
	GString* ccode_params;
	char* separator;
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	GeeCollection* _tmp8;
	GeeCollection* _tmp9;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (m == NULL || VALA_IS_METHOD (m));
	if (vala_code_node_get_source_reference (VALA_CODE_NODE (m)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m))))) {
		return;
	}
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (m)) || vala_method_get_overrides (m) || vala_method_get_base_interface_method (m) != NULL) {
		return;
	}
	if (vala_method_get_no_array_length (m)) {
		gboolean array_found;
		array_found = (vala_method_get_return_type (m) != NULL && VALA_IS_ARRAY (vala_data_type_get_data_type (vala_method_get_return_type (m))));
		{
			GeeCollection* param_collection;
			GeeIterator* param_it;
			param_collection = vala_method_get_parameters (m);
			param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = gee_iterator_get (param_it);
				{
					if (vala_formal_parameter_get_type_reference (param) != NULL && VALA_IS_ARRAY (vala_data_type_get_data_type (vala_formal_parameter_get_type_reference (param)))) {
						array_found = TRUE;
						(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
						break;
					}
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				}
			}
			(param_collection == NULL ? NULL : (param_collection = (g_object_unref (param_collection), NULL)));
			(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
		}
		if (array_found) {
			vala_interface_writer_write_indent (self);
			vala_interface_writer_write_string (self, "[NoArrayLength]");
		}
	}
	if (vala_method_get_instance_last (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceLast]");
	}
	ccode_params = g_string_new ("");
	separator = g_strdup ("");
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = g_utf8_collate ((_tmp0 = vala_method_get_cname (m)), (_tmp1 = vala_method_get_default_cname (m))) != 0, (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2)) {
		char* _tmp3;
		char* _tmp4;
		_tmp3 = NULL;
		g_string_append_printf (ccode_params, "%scname = \"%s\"", separator, (_tmp3 = vala_method_get_cname (m)));
		(_tmp3 = (g_free (_tmp3), NULL));
		_tmp4 = NULL;
		separator = (_tmp4 = g_strdup (", "), (separator = (g_free (separator), NULL)), _tmp4);
	}
	if (g_utf8_collate (vala_method_get_sentinel (m), VALA_METHOD_DEFAULT_SENTINEL) != 0) {
		char* _tmp5;
		g_string_append_printf (ccode_params, "%ssentinel = \"%s\"", separator, vala_method_get_sentinel (m));
		_tmp5 = NULL;
		separator = (_tmp5 = g_strdup (", "), (separator = (g_free (separator), NULL)), _tmp5);
	}
	if (ccode_params->len > 0) {
		char* _tmp6;
		vala_interface_writer_write_indent (self);
		_tmp6 = NULL;
		vala_interface_writer_write_string (self, (_tmp6 = g_strdup_printf ("[CCode (%s)]", ccode_params->str)));
		(_tmp6 = (g_free (_tmp6), NULL));
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (m));
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaTypesymbol* _tmp7;
		ValaTypesymbol* datatype;
		_tmp7 = NULL;
		datatype = (_tmp7 = VALA_TYPESYMBOL (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)));
		vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (datatype)));
		vala_interface_writer_write_identifier (self, g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (m)), g_utf8_strlen (".new", -1)));
		vala_interface_writer_write_string (self, " ");
		(datatype == NULL ? NULL : (datatype = (g_object_unref (datatype), NULL)));
	} else {
		if (!vala_method_get_instance (m)) {
			vala_interface_writer_write_string (self, "static ");
		} else {
			if (vala_method_get_is_abstract (m)) {
				vala_interface_writer_write_string (self, "abstract ");
			} else {
				if (vala_method_get_is_virtual (m)) {
					vala_interface_writer_write_string (self, "virtual ");
				}
			}
		}
	}
	if (!(VALA_IS_CREATION_METHOD (m))) {
		vala_interface_writer_write_return_type (self, vala_method_get_return_type (m));
		vala_interface_writer_write_string (self, " ");
		vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (m)));
		vala_interface_writer_write_string (self, " ");
	}
	_tmp8 = NULL;
	vala_interface_writer_write_params (self, (_tmp8 = vala_method_get_parameters (m)));
	(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
	_tmp9 = NULL;
	vala_interface_writer_write_error_domains (self, (_tmp9 = vala_method_get_error_domains (m)));
	(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(ccode_params == NULL ? NULL : (ccode_params = (g_string_free (ccode_params, TRUE), NULL)));
	(separator = (g_free (separator), NULL));
}


static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (m == NULL || VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (prop == NULL || VALA_IS_PROPERTY (prop));
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (prop))) {
		return;
	}
	if (vala_property_get_no_accessor_method (prop)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[NoAccessorMethod]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (prop));
	if (vala_property_get_is_abstract (prop)) {
		vala_interface_writer_write_string (self, "abstract ");
	} else {
		if (vala_property_get_is_virtual (prop)) {
			vala_interface_writer_write_string (self, "virtual ");
		}
	}
	if (!vala_data_type_get_takes_ownership (vala_property_get_type_reference (prop))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	vala_interface_writer_write_type (self, vala_property_get_type_reference (prop));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (prop)));
	vala_interface_writer_write_string (self, " {");
	if (vala_property_get_get_accessor (prop) != NULL) {
		vala_interface_writer_write_string (self, " get;");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " set");
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " construct");
		}
		vala_interface_writer_write_string (self, ";");
	}
	vala_interface_writer_write_string (self, " }");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaInterfaceWriter * self;
	GeeCollection* _tmp0;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (sig == NULL || VALA_IS_SIGNAL (sig));
	if (!vala_interface_writer_check_accessibility (self, VALA_SYMBOL (sig))) {
		return;
	}
	if (vala_signal_get_has_emitter (sig)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[HasEmitter]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_accessibility (self, VALA_SYMBOL (sig));
	vala_interface_writer_write_string (self, "signal ");
	vala_interface_writer_write_return_type (self, vala_signal_get_return_type (sig));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (VALA_SYMBOL (sig)));
	vala_interface_writer_write_string (self, " ");
	_tmp0 = NULL;
	vala_interface_writer_write_params (self, (_tmp0 = vala_signal_get_parameters (sig)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_write_indent (ValaInterfaceWriter* self) {
	gint i;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	i = 0;
	if (!self->priv->bol) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	if (g_utf8_collate (s, "base") == 0 || g_utf8_collate (s, "break") == 0 || g_utf8_collate (s, "class") == 0 || g_utf8_collate (s, "construct") == 0 || g_utf8_collate (s, "delegate") == 0 || g_utf8_collate (s, "do") == 0 || g_utf8_collate (s, "foreach") == 0 || g_utf8_collate (s, "in") == 0 || g_utf8_collate (s, "interface") == 0 || g_utf8_collate (s, "lock") == 0 || g_utf8_collate (s, "namespace") == 0 || g_utf8_collate (s, "new") == 0 || g_utf8_collate (s, "out") == 0 || g_utf8_collate (s, "ref") == 0 || g_utf8_collate (s, "signal") == 0) {
		fputc ('@', self->priv->stream);
	}
	vala_interface_writer_write_string (self, s);
}


static void vala_interface_writer_write_return_type (ValaInterfaceWriter* self, ValaDataType* type) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (type == NULL || VALA_IS_DATA_TYPE (type));
	if ((vala_data_type_get_data_type (type) != NULL && vala_typesymbol_is_reference_type (vala_data_type_get_data_type (type))) || vala_data_type_get_type_parameter (type) != NULL) {
		if (!vala_data_type_get_transfers_ownership (type)) {
			vala_interface_writer_write_string (self, "weak ");
		}
	}
	vala_interface_writer_write_type (self, type);
}


static void vala_interface_writer_write_type (ValaInterfaceWriter* self, ValaDataType* type) {
	char* _tmp0;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (type == NULL || VALA_IS_DATA_TYPE (type));
	_tmp0 = NULL;
	vala_interface_writer_write_string (self, (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (type))));
	(_tmp0 = (g_free (_tmp0), NULL));
}


static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	fprintf (self->priv->stream, "%s", s);
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_newline (ValaInterfaceWriter* self) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	fputc ('\n', self->priv->stream);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	if (!self->priv->bol) {
		fputc (' ', self->priv->stream);
	} else {
		vala_interface_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_interface_writer_write_newline (self);
	self->priv->indent++;
}


static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	self->priv->indent--;
	vala_interface_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


static gboolean vala_interface_writer_check_accessibility (ValaInterfaceWriter* self, ValaSymbol* sym) {
	g_return_val_if_fail (VALA_IS_INTERFACE_WRITER (self), FALSE);
	g_return_val_if_fail (sym == NULL || VALA_IS_SYMBOL (sym), FALSE);
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC || vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
		return TRUE;
	}
	return FALSE;
}


static void vala_interface_writer_write_accessibility (ValaInterfaceWriter* self, ValaSymbol* sym) {
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (sym == NULL || VALA_IS_SYMBOL (sym));
	if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
		vala_interface_writer_write_string (self, "public ");
	} else {
		if (vala_symbol_get_access (sym) == VALA_SYMBOL_ACCESSIBILITY_PROTECTED) {
			vala_interface_writer_write_string (self, "protected ");
		} else {
			g_assert_not_reached ();
		}
	}
}


/**
 * Code visitor generating Vala API file for the public interface.
 */
ValaInterfaceWriter* vala_interface_writer_new (void) {
	ValaInterfaceWriter * self;
	self = g_object_newv (VALA_TYPE_INTERFACE_WRITER, 0, NULL);
	return self;
}


static void vala_interface_writer_class_init (ValaInterfaceWriterClass * klass) {
	vala_interface_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfaceWriterPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_interface_writer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_interface_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_interface_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_interface_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_interface_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_interface_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_interface_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_interface_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_interface_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_interface_writer_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_interface_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_interface_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_interface_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_interface_writer_real_visit_signal;
}


static void vala_interface_writer_init (ValaInterfaceWriter * self) {
	self->priv = VALA_INTERFACE_WRITER_GET_PRIVATE (self);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_dispose (GObject * obj) {
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	(self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL));
	G_OBJECT_CLASS (vala_interface_writer_parent_class)->dispose (obj);
}


GType vala_interface_writer_get_type (void) {
	static GType vala_interface_writer_type_id = 0;
	if (G_UNLIKELY (vala_interface_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterfaceWriter), 0, (GInstanceInitFunc) vala_interface_writer_init };
		vala_interface_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaInterfaceWriter", &g_define_type_info, 0);
	}
	return vala_interface_writer_type_id;
}




