/* valainvocationexpression.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valainvocationexpression.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valacodegenerator.h>




struct _ValaInvocationExpressionPrivate {
	ValaCCodeExpression* _delegate_target;
	GeeList* argument_list;
	GeeList* array_sizes;
};

#define VALA_INVOCATION_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INVOCATION_EXPRESSION, ValaInvocationExpressionPrivate))
enum  {
	VALA_INVOCATION_EXPRESSION_DUMMY_PROPERTY,
	VALA_INVOCATION_EXPRESSION_CALL,
	VALA_INVOCATION_EXPRESSION_DELEGATE_TARGET
};
static void vala_invocation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_invocation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_invocation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_invocation_expression_real_is_pure (ValaExpression* base);
static ValaCodeBinding* vala_invocation_expression_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen);
static gpointer vala_invocation_expression_parent_class = NULL;
static void vala_invocation_expression_finalize (GObject * obj);



/**
 * Creates a new invocation expression.
 *
 * @param call             method to call
 * @param source_reference reference to source code
 * @return                 newly created invocation expression
 */
ValaInvocationExpression* vala_invocation_expression_new (ValaExpression* call, ValaSourceReference* source_reference) {
	GParameter * __params;
	GParameter * __params_it;
	ValaInvocationExpression * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (call), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "call";
	g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION);
	g_value_set_object (&__params_it->value, call);
	__params_it++;
	self = g_object_newv (VALA_TYPE_INVOCATION_EXPRESSION, __params_it - __params, __params);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


/**
 * Appends the specified expression to the list of arguments.
 *
 * @param arg an argument
 */
void vala_invocation_expression_add_argument (ValaInvocationExpression* self, ValaExpression* arg) {
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_EXPRESSION (arg));
	gee_collection_add (GEE_COLLECTION (self->priv->argument_list), arg);
	vala_code_node_set_parent_node (VALA_CODE_NODE (arg), VALA_CODE_NODE (self));
}


/**
 * Returns a copy of the argument list.
 *
 * @return argument list
 */
GeeList* vala_invocation_expression_get_argument_list (ValaInvocationExpression* self) {
	g_return_val_if_fail (VALA_IS_INVOCATION_EXPRESSION (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->argument_list));
}


/**
 * Add an array size C code expression.
 */
void vala_invocation_expression_append_array_size (ValaInvocationExpression* self, ValaCCodeExpression* size) {
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (self));
	g_return_if_fail (VALA_IS_CCODE_EXPRESSION (size));
	gee_collection_add (GEE_COLLECTION (self->priv->array_sizes), size);
}


/**
 * Get the C code expression for array sizes for all dimensions
 * ascending from left to right.
 */
GeeList* vala_invocation_expression_get_array_sizes (ValaInvocationExpression* self) {
	g_return_val_if_fail (VALA_IS_INVOCATION_EXPRESSION (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_CCODE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->array_sizes));
}


static void vala_invocation_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_invocation_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, VALA_EXPRESSION (self));
}


static void vala_invocation_expression_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_invocation_expression_get_call (self)), visitor);
	{
		GeeList* expr_collection;
		int expr_it;
		expr_collection = self->priv->argument_list;
		for (expr_it = 0; expr_it < gee_collection_get_size (GEE_COLLECTION (expr_collection)); expr_it = expr_it + 1) {
			ValaExpression* expr;
			expr = ((ValaExpression*) (gee_list_get (GEE_LIST (expr_collection), expr_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (expr), visitor);
				(expr == NULL ? NULL : (expr = (g_object_unref (expr), NULL)));
			}
		}
	}
}


static void vala_invocation_expression_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaInvocationExpression * self;
	gint index;
	self = VALA_INVOCATION_EXPRESSION (base);
	g_return_if_fail (VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (VALA_IS_EXPRESSION (new_node));
	if (vala_invocation_expression_get_call (self) == old_node) {
		vala_invocation_expression_set_call (self, new_node);
	}
	index = gee_list_index_of (self->priv->argument_list, old_node);
	if (index >= 0 && vala_code_node_get_parent_node (VALA_CODE_NODE (new_node)) == NULL) {
		gee_list_set (((GeeList*) (self->priv->argument_list)), index, new_node);
		vala_code_node_set_parent_node (VALA_CODE_NODE (new_node), VALA_CODE_NODE (self));
	}
}


static gboolean vala_invocation_expression_real_is_pure (ValaExpression* base) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (base);
	return FALSE;
}


static ValaCodeBinding* vala_invocation_expression_real_create_code_binding (ValaCodeNode* base, ValaCodeGenerator* codegen) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (base);
	g_return_val_if_fail (VALA_IS_CODE_GENERATOR (codegen), NULL);
	return vala_code_generator_create_invocation_expression_binding (codegen, self);
}


ValaExpression* vala_invocation_expression_get_call (ValaInvocationExpression* self) {
	g_return_val_if_fail (VALA_IS_INVOCATION_EXPRESSION (self), NULL);
	return self->_call;
}


void vala_invocation_expression_set_call (ValaInvocationExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->_call = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->_call == NULL ? NULL : (self->_call = (g_object_unref (self->_call), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->_call), VALA_CODE_NODE (self));
	g_object_notify (((GObject *) (self)), "call");
}


ValaCCodeExpression* vala_invocation_expression_get_delegate_target (ValaInvocationExpression* self) {
	g_return_val_if_fail (VALA_IS_INVOCATION_EXPRESSION (self), NULL);
	return self->priv->_delegate_target;
}


void vala_invocation_expression_set_delegate_target (ValaInvocationExpression* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2;
	ValaCCodeExpression* _tmp1;
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_delegate_target = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_delegate_target == NULL ? NULL : (self->priv->_delegate_target = (g_object_unref (self->priv->_delegate_target), NULL))), _tmp2);
	g_object_notify (((GObject *) (self)), "delegate-target");
}


static void vala_invocation_expression_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (object);
	switch (property_id) {
		case VALA_INVOCATION_EXPRESSION_CALL:
		g_value_set_object (value, vala_invocation_expression_get_call (self));
		break;
		case VALA_INVOCATION_EXPRESSION_DELEGATE_TARGET:
		g_value_set_object (value, vala_invocation_expression_get_delegate_target (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_invocation_expression_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (object);
	switch (property_id) {
		case VALA_INVOCATION_EXPRESSION_CALL:
		vala_invocation_expression_set_call (self, g_value_get_object (value));
		break;
		case VALA_INVOCATION_EXPRESSION_DELEGATE_TARGET:
		vala_invocation_expression_set_delegate_target (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_invocation_expression_class_init (ValaInvocationExpressionClass * klass) {
	vala_invocation_expression_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInvocationExpressionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_invocation_expression_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_invocation_expression_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_invocation_expression_finalize;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_invocation_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_invocation_expression_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_invocation_expression_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_invocation_expression_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->create_code_binding = vala_invocation_expression_real_create_code_binding;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INVOCATION_EXPRESSION_CALL, g_param_spec_object ("call", "call", "call", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INVOCATION_EXPRESSION_DELEGATE_TARGET, g_param_spec_object ("delegate-target", "delegate-target", "delegate-target", VALA_TYPE_CCODE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_invocation_expression_instance_init (ValaInvocationExpression * self) {
	self->priv = VALA_INVOCATION_EXPRESSION_GET_PRIVATE (self);
	self->priv->argument_list = GEE_LIST (gee_array_list_new (VALA_TYPE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
	self->priv->array_sizes = GEE_LIST (gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_invocation_expression_finalize (GObject * obj) {
	ValaInvocationExpression * self;
	self = VALA_INVOCATION_EXPRESSION (obj);
	(self->priv->_delegate_target == NULL ? NULL : (self->priv->_delegate_target = (g_object_unref (self->priv->_delegate_target), NULL)));
	(self->_call == NULL ? NULL : (self->_call = (g_object_unref (self->_call), NULL)));
	(self->priv->argument_list == NULL ? NULL : (self->priv->argument_list = (g_object_unref (self->priv->argument_list), NULL)));
	(self->priv->array_sizes == NULL ? NULL : (self->priv->array_sizes = (g_object_unref (self->priv->array_sizes), NULL)));
	G_OBJECT_CLASS (vala_invocation_expression_parent_class)->finalize (obj);
}


GType vala_invocation_expression_get_type (void) {
	static GType vala_invocation_expression_type_id = 0;
	if (vala_invocation_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInvocationExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_invocation_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInvocationExpression), 0, (GInstanceInitFunc) vala_invocation_expression_instance_init };
		vala_invocation_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInvocationExpression", &g_define_type_info, 0);
	}
	return vala_invocation_expression_type_id;
}




