/* hashmap.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <gee/hashmap.h>

typedef struct _GeeHashMapNode GeeHashMapNode;

#define GEE_HASH_MAP_TYPE_KEY_SET (gee_hash_map_key_set_get_type ())
#define GEE_HASH_MAP_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySet))
#define GEE_HASH_MAP_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))
#define GEE_HASH_MAP_IS_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_IS_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_KEY_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))

typedef struct _GeeHashMapKeySet GeeHashMapKeySet;
typedef struct _GeeHashMapKeySetClass GeeHashMapKeySetClass;
typedef struct _GeeHashMapKeySetPrivate GeeHashMapKeySetPrivate;

#define GEE_HASH_MAP_TYPE_KEY_ITERATOR (gee_hash_map_key_iterator_get_type ())
#define GEE_HASH_MAP_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIterator))
#define GEE_HASH_MAP_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))
#define GEE_HASH_MAP_IS_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_IS_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_KEY_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))

typedef struct _GeeHashMapKeyIterator GeeHashMapKeyIterator;
typedef struct _GeeHashMapKeyIteratorClass GeeHashMapKeyIteratorClass;
typedef struct _GeeHashMapKeyIteratorPrivate GeeHashMapKeyIteratorPrivate;

#define GEE_HASH_MAP_TYPE_VALUE_COLLECTION (gee_hash_map_value_collection_get_type ())
#define GEE_HASH_MAP_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollection))
#define GEE_HASH_MAP_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))

typedef struct _GeeHashMapValueCollection GeeHashMapValueCollection;
typedef struct _GeeHashMapValueCollectionClass GeeHashMapValueCollectionClass;
typedef struct _GeeHashMapValueCollectionPrivate GeeHashMapValueCollectionPrivate;

#define GEE_HASH_MAP_TYPE_VALUE_ITERATOR (gee_hash_map_value_iterator_get_type ())
#define GEE_HASH_MAP_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIterator))
#define GEE_HASH_MAP_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))

typedef struct _GeeHashMapValueIterator GeeHashMapValueIterator;
typedef struct _GeeHashMapValueIteratorClass GeeHashMapValueIteratorClass;
typedef struct _GeeHashMapValueIteratorPrivate GeeHashMapValueIteratorPrivate;

struct _GeeHashMapNode {
	gpointer key;
	gpointer value;
	GeeHashMapNode* next;
	guint key_hash;
};

struct _GeeHashMapKeySet {
	GeeCollectionObject parent_instance;
	GeeHashMapKeySetPrivate * priv;
};

struct _GeeHashMapKeySetClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapKeyIterator {
	GeeCollectionObject parent_instance;
	GeeHashMapKeyIteratorPrivate * priv;
};

struct _GeeHashMapKeyIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapValueCollection {
	GeeCollectionObject parent_instance;
	GeeHashMapValueCollectionPrivate * priv;
};

struct _GeeHashMapValueCollectionClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapValueIterator {
	GeeCollectionObject parent_instance;
	GeeHashMapValueIteratorPrivate * priv;
};

struct _GeeHashMapValueIteratorClass {
	GeeCollectionObjectClass parent_class;
};



struct _GeeHashMapPrivate {
	gint _array_size;
	gint _nnodes;
	GeeHashMapNode** _nodes;
	gint _nodes_length1;
	gint _nodes_size;
	gint _stamp;
	GHashFunc _key_hash_func;
	GEqualFunc _key_equal_func;
	GEqualFunc _value_equal_func;
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};

#define GEE_HASH_MAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_MAP, GeeHashMapPrivate))
enum  {
	GEE_HASH_MAP_DUMMY_PROPERTY
};
#define GEE_HASH_MAP_MIN_SIZE 11
#define GEE_HASH_MAP_MAX_SIZE 13845163
static GeeSet* gee_hash_map_real_get_keys (GeeMap* base);
static GeeCollection* gee_hash_map_real_get_values (GeeMap* base);
static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key);
static gboolean gee_hash_map_real_contains (GeeMap* base, gconstpointer key);
static gpointer gee_hash_map_real_get (GeeMap* base, gconstpointer key);
static void gee_hash_map_real_set (GeeMap* base, gconstpointer key, gconstpointer value);
static gboolean gee_hash_map_real_remove (GeeMap* base, gconstpointer key);
static void gee_hash_map_real_clear (GeeMap* base);
static void gee_hash_map_resize (GeeHashMap* self);
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static void gee_hash_map_node_instance_init (GeeHashMapNode * self);
static void gee_hash_map_node_free (GeeHashMapNode* self);
struct _GeeHashMapKeySetPrivate {
	GeeHashMap* _map;
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};

#define GEE_HASH_MAP_KEY_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetPrivate))
enum  {
	GEE_HASH_MAP_KEY_SET_DUMMY_PROPERTY
};
static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_key_set_real_get_element_type (GeeIterable* base);
static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gconstpointer key);
static void gee_hash_map_key_set_real_clear (GeeCollection* base);
static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gconstpointer key);
static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gconstpointer key);
static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value);
static gpointer gee_hash_map_key_set_parent_class = NULL;
static GeeIterableIface* gee_hash_map_key_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_key_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_map_key_set_gee_set_parent_iface = NULL;
static void gee_hash_map_key_set_finalize (GeeCollectionObject* obj);
static GType gee_hash_map_key_set_get_type (void);
struct _GeeHashMapKeyIteratorPrivate {
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};

#define GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorPrivate))
enum  {
	GEE_HASH_MAP_KEY_ITERATOR_DUMMY_PROPERTY
};
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base);
static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value);
static gpointer gee_hash_map_key_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_key_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_map_key_iterator_finalize (GeeCollectionObject* obj);
static GType gee_hash_map_key_iterator_get_type (void);
struct _GeeHashMapValueCollectionPrivate {
	GeeHashMap* _map;
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};

#define GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionPrivate))
enum  {
	GEE_HASH_MAP_VALUE_COLLECTION_DUMMY_PROPERTY
};
static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_value_collection_real_get_element_type (GeeIterable* base);
static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gconstpointer value);
static void gee_hash_map_value_collection_real_clear (GeeCollection* base);
static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gconstpointer value);
static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gconstpointer value);
static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value);
static gpointer gee_hash_map_value_collection_parent_class = NULL;
static GeeIterableIface* gee_hash_map_value_collection_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_value_collection_gee_collection_parent_iface = NULL;
static void gee_hash_map_value_collection_finalize (GeeCollectionObject* obj);
static GType gee_hash_map_value_collection_get_type (void);
struct _GeeHashMapValueIteratorPrivate {
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
};

#define GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorPrivate))
enum  {
	GEE_HASH_MAP_VALUE_ITERATOR_DUMMY_PROPERTY
};
static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base);
static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value);
static gpointer gee_hash_map_value_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_value_iterator_gee_iterator_parent_iface = NULL;
static void gee_hash_map_value_iterator_finalize (GeeCollectionObject* obj);
static GType gee_hash_map_value_iterator_get_type (void);
static gpointer gee_hash_map_parent_class = NULL;
static GeeMapIface* gee_hash_map_gee_map_parent_iface = NULL;
static void gee_hash_map_finalize (GeeCollectionObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



GeeHashMap* gee_hash_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	GeeHashMap* self;
	GeeHashMapNode** _tmp0;
	self = (GeeHashMap*) g_type_create_instance (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_set_key_hash_func (self, key_hash_func);
	gee_hash_map_set_key_equal_func (self, key_equal_func);
	gee_hash_map_set_value_equal_func (self, value_equal_func);
	self->priv->_array_size = GEE_HASH_MAP_MIN_SIZE;
	_tmp0 = NULL;
	self->priv->_nodes = (_tmp0 = g_new0 (GeeHashMapNode*, self->priv->_array_size + 1), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = self->priv->_array_size, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp0);
	return self;
}


GeeHashMap* gee_hash_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	return gee_hash_map_construct (GEE_TYPE_HASH_MAP, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, key_hash_func, key_equal_func, value_equal_func);
}


static GeeSet* gee_hash_map_real_get_keys (GeeMap* base) {
	GeeHashMap * self;
	self = (GeeHashMap*) base;
	return (GeeSet*) gee_hash_map_key_set_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
}


static GeeCollection* gee_hash_map_real_get_values (GeeMap* base) {
	GeeHashMap * self;
	self = (GeeHashMap*) base;
	return (GeeCollection*) gee_hash_map_value_collection_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
}


static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key) {
	guint hash_value;
	GeeHashMapNode** node;
	g_return_val_if_fail (self != NULL, NULL);
	hash_value = self->priv->_key_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if ((*node) != NULL) {
			gboolean _tmp1;
			_tmp1 = FALSE;
			if (hash_value != (*node)->key_hash) {
				_tmp1 = TRUE;
			} else {
				_tmp1 = !self->priv->_key_equal_func ((*node)->key, key);
			}
			_tmp0 = _tmp1;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		node = &(*node)->next;
	}
	return node;
}


static gboolean gee_hash_map_real_contains (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	return (*node) != NULL;
}


static gpointer gee_hash_map_real_get (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	GeeHashMapNode* node;
	self = (GeeHashMap*) base;
	node = *gee_hash_map_lookup_node (self, key);
	if (node != NULL) {
		gconstpointer _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = node->value, ((_tmp0 == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0) : self->priv->v_dup_func ((gpointer) _tmp0));
	} else {
		return NULL;
	}
}


static void gee_hash_map_real_set (GeeMap* base, gconstpointer key, gconstpointer value) {
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		gpointer _tmp1;
		gconstpointer _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		(*node)->value = (_tmp1 = (_tmp0 = value, ((_tmp0 == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0) : self->priv->v_dup_func ((gpointer) _tmp0)), (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp1);
	} else {
		guint hash_value;
		gconstpointer _tmp3;
		gconstpointer _tmp2;
		hash_value = self->priv->_key_hash_func (key);
		_tmp3 = NULL;
		_tmp2 = NULL;
		*node = gee_hash_map_node_new ((_tmp2 = key, ((_tmp2 == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp2) : self->priv->k_dup_func ((gpointer) _tmp2)), (_tmp3 = value, ((_tmp3 == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp3) : self->priv->v_dup_func ((gpointer) _tmp3)), hash_value);
		self->priv->_nnodes++;
		gee_hash_map_resize (self);
	}
	self->priv->_stamp++;
}


static gboolean gee_hash_map_real_remove (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		gpointer _tmp0;
		gpointer _tmp1;
		_tmp0 = NULL;
		(*node)->key = (_tmp0 = NULL, (((*node)->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->k_destroy_func ((*node)->key), NULL)), _tmp0);
		_tmp1 = NULL;
		(*node)->value = (_tmp1 = NULL, (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp1);
		*node = (*node)->next;
		self->priv->_nnodes--;
		gee_hash_map_resize (self);
		self->priv->_stamp++;
		return TRUE;
	}
	return FALSE;
}


static void gee_hash_map_real_clear (GeeMap* base) {
	GeeHashMap * self;
	self = (GeeHashMap*) base;
	{
		gint i;
		i = 0;
		for (; i < self->priv->_array_size; i++) {
			GeeHashMapNode* _tmp0;
			GeeHashMapNode* node;
			_tmp0 = NULL;
			node = (_tmp0 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp0);
			while (node != NULL) {
				GeeHashMapNode* _tmp1;
				GeeHashMapNode* next;
				gpointer _tmp2;
				gpointer _tmp3;
				GeeHashMapNode* _tmp5;
				GeeHashMapNode* _tmp4;
				_tmp1 = NULL;
				next = (_tmp1 = node->next, node->next = NULL, _tmp1);
				_tmp2 = NULL;
				node->key = (_tmp2 = NULL, ((node->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : (node->key = (self->priv->k_destroy_func (node->key), NULL)), _tmp2);
				_tmp3 = NULL;
				node->value = (_tmp3 = NULL, ((node->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (node->value = (self->priv->v_destroy_func (node->value), NULL)), _tmp3);
				_tmp5 = NULL;
				_tmp4 = NULL;
				node = (_tmp5 = (_tmp4 = next, next = NULL, _tmp4), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp5);
				(next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL));
			}
			(node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL));
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_map_resize (self);
}


static void gee_hash_map_resize (GeeHashMap* self) {
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->_array_size >= (3 * self->priv->_nnodes)) {
		_tmp1 = self->priv->_array_size >= GEE_HASH_MAP_MIN_SIZE;
	} else {
		_tmp1 = FALSE;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if ((3 * self->priv->_array_size) <= self->priv->_nnodes) {
			_tmp2 = self->priv->_array_size < GEE_HASH_MAP_MAX_SIZE;
		} else {
			_tmp2 = FALSE;
		}
		_tmp0 = _tmp2;
	}
	if (_tmp0) {
		gint new_array_size;
		GeeHashMapNode** _tmp3;
		gint new_nodes_size;
		gint new_nodes_length1;
		GeeHashMapNode** new_nodes;
		GeeHashMapNode** _tmp15;
		GeeHashMapNode** _tmp14;
		new_array_size = (gint) g_spaced_primes_closest ((guint) self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_MAP_MIN_SIZE, GEE_HASH_MAP_MAX_SIZE);
		_tmp3 = NULL;
		new_nodes = (_tmp3 = g_new0 (GeeHashMapNode*, new_array_size + 1), new_nodes_length1 = new_array_size, new_nodes_size = new_nodes_length1, _tmp3);
		{
			gint i;
			i = 0;
			for (; i < self->priv->_array_size; i++) {
				GeeHashMapNode* node;
				GeeHashMapNode* next;
				GeeHashMapNode* _tmp7;
				GeeHashMapNode* _tmp6;
				GeeHashMapNode* _tmp5;
				GeeHashMapNode* _tmp4;
				node = NULL;
				next = NULL;
				_tmp7 = NULL;
				_tmp6 = NULL;
				_tmp5 = NULL;
				_tmp4 = NULL;
				for (node = (_tmp5 = (_tmp4 = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp4), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp5); node != NULL; node = (_tmp7 = (_tmp6 = next, next = NULL, _tmp6), (node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL)), _tmp7)) {
					GeeHashMapNode* _tmp9;
					GeeHashMapNode* _tmp8;
					guint hash_val;
					GeeHashMapNode* _tmp11;
					GeeHashMapNode* _tmp10;
					GeeHashMapNode* _tmp13;
					GeeHashMapNode* _tmp12;
					_tmp9 = NULL;
					_tmp8 = NULL;
					next = (_tmp9 = (_tmp8 = node->next, node->next = NULL, _tmp8), (next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL)), _tmp9);
					hash_val = node->key_hash % new_array_size;
					_tmp11 = NULL;
					_tmp10 = NULL;
					node->next = (_tmp11 = (_tmp10 = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp10), (node->next == NULL) ? NULL : (node->next = (gee_hash_map_node_free (node->next), NULL)), _tmp11);
					_tmp13 = NULL;
					_tmp12 = NULL;
					new_nodes[hash_val] = (_tmp13 = (_tmp12 = node, node = NULL, _tmp12), (new_nodes[hash_val] == NULL) ? NULL : (new_nodes[hash_val] = (gee_hash_map_node_free (new_nodes[hash_val]), NULL)), _tmp13);
				}
				(node == NULL) ? NULL : (node = (gee_hash_map_node_free (node), NULL));
				(next == NULL) ? NULL : (next = (gee_hash_map_node_free (next), NULL));
			}
		}
		_tmp15 = NULL;
		_tmp14 = NULL;
		self->priv->_nodes = (_tmp15 = (_tmp14 = new_nodes, new_nodes = NULL, _tmp14), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = new_nodes_length1, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp15);
		self->priv->_array_size = new_array_size;
		new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	}
}


static gint gee_hash_map_real_get_size (GeeMap* base) {
	GeeHashMap* self;
	self = (GeeHashMap*) base;
	return self->priv->_nnodes;
}


void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_hash_func = value;
}


void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_equal_func = value;
}


void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_equal_func = value;
}


static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash) {
	GeeHashMapNode* self;
	gpointer _tmp0;
	gpointer _tmp1;
	self = g_slice_new0 (GeeHashMapNode);
	gee_hash_map_node_instance_init (self);
	_tmp0 = NULL;
	self->key = (_tmp0 = k, k = NULL, _tmp0);
	_tmp1 = NULL;
	self->value = (_tmp1 = v, v = NULL, _tmp1);
	self->key_hash = hash;
	return self;
}


static void gee_hash_map_node_instance_init (GeeHashMapNode * self) {
}


static void gee_hash_map_node_free (GeeHashMapNode* self) {
	(self->next == NULL) ? NULL : (self->next = (gee_hash_map_node_free (self->next), NULL));
	g_slice_free (GeeHashMapNode, self);
}


static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapKeySet* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapKeySet*) g_type_create_instance (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_key_set_set_map (self, map);
	return self;
}


static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_set_construct (GEE_HASH_MAP_TYPE_KEY_SET, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GType gee_hash_map_key_set_real_get_element_type (GeeIterable* base) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	return self->priv->k_type;
}


static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	return (GeeIterator*) gee_hash_map_key_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
}


static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_key_set_real_clear (GeeCollection* base) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	return gee_map_contains ((GeeMap*) self->priv->_map, key);
}


static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value) {
	GeeHashMap* _tmp1;
	GeeHashMap* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_map = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL)), _tmp1);
}


static gint gee_hash_map_key_set_real_get_size (GeeCollection* base) {
	GeeHashMapKeySet* self;
	self = (GeeHashMapKeySet*) base;
	return gee_map_get_size ((GeeMap*) self->priv->_map);
}


static void gee_hash_map_key_set_class_init (GeeHashMapKeySetClass * klass) {
	gee_hash_map_key_set_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_set_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapKeySetPrivate));
}


static void gee_hash_map_key_set_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_map_key_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_map_key_set_real_get_element_type;
	iface->iterator = gee_hash_map_key_set_real_iterator;
}


static void gee_hash_map_key_set_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_map_key_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_key_set_real_add;
	iface->clear = gee_hash_map_key_set_real_clear;
	iface->remove = gee_hash_map_key_set_real_remove;
	iface->contains = gee_hash_map_key_set_real_contains;
	iface->get_size = gee_hash_map_key_set_real_get_size;
}


static void gee_hash_map_key_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_map_key_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_map_key_set_instance_init (GeeHashMapKeySet * self) {
	self->priv = GEE_HASH_MAP_KEY_SET_GET_PRIVATE (self);
}


static void gee_hash_map_key_set_finalize (GeeCollectionObject* obj) {
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_key_set_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_set_get_type (void) {
	static GType gee_hash_map_key_set_type_id = 0;
	if (gee_hash_map_key_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeySetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeySet), 0, (GInstanceInitFunc) gee_hash_map_key_set_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_set_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapKeySet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_map_key_set_type_id;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapKeyIterator* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapKeyIterator*) g_type_create_instance (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_key_iterator_set_map (self, map);
	return self;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_iterator_construct (GEE_HASH_MAP_TYPE_KEY_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	self = (GeeHashMapKeyIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0 = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	return self->priv->_node != NULL;
}


static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	gconstpointer _tmp0;
	self = (GeeHashMapKeyIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_node->key, ((_tmp0 == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp0) : self->priv->k_dup_func ((gpointer) _tmp0));
}


static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp1;
	GeeHashMap* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_map = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL)), _tmp1);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_key_iterator_class_init (GeeHashMapKeyIteratorClass * klass) {
	gee_hash_map_key_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapKeyIteratorPrivate));
}


static void gee_hash_map_key_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_key_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_key_iterator_real_next;
	iface->get = gee_hash_map_key_iterator_real_get;
}


static void gee_hash_map_key_iterator_instance_init (GeeHashMapKeyIterator * self) {
	self->priv = GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_key_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashMapKeyIterator * self;
	self = GEE_HASH_MAP_KEY_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_key_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_iterator_get_type (void) {
	static GType gee_hash_map_key_iterator_type_id = 0;
	if (gee_hash_map_key_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeyIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeyIterator), 0, (GInstanceInitFunc) gee_hash_map_key_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_key_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapKeyIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_key_iterator_type_id;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapValueCollection* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapValueCollection*) g_type_create_instance (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_value_collection_set_map (self, map);
	return self;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_collection_construct (GEE_HASH_MAP_TYPE_VALUE_COLLECTION, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GType gee_hash_map_value_collection_real_get_element_type (GeeIterable* base) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	return self->priv->v_type;
}


static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	return (GeeIterator*) gee_hash_map_value_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
}


static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_value_collection_real_clear (GeeCollection* base) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	GeeIterator* it;
	gboolean _tmp3;
	self = (GeeHashMapValueCollection*) base;
	it = gee_iterable_iterator ((GeeIterable*) self);
	while (gee_iterator_next (it)) {
		gpointer _tmp0;
		gboolean _tmp1;
		_tmp0 = NULL;
		if ((_tmp1 = self->priv->_map->priv->_value_equal_func (_tmp0 = gee_iterator_get (it), value), ((_tmp0 == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (_tmp0 = (self->priv->v_destroy_func (_tmp0), NULL)), _tmp1)) {
			gboolean _tmp2;
			return (_tmp2 = TRUE, (it == NULL) ? NULL : (it = (gee_collection_object_unref (it), NULL)), _tmp2);
		}
	}
	return (_tmp3 = FALSE, (it == NULL) ? NULL : (it = (gee_collection_object_unref (it), NULL)), _tmp3);
}


static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value) {
	GeeHashMap* _tmp1;
	GeeHashMap* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_map = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL)), _tmp1);
}


static gint gee_hash_map_value_collection_real_get_size (GeeCollection* base) {
	GeeHashMapValueCollection* self;
	self = (GeeHashMapValueCollection*) base;
	return gee_map_get_size ((GeeMap*) self->priv->_map);
}


static void gee_hash_map_value_collection_class_init (GeeHashMapValueCollectionClass * klass) {
	gee_hash_map_value_collection_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_collection_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapValueCollectionPrivate));
}


static void gee_hash_map_value_collection_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_map_value_collection_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_map_value_collection_real_get_element_type;
	iface->iterator = gee_hash_map_value_collection_real_iterator;
}


static void gee_hash_map_value_collection_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_map_value_collection_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_value_collection_real_add;
	iface->clear = gee_hash_map_value_collection_real_clear;
	iface->remove = gee_hash_map_value_collection_real_remove;
	iface->contains = gee_hash_map_value_collection_real_contains;
	iface->get_size = gee_hash_map_value_collection_real_get_size;
}


static void gee_hash_map_value_collection_instance_init (GeeHashMapValueCollection * self) {
	self->priv = GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE (self);
}


static void gee_hash_map_value_collection_finalize (GeeCollectionObject* obj) {
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_value_collection_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_collection_get_type (void) {
	static GType gee_hash_map_value_collection_type_id = 0;
	if (gee_hash_map_value_collection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueCollection), 0, (GInstanceInitFunc) gee_hash_map_value_collection_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_collection_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapValueCollection", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
	}
	return gee_hash_map_value_collection_type_id;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapValueIterator* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapValueIterator*) g_type_create_instance (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_value_iterator_set_map (self, map);
	return self;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_iterator_construct (GEE_HASH_MAP_TYPE_VALUE_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	self = (GeeHashMapValueIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0;
		_tmp0 = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0 = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0 = FALSE;
		}
		if (!_tmp0) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	return self->priv->_node != NULL;
}


static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	gconstpointer _tmp0;
	self = (GeeHashMapValueIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	_tmp0 = NULL;
	return (_tmp0 = self->priv->_node->value, ((_tmp0 == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0) : self->priv->v_dup_func ((gpointer) _tmp0));
}


static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp1;
	GeeHashMap* _tmp0;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_map = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL) ? NULL : gee_collection_object_ref (_tmp0)), (self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL)), _tmp1);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_value_iterator_class_init (GeeHashMapValueIteratorClass * klass) {
	gee_hash_map_value_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapValueIteratorPrivate));
}


static void gee_hash_map_value_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_value_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_value_iterator_real_next;
	iface->get = gee_hash_map_value_iterator_real_get;
}


static void gee_hash_map_value_iterator_instance_init (GeeHashMapValueIterator * self) {
	self->priv = GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_value_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashMapValueIterator * self;
	self = GEE_HASH_MAP_VALUE_ITERATOR (obj);
	(self->priv->_map == NULL) ? NULL : (self->priv->_map = (gee_collection_object_unref (self->priv->_map), NULL));
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_value_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_iterator_get_type (void) {
	static GType gee_hash_map_value_iterator_type_id = 0;
	if (gee_hash_map_value_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueIterator), 0, (GInstanceInitFunc) gee_hash_map_value_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_value_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapValueIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_value_iterator_type_id;
}


static void gee_hash_map_class_init (GeeHashMapClass * klass) {
	gee_hash_map_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapPrivate));
}


static void gee_hash_map_gee_map_interface_init (GeeMapIface * iface) {
	gee_hash_map_gee_map_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_keys = gee_hash_map_real_get_keys;
	iface->get_values = gee_hash_map_real_get_values;
	iface->contains = gee_hash_map_real_contains;
	iface->get = gee_hash_map_real_get;
	iface->set = gee_hash_map_real_set;
	iface->remove = gee_hash_map_real_remove;
	iface->clear = gee_hash_map_real_clear;
	iface->get_size = gee_hash_map_real_get_size;
}


static void gee_hash_map_instance_init (GeeHashMap * self) {
	self->priv = GEE_HASH_MAP_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_map_finalize (GeeCollectionObject* obj) {
	GeeHashMap * self;
	self = GEE_HASH_MAP (obj);
	{
		gee_map_clear ((GeeMap*) self);
	}
	self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_parent_class)->finalize (obj);
}


GType gee_hash_map_get_type (void) {
	static GType gee_hash_map_type_id = 0;
	if (gee_hash_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMap), 0, (GInstanceInitFunc) gee_hash_map_instance_init, NULL };
		static const GInterfaceInfo gee_map_info = { (GInterfaceInitFunc) gee_hash_map_gee_map_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMap", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_type_id, GEE_TYPE_MAP, &gee_map_info);
	}
	return gee_hash_map_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}




