#!/usr/bin/python

# VBoxGtk: A VirtualBox GTK+ GUI
# Copyright (C) 2008 Francisco J. Vazquez-Araujo, Spain
# franjva at gmail dot com

# This file is part of VBoxGtk.

# VBoxGtk is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# VBoxGtk is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with VBoxGtk.  If not, see <http://www.gnu.org/licenses/>.



# DAO for VM and VDI using VBoxManage. Should use VirtualBox libraries, but
# using VBoxManage and parsing its output is easier :).

import re
import subprocess
import os

import util
import vmvdi

def command_run_iface(vboxiface, run_str, verbose = False, from_thread = False):
	status, msg = util.command_run(run_str)
	if status == 1: 
		vboxiface.show_msg(msg, from_thread, True)
		return 1
	if len(msg) > 10 and verbose: vboxiface.show_msg(msg, from_thread, False)
	return 0

## VBoxMgr

def get_system_properties():
	running = subprocess.Popen('VBoxManage list systemproperties'.split(), stdin = None, stdout = subprocess.PIPE, stderr = None)
	version = ''
	min_ram = 4
	max_ram = 2000
	max_vram = 128
	max_vdi = 200000
	default_vdi_folder = os.path.expanduser('~') + '/.VirtualBox/VDI/'
	for line in running.stdout:
		re_res = re.search('VirtualBox Command Line Management Interface Version ([0-9\.]*)(_OSE)?.*\n', line)
		if re_res:
			version = re_res.group(1)
			ose = re_res.group(2) != None
		re_res = re.search('([\w \.]+): *(.+)\n', line)
		if not re_res: continue
		if re_res.group(1) == 'Minimum guest RAM size': min_ram = int(re_res.group(2).split()[0])
		elif re_res.group(1) == 'Maximum guest RAM size': max_ram = int(re_res.group(2).split()[0])
		elif re_res.group(1) == 'Maximum video RAM size': max_vram = int(re_res.group(2).split()[0])
		elif re_res.group(1) == 'Maximum VDI size': max_vdi = int(re_res.group(2).split()[0])
		elif re_res.group(1) == 'Default VDI filder': default_vdi_folder = re_res.group(2) + '/'
	return version, ose, min_ram, max_ram, max_vram, max_vdi, default_vdi_folder

def get_ostypes():
	ostypes = []
	running = subprocess.Popen('VBoxManage -nologo list ostypes'.split(), stdin = None, stdout = subprocess.PIPE, stderr = None)
	for line in running.stdout:
		re_res = re.search('(\w+): *(.+)\n', line)
		if not re_res: continue
		if re_res.group(1) == 'ID': os_id = re_res.group(2)
		elif re_res.group(1) == 'Description': ostypes.append((os_id, re_res.group(2)))
	return ostypes

def get_hostdvds():
	vdis = []
	running = subprocess.Popen('VBoxManage -nologo list hostdvds'.split(), stdin = None, stdout = subprocess.PIPE, stderr = None)
	for line in running.stdout:
		re_res = re.search('(Name|Path): *(.+)\n', line)
		if not re_res: continue
		vdis.append(vmvdi.VDI(re_res.group(2), 'hostdvd'))
	return vdis
	
def create_hd(vboxiface, hd, hd_size, dynamic):
	run_str = 'VBoxManage  -nologo createvdi -filename'.split() + [hd.path, '-size', str(hd_size), '-register']
	if not dynamic: run_str += ['-static']
	return command_run_iface(vboxiface, run_str)

def add_vdi(vboxiface, vdi):
	run_str = 'VBoxManage  -nologo registerimage'.split() + [vdi.type, vdi.path]
	return command_run_iface(vboxiface, run_str)

def remove_vdi(vboxiface, vdi):
	run_str = 'VBoxManage  -nologo unregisterimage'.split() + [vdi.type, vdi.path]
	return command_run_iface(vboxiface, run_str)
	
	
## VM

def create_vm(vboxiface, vm_name):
	run_str = 'VBoxManage  -nologo createvm -name'.split() + [vm_name, '-register']
	return command_run_iface(vboxiface, run_str)
		
def delete_vm(vboxiface, vm_name):
	run_str = 'VBoxManage -nologo unregistervm'.split() + [vm_name, '-delete']
	return command_run_iface(vboxiface, run_str)

def empty_vdis(vboxiface, vm):
	run_str = 'VBoxManage -nologo modifyvm'.split() + [vm.name] + '-hda none -dvd none'.split()
	return command_run_iface(vboxiface, run_str)

def save_vm(vboxiface, vm):
	run_str = 'VBoxManage -nologo modifyvm'.split() + [vm.name]
	if vm.need_saving_vdi:
		if vm.hw.hd == None: hd_str = 'none'
		else: hd_str = vm.hw.hd.path
		if vm.hw.dvd == None: dvd_str = 'none'
		elif vm.hw.dvd.type == 'hostdvd': dvd_str = 'host:' + vm.hw.dvd.path
		else: dvd_str = vm.hw.dvd.path
		run_str += ['-hda', hd_str, '-dvd', dvd_str]
	run_str += ['-ostype', vm.ostype, '-memory', str(int(vm.hw.mem)), '-vram', str(int(vm.hw.vmem))]
	if vm.hw.hwvirtex: run_str += '-hwvirtex on'.split()
	else: run_str += '-hwvirtex off'.split()
	if vm.hw.dvdpassthrough: run_str += '-dvdpassthrough on'.split()
	else: run_str += '-dvdpassthrough off'.split()
	run_str += ['-audio', vm.hw.audiodev]
	nicnumber = 0
	if vm.hw.nics[nicnumber] != None:
		run_str += [ '-macaddress' + str(nicnumber + 1), vm.hw.nics[nicnumber].mac, 
					 '-nic' + str(nicnumber + 1), vm.hw.nics[nicnumber].attachment,
					 '-nictype' + str(nicnumber + 1), vm.hw.nics[nicnumber].device]
		if vm.hw.nics[nicnumber].cable: run_str += ('-cableconnected' + str(nicnumber + 1) + ' on').split()
		else: run_str += ('-cableconnected' + str(nicnumber + 1) + ' off').split()
		if vm.hw.nics[nicnumber].trace: run_str += ['-nictrace' + str(nicnumber + 1), 'on', '-nictracefile' + str(nicnumber + 1), vm.hw.nics[nicnumber].tracefile]
		else: run_str += ['-nictrace' + str(nicnumber + 1), 'off']
		if vm.hw.nics[nicnumber].attachment == 'intnet': run_str += ['-intnet' + str(nicnumber + 1), vm.hw.nics[nicnumber].intnet]
	return command_run_iface(vboxiface, run_str)

def swap_dvd(vboxiface, vm, dvd):
	if dvd == None: dvd_str = 'none'
	elif dvd.type == 'hostdvd': dvd_str = 'host:' + dvd.path
	else: dvd_str = dvd.path
	run_str = 'VBoxManage -nologo controlvm'.split() + [vm.name, 'dvdattach', dvd_str]
	return command_run_iface(vboxiface, run_str)

def change_state(vboxiface, vm, state):
	run_str = 'VBoxManage -nologo controlvm'.split() + [vm.name, state]
	return command_run_iface(vboxiface, run_str)

def save_shared(vboxiface, vm, shared):
	run_str = 'VBoxManage -nologo sharedfolder add'.split() + [vm.name, '-name', shared.name, '-hostpath', shared.path]
	if not shared.writable: run_str += ['-readonly']
	if vm.state == 'running': run_str += ['-transient']
	return command_run_iface(vboxiface, run_str)

def delete_shared(vboxiface, vm, shared_num):
	run_str = 'VBoxManage -nologo sharedfolder remove'.split() + [vm.name, '-name', vm.hw.shared_folders[shared_num].name]
	if vm.state == 'running': run_str += ['-transient']
	return command_run_iface(vboxiface, run_str)

def take_snapshot(vboxiface, vm, snapshot):
	run_str = 'VBoxManage -nologo snapshot'.split() + [vm.name, 'take', snapshot.name]
	return command_run_iface(vboxiface, run_str)
	
def discard_snapshot(vboxiface, vm, snapshot_num):
	run_str = 'VBoxManage -nologo snapshot'.split() + [vm.name, 'discard']
	if vm.snapshots[snapshot_num].uuid == 0: run_str += [vm.snapshots[snapshot_num].name]
	else: run_str += [util.remove_chars(vm.snapshots[snapshot_num].uuid,'{}')]
	return command_run_iface(vboxiface, run_str)
	
def discard_current_state(vboxiface, vm):
	run_str = 'VBoxManage -nologo snapshot'.split() + [vm.name, 'discardcurrent', '-state']
	return command_run_iface(vboxiface, run_str)
	
def edit_snapshot(vboxiface, vm, snapshot_num, name):
	run_str = 'VBoxManage -nologo snapshot'.split() + [vm.name, 'edit']
	if vm.snapshots[snapshot_num].uuid == 0: run_str += [vm.snapshots[snapshot_num].name]
	else: run_str += [vm.snapshots[snapshot_num].uuid]
	run_str += ['-newname',name]
	return command_run_iface(vboxiface, run_str)
