#!/usr/bin/python

# VBoxGtk: A VirtualBox GTK+ GUI
# Copyright (C) 2008 Francisco J. Vazquez-Araujo, Spain
# franjva at gmail dot com

# This file is part of VBoxGtk.

# VBoxGtk is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# VBoxGtk is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with VBoxGtk.  If not, see <http://www.gnu.org/licenses/>.



# DAO for VM and VDI that parses the .xml files in ~/.VirtualBox.


import xml.dom.minidom
import os

import vmvdi

class VMXML:
    def __init__(self, xmlfile):
        self.xmlfile = xmlfile
        self.xml = xml.dom.minidom.parse(self.xmlfile)
        self.vm = None
        
    def parse_vm(self, vdis):
        vboxnode = self.xml.childNodes[1]
        machinenode = vboxnode.childNodes[1]
        self.vm = vmvdi.VM(machinenode.getAttribute('name'))
        self.vm.ostype = machinenode.getAttribute('OSType')
        if machinenode.getAttribute('stateFile') != '': self.vm.state = 'saved'
        self.parse_machine(machinenode, vdis, self.vm)
        
    def parse_machine(self, node, vdis, snapshot):
        for subnode in node.childNodes:
            if subnode.localName == 'Hardware': self.parse_hardware(subnode, vdis, snapshot)
            elif subnode.localName == 'HardDiskAttachments': self.parse_disk_attachments(subnode, vdis, snapshot)
            elif subnode.localName == 'Snapshots': self.parse_snapshot(subnode.childNodes[1], vdis, snapshot)
            elif subnode.localName == 'Snapshot': self.parse_snapshot(subnode, vdis, snapshot)
                    
    def parse_hardware(self, node, vdis, snapshot):
        for subnode in node.childNodes:
            name = subnode.localName
            if name == 'CPU': snapshot.hw.hwvirtex = subnode.childNodes[1].getAttribute('enabled') == 'true'
            if name == 'Memory': snapshot.hw.mem = int(subnode.getAttribute('RAMSize'))
            elif name == 'Display': snapshot.hw.vmem = int(subnode.getAttribute('VRAMSize'))
            elif name == 'DVDDrive': self.parse_dvddrive(subnode, vdis, snapshot)
            elif name == 'AudioAdapter': 
                if subnode.getAttribute('enabled') != 'true': continue
                audio_dic = { 'ALSA':'alsa', 'alsa':'alsa', 'OSS':'oss', 'oss':'oss', 
                              'Null':'null', 'null':'null', 'Pulse':'pulse', 'pulse':'pulse' }
                snapshot.hw.audiodev = audio_dic[subnode.getAttribute('driver')]
            elif name == 'Network': self.parse_network(subnode, snapshot)
            elif name == 'SharedFolders': 
                for shared_node in subnode.childNodes:
                    if shared_node.localName != 'SharedFolder': continue
                    shared = vmvdi.Shared(shared_node.getAttribute('name'), shared_node.getAttribute('hostPath'),
                                          shared_node.getAttribute('writable') == 'true')
                    snapshot.hw.shared_folders.append(shared)
                    
    def parse_disk_attachments(self, node, vdis, snapshot):
        for subnode in node.childNodes:
            if subnode.localName != 'HardDiskAttachment': continue
            uuid = subnode.getAttribute('hardDisk')
            for vdi in vdis:
                if not uuid in vdi.uuid: continue
                snapshot.hw.hd = vdi
                if uuid == vdi.uuid[0] and snapshot == self.vm: 
                    vdi.attached_to.append(self.vm)
                    vdi.attached_to_in_db = self.vm
                break
                    
    def parse_snapshot(self, node, vdis, snapshot):
        new_snapshot = vmvdi.Snapshot(node.getAttribute('name'), node.getAttribute('uuid'))
        self.vm.snapshots.append(new_snapshot)
        self.parse_machine(node, vdis, new_snapshot)
                
    def parse_dvddrive(self, node, vdis, snapshot):
        snapshot.hw.dvdpassthrough = node.getAttribute('passthrough') == 'true'
        if len(node.childNodes) == 0: return
        if node.childNodes[1].localName == 'HostDrive':
            hostdrive = node.childNodes[1].getAttribute('src')
            for vdi in vdis:
                if hostdrive != vdi.path: continue
                snapshot.hw.dvd = vdi
                vdi.attached_to.append(self.vm)
                break
        elif node.childNodes[1].localName == 'Image':
            uuid = node.childNodes[1].getAttribute('uuid')
            for vdi in vdis:
                if not uuid in vdi.uuid: continue
                snapshot.hw.dvd = vdi
                vdi.attached_to.append(self.vm)
                break
        
    def parse_network(self, node, snapshot):
        for subnode in node.childNodes:
            if subnode.localName != 'Adapter': continue
            nic_num = int(subnode.getAttribute('slot'))
            if subnode.getAttribute('enabled') == 'false': continue
            snapshot.hw.nics[nic_num] = vmvdi.NIC(subnode.getAttribute('MACAddress'), 'null', subnode.getAttribute('type'), 
                                                  subnode.getAttribute('cable') == 'true', subnode.getAttribute('trace') == 'true')
            snapshot.hw.nics[nic_num].tracefile = subnode.getAttribute('tracefile')		
            if len(subnode.childNodes) == 0: continue
            att = subnode.childNodes[1]
            if att.localName == 'NAT':
                snapshot.hw.nics[nic_num].attachment = 'nat'
            elif att.localName == 'HostInterface':
                snapshot.hw.nics[nic_num].attachment = 'hostif'
            elif att.localName == 'InternalNetwork':
                snapshot.hw.nics[nic_num].attachment = 'intnet'
                snapshot.hw.nics[nic_num].intnet = att.getAttribute('name')


class VBoxDAOXML:

    def __init__(self):
        self.vdis = []
        self.vm_xml = []
        try:
            self.xml_file = open(os.path.expanduser('~') + '/.VirtualBox/VirtualBox.xml')
            self.main_xml = xml.dom.minidom.parse(self.xml_file)
            self.parse_main_xml()
        except:
            pass
                                    
    def parse_main_xml(self):
        globalnode = self.main_xml.childNodes[1].childNodes[1]
        for subnode in globalnode.childNodes:
            if subnode.localName == 'MachineRegistry': self.parse_machine_registry(subnode)
            elif subnode.localName == 'MediaRegistry': self.parse_media_registry(subnode)
                
    def parse_machine_registry(self, node):
        for subnode in node.childNodes:
            if subnode.localName != 'MachineEntry': continue
            xmlname = subnode.getAttribute('src')
            if xmlname[0] != '/': xmlname = os.path.expanduser('~') + '/.VirtualBox/' + xmlname
            self.vm_xml.append(VMXML(xmlname))
        
    def parse_media_registry(self, node):
        hdnode = node.childNodes[1] # FIXME: search for them instead of assuming they are in 1, 3 and 5
        dvdnode = node.childNodes[3]
        floppynode = node.childNodes[5]
        for subnode in hdnode.childNodes:
            if subnode.localName != 'HardDisk': continue
            self.vdis.append(vmvdi.VDI(subnode.getAttribute('location'),'disk'))
            self.vdis[-1].uuid = [subnode.getAttribute('uuid')]
            for diffnode in subnode.childNodes:
                if diffnode.localName != 'HardDisk': continue
                diff = diffnode
                while diff.localName == 'HardDisk':
                    self.vdis[-1].uuid.append(diff.getAttribute('uuid'))
                    diff = diff.childNodes[-2]
        for subnode in dvdnode.childNodes:
            if subnode.localName != 'Image': continue
            self.vdis.append(vmvdi.VDI(subnode.getAttribute('location'),'dvd'))
            self.vdis[-1].uuid.append(subnode.getAttribute('uuid'))
        for subnode in floppynode.childNodes:
            if subnode.localName != 'Image': continue
            self.vdis.append(vmvdi.VDI(subnode.getAttribute('location'),'floppy'))		
            self.vdis[-1].uuid.append(subnode.getAttribute('uuid'))
        
    def get_vdis(self):
        return self.vdis
        
    def get_vms(self):
        vms = []
        for xml in self.vm_xml:
            xml.parse_vm(self.vdis)
            vms.append(xml.vm)
        return vms
            
    
