/*
 * ts_scanner.c:	Demostrate how a ts scanner for AC3 data
 *			within a PS1 stream could work.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * Copyright (C) 2003 Werner Fink, <werner@suse.de>
 */

#include <stdio.h>
#include <unistd.h>
#include <asm/types.h>
#include <sys/types.h>

#ifndef AARONS_TYPES
#define AARONS_TYPES
typedef unsigned long long uint_64;
typedef unsigned int   uint_32;
typedef unsigned short uint_16;
typedef unsigned char  uint_8;

typedef signed long long sint_64;
typedef signed int     sint_32;
typedef signed short   sint_16;
typedef signed char    sint_8;
#endif

#ifndef WORDS_BIGENDIAN
# define char2short(a,b)	((((a) << 8) & 0xff00) ^ ((b) & 0x00ff))
# define shorts(a)		(a)
# define char2int(a,b,c,d)	((((a)<<24)&0xff000000)^(((b)<<16)&0x00ff0000)^\
				(((c)<<8)&0x0000ff00)^((d)&0x000000ff))
#else
# define char2short(a,b)	((((b) << 8) & 0xff00) ^ ((a) & 0x00ff))
# define shorts(a)		char2short(((a) & 0xff),(((a) >> 8) & 0xff));
# define char2int(a,b,c,d)	((((d)<<24)&0xff000000)^(((c)<<16)&0x00ff0000)^\
				(((b)<<8)&0x0000ff00)^((a)&0x000000ff))
#endif

#ifndef TS_SIZE
# define TS_SIZE	188
#endif

typedef enum _bool {
    false = 0,
    true = 1
} bool_t;

const static uint PS1magic = char2int(0x00,0x00,0x01,0xBD);

typedef struct _sub {
    uint_8	type;
    uint_8	track;
    uint_16	pos;
    uint_16	off;
} sub_t;

typedef struct _pes {
    uint_32	word;
    uint_16	bytes;
    uint_16	ptsoff;
    uint_16	ctrl;
    uint_16	len;
    sub_t	sub;
} pes_t;

#define CTRL_HEAD	0x01
#define CTRL_ALIGNED	0x02
#define CTRL_START	0x04
#define CTRL_ERROR	0x08

static pes_t ps1;

static void ScanTSpay(const uint_8 *buf, const uint_8 *const tail)
{
skip:
    if (ps1.ctrl & CTRL_ERROR) {
	memset(&ps1, 0, sizeof(pes_t));
	ps1.word = 0xFFFFFFFF;
    }

    if ((ps1.bytes < 9)) {
	switch (ps1.bytes) {
	case 0 ... 3:
	    while (ps1.word != PS1magic) {	/* Search for the PS1 magic */
		if (buf >= tail)
		    goto out;
		ps1.word = (ps1.word << 8) | *buf++;
	    }
	    ps1.ctrl |=  CTRL_HEAD;
	    ps1.ctrl &= ~CTRL_ALIGNED;
	    ps1.bytes = 4;
	case 4:
	    if (buf >= tail)
		goto out;
	    ps1.len = (*buf++) << 8;
	    ps1.bytes++;
	case 5:
	    if (buf >= tail)
		goto out;
	    ps1.len |= (*buf++);
	    ps1.len += 6;
	    ps1.bytes++;
	case 6:
	    if (buf >= tail)
		goto out;
	    if (((*buf) & 0xC0) != 0x80) {	/* First mpeg2 PTS flag byte */
		ps1.ctrl |= CTRL_ERROR;
		goto skip;			/* Not a mpeg2 PS1 stream */
	    }
	    if ((*buf) & 0x04)			/* Is there a sub stream aligned? */
		ps1.ctrl |= CTRL_ALIGNED;
	    buf++;
	    ps1.bytes++;
	case 7:
	    if (buf >= tail)
		goto out;
	    if ((*buf++) & 0xC0)		/* Second mpeg2 PTS flag byte (PTS exists) */
		ps1.ctrl |= CTRL_ALIGNED;
	    ps1.bytes++;
	case 8:
	    if (buf >= tail)
		goto out;
	    ps1.ptsoff = (*buf++);		/* Third mpeg2 PTS flag byte (PTS data length) */
	    ps1.bytes++;
	    ps1.sub.pos = ps1.bytes + ps1.ptsoff;
	default:
	    break;
	}
    }

    while (ps1.ptsoff) {			/* Skip PTS info */
	int skip = tail - buf;
	if (skip <= 0)
	    goto out;
	if (skip > ps1.ptsoff)
	    skip = ps1.ptsoff;
	ps1.ptsoff -= skip;
	ps1.bytes  += skip;
	buf	   += skip;
    }

    while (ps1.ctrl & CTRL_HEAD) {
	if (buf >= tail)
	    goto out;
	switch (ps1.bytes - ps1.sub.pos) {
	    uint_8 type, track;
	case 0:					/* DVD sub frame head or DVB AC3 data? */
	    type  = ((*buf) & 0xf8);
	    track = ((*buf) & 0x07);
	    if (!ps1.sub.type) {
		if (!(ps1.ctrl & CTRL_ALIGNED)) {
		    ps1.ctrl |= CTRL_ERROR;
		    goto skip;			/* Continue the scan because not unique */
		}
		ps1.sub.track = track;
		ps1.sub.type  = type;
		switch (*buf) {
		case 0x80 ... 0x87:
		case 0x88 ... 0x8F:
		case 0xA0 ... 0xA7:
		    buf++;
		    ps1.bytes++;
		    break;
		case 0x0b:			/* Break loop: aligned AC3 data found (DVB) */
		    ps1.ctrl &= ~(CTRL_HEAD|CTRL_ALIGNED);
		    break;
		default:
		    ps1.ctrl |= CTRL_ERROR;
		    goto skip;			/* Continue the scan, no valid data found */
		    break;
		}
		ps1.ctrl |= CTRL_START;		/* We seek later for start within payload */
	    } else {
		if (ps1.ctrl & CTRL_ALIGNED) {
		    /* Sanity check: type and track should be equal */
		    if ((ps1.sub.type != type) && (ps1.sub.track != track)) {
			ps1.ctrl |= CTRL_ERROR;
			goto skip;		/* Wrong type found and/or track */
		    }
		}
		if ((ps1.sub.type == 0x08) && (ps1.sub.track == 0x03)) {
		    /* DVB AC3 stream */
		    ps1.ctrl &= ~(CTRL_HEAD|CTRL_ALIGNED);
		    break;
		}
		buf++;
		ps1.bytes++;
	    }
	    break;
	case 1:					/* Count of frames which begins with this frame */
#if 0
	    frames = (*buf);
#endif
	    buf++;
	    ps1.bytes++;
	    break;
	case 2:					/* First byte for offset to start within payload */
	    if (ps1.ctrl & CTRL_START)
		ps1.sub.off = (*buf) << 8;
	    buf++;
	    ps1.bytes++;
	    break;
	case 3:					/* Second byte for offset to start within payload */
	    if (ps1.ctrl & CTRL_START) {
		ps1.sub.off |= (*buf);
		if (ps1.sub.off)
		    ps1.sub.off--;		/* We count this byte within next few lines */
		ps1.ctrl &= ~CTRL_START;
	    }
	    if (ps1.sub.type != 0xA0)		/* Only sub header for linear PCM has seven bytes */
		ps1.ctrl &= ~(CTRL_HEAD|CTRL_ALIGNED);
	    buf++;
	    ps1.bytes++;
	    break;
	case 4:					/* Fifth flag byte of sub header for linear PCM */
#if 0
	    emphasis = ((*buf) & 0x80);
	    mute     = ((*buf) & 0x40);
	    samples  = ((*buf) & 0x1F);
#endif
	    buf++;
	    ps1.bytes++;
	    break;
	case 5:					/* Sixth flag byte of sub header for linear PCM */
#if 0
	    switch ((*buf) & 0x30) {
	    case 0x00: sample_rate = 48000; break;
	    case 0x20: sample_rate = 44100; break;
	    case 0x30: sample_rate = 32000; break;
	    case 0x10: sample_rate = 96000; break;
	    default:   /* nothing */	break;
	    }
	    switch ((*buf) & 0xC0) {
	    case 0x00: sample_bit = 16; break;
	    case 0x40: sample_bit = 20; break;
	    case 0x80: sample_bit = 24; break;
	    default:   sample_bit = -1; break;
	    }
	    channels = ((*buf) & 0x07) + 1;
#endif
	    buf++;
	    ps1.bytes++;
	    break;
	case 6:					/* Seventh byte of sub header for linear PCM (dynamic range) */
#if 0
	    dynamic_range = (*buf);
#endif
	    buf++;
	    ps1.bytes++;
	    ps1.ctrl &= ~(CTRL_HEAD|CTRL_ALIGNED);
	    break;
	default:
	    break;
	}
    }

    while (ps1.sub.off) {			/* Skip bytes upto a valid start within the payload */
	int skip = tail - buf;
	if (skip <= 0)
	    goto out;
	if (skip > ps1.sub.off)
	    skip = ps1.sub.off;
	ps1.sub.off -= skip;
	ps1.bytes   += skip;
	buf	    += skip;
    }

    while (ps1.bytes < ps1.len) {
	int transmit = tail - buf;
	if (transmit <= 0)
	    goto out;
	if (transmit + ps1.bytes > ps1.len)
	    transmit = ps1.len - ps1.bytes;
	fwrite(buf, sizeof(uint_8), transmit, stdout); 		/* Copy to output buffer */
	ps1.bytes += transmit;
	buf	  += transmit;
    }

    if (ps1.len && (ps1.len == ps1.bytes)) {
	ps1.bytes = ps1.len = ps1.sub.off = 0;
	ps1.word = 0xFFFFFFFF;
	ps1.ctrl &= ~CTRL_ALIGNED;
	goto skip;				/* Is there any data beyond frame lenght? */
    }
out:
    return;
}

int main ()
{
    uint_8 buf[TS_SIZE - 4];
    ssize_t n = 0;

    memset(&ps1, 0, sizeof(pes_t));
    ps1.word = 0xFFFFFFFF;
    while ((n = read(0, (void *)&buf[0], sizeof(buf)))) {
	ScanTSpay(&buf[0], (&buf[0]) + n);
    }
    return 0;
}
